#define INCL_GPI
#include <os2.h>
#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <process.h>
#include <stdlib.h>
#include <pmbitmap.h>
#include "imagconv.h"
#include "copyr.h"       /* MS copyright msg */

#define FILE_NAME_SIZE	260

char PMFile[FILE_NAME_SIZE];   /* PM Icon file name.                       */
char WinFile[FILE_NAME_SIZE];  /* Windows icon file name.                  */
FILE *InFile, *OutFile;        /* Input and Output file handles.           */

SHORT FileType;                /* Type of images: FT_BMP, FT_ICO, FT_CUR.  */

#define FT_BMP    0
#define FT_ICO    1
#define FT_CUR    2

char *typenames[3] = { "bitmap", "icon", "cursor" };


FILEIMAGES *pfiFirst;          /* Head of linked-list of file images.      */
SHORT cImages;                 /* Number of images in the linked-list.     */
USHORT iBmpToProcess = 0;      /* Only one bmp can be written. This is it. */

SHORT cbOutHeader;             /* Size of the header info for output file. */
LONG offNextWrite;             /* Next place in the output file to write.  */


UCHAR *title = MSTITLE(PM->Windows Image Converter, 0.1);
UCHAR *copyr = MSCOPY;


/***************************************************************************
    
    PROGRAM IMAGCONV

    This program converts PM images to Windows images.

****************************************************************************/
int main(int argc, char **argv)
{
    FILEIMAGES *pfiCurrent;
    BOOL fDelOutFile;

    printf("%s\n%s\n", title, copyr);
    printf("\n");

/*  Parse the arguments.
    ==================== */

    if (ParseArgs(argc, argv) <= -1) {
        Usage();
        exit(1);
    }

/*  Try to open the input file.
    =========================== */

	if ((InFile = fopen(PMFile, "rb")) == NULL) {
		fprintf(stderr, "ERROR: Cannot open the icon file %s\n", PMFile);
		exit(1);
	}

/*  Scan for images in the input file.
    ================================== */

    if ((cImages = ScanImages()) <= 0) {
        if (cImages == 0) printf("\nNo convertable images were found.\n");
        exit(1);
    }

/*  Create the output file.
    ======================= */

	if ((OutFile = fopen(WinFile, "wb")) == NULL) {
		fprintf(stderr, "\nERROR:  Cannot create the output file (%s).\n", WinFile);
		exit(1);
	}

/*  Determine the size of the header information for the output files.
    ================================================================== */
    
    cbOutHeader = 0;
    if (FileType == FT_ICO || FileType == FT_CUR) {
        cbOutHeader = sizeof(WINICONHEADER) + cImages * sizeof(WINICONARRAY);
    }

/*  Process the image(s).
    ===================== */

    fDelOutFile = FALSE;
    offNextWrite = (LONG) cbOutHeader;

    pfiCurrent = pfiFirst;
    while (pfiCurrent != NULL && !fDelOutFile) {
        if (FileType != FT_BMP || pfiCurrent->usImageNum == iBmpToProcess) {
            if (ProcessImage(pfiCurrent) <= -1) {
                fDelOutFile = TRUE;
            }
        }
        pfiCurrent = pfiCurrent->pfiNext;
    }

/*  Output the header information for icon and cursor files.
    ======================================================== */

    if (!fDelOutFile && (FileType == FT_ICO || FileType == FT_CUR)) {
        if (WriteHeaderInfo() <= -1) {
            fDelOutFile = TRUE;
        }
    }

/*  Close the input and output files.    
    ================================= */

	fclose(InFile);
	fclose(OutFile);

    if (fDelOutFile) {
        remove(WinFile);
        return(1);
    }

/*  Finished.
    ========= */

    return(0);
}

/****************************************************************************

    Function:  ParseArgs

    This function checks for proper usage, and parses the input and output
    filenames into their respective variables, "PMFile" and "WinFile".
    If there are any errors, a -1 is returned; otherwise, 0 is returned.

****************************************************************************/
SHORT ParseArgs(int argc, char **argv)
{
    int len;
    char *pLastDot;
    char *pLastSlash;
    SHORT argloc;
    SHORT haveType;

    if (argc < 3)
        return(-1);

/*  Get the type of file we are converting.
    ======================================= */

    FileType = -1;
    argloc = 1;

    if      (stricmp(argv[1], "-b") == 0) FileType = FT_BMP;
    else if (stricmp(argv[1], "-i") == 0) FileType = FT_ICO;
    else if (stricmp(argv[1], "-c") == 0) FileType = FT_CUR;

    if (argv[1][0] == '-' && FileType == -1) {
        printf("Unknown option in arg list: '%s'\n", argv[1]);
        return(-1);
    }
    if (FileType != -1) argloc = 2;

/*  Get the input file name.
    ======================== */

    len = strlen(argv[argloc]);
    if (len >= FILE_NAME_SIZE) {
        printf("\nInput file name is too long.\n");
        return(-1);
    }
    strcpy(PMFile, argv[argloc]);
    strupr(PMFile);

/*  If the filename doesn't have an extension, add one.
    =================================================== */

    pLastDot = strrchr(PMFile, (int)'.');
    pLastSlash = strrchr(PMFile, (int)'\\');
    if (pLastDot == NULL || (pLastSlash != NULL && pLastSlash > pLastDot)) {
        if (len + 4 >= FILE_NAME_SIZE) {
            printf("\nInput file name is too long.\n");
            return(-1);
        }
        if      (FileType == FT_BMP) strcat(PMFile, ".BMP");
        else if (FileType == FT_ICO) strcat(PMFile, ".ICO");
        else if (FileType == FT_CUR) strcat(PMFile, ".PTR");
        else {
            printf("\nERROR: The type of image file must be specified.\n\n");
            return(-1);
        }
    }

/*  Make sure the extension we have matches the file type.
    ====================================================== */

    pLastDot = strrchr(PMFile, (int)'.');

    haveType = -1;
    if      (strcmp(pLastDot, ".BMP") == 0) haveType = FT_BMP;
    else if (strcmp(pLastDot, ".ICO") == 0) haveType = FT_ICO;
    else if (strcmp(pLastDot, ".PTR") == 0) haveType = FT_CUR;

    if (haveType == -1 && FileType == -1) {
        printf("\nERROR: The type of image file must be specified.\n\n");
        return(-1);
    } else if (haveType != -1 && FileType == -1) {
        FileType = haveType;
    } else if (haveType != -1 && haveType != FileType) {
        printf("\nERROR: The type of image specified does not match the file extension.\n\n");
        return(-1);
    }

/*  Get the output file name.
    ========================= */

    len = strlen(argv[argloc+1]);
    if (len >= FILE_NAME_SIZE) {
        printf("\nOutput file name is too long.\n");
        return(-1);
    }
    strcpy(WinFile, argv[argloc+1]);
    strupr(WinFile);

/*  If the filename doesn't have an extension, add one.
    =================================================== */

    pLastDot = strrchr(WinFile, (int)'.');
    pLastSlash = strrchr(WinFile, (int)'\\');
    if (pLastDot == NULL || (pLastSlash != NULL && pLastSlash > pLastDot)) {
        if (len + 4 >= FILE_NAME_SIZE) {
            printf("\nOutput file name is too long.\n");
            return(-1);
        }
        if      (FileType == FT_BMP) strcat(WinFile, ".BMP");
        else if (FileType == FT_ICO) strcat(WinFile, ".ICO");
        else if (FileType == FT_CUR) strcat(WinFile, ".CUR");
    }

/*  Finished.
    ========= */

    return(0);
}

/****************************************************************************

    Function:  Usage

    Outputs proper program usage to the console.

****************************************************************************/
void Usage(void)
{
    printf("\nPM->Windows converter for bitmap, icon, and cursor files.\n");
    printf("Usage: IMAGCONV <type> <input> <output>\n");
    printf("       type    Type of image: -b (for bitmap), -i (icon), -c (cursor)\n");
    printf("       input   PM format input file\n");
    printf("       output  Windows format output file\n");
    return;
}

/****************************************************************************

    Function: ScanImages
    
    This function scans the PM file, determining how many images are in the
    file and setting up a linked list of FILEIMAGE structures identifying
    them.  The function returns the number of images found.

****************************************************************************/
SHORT ScanImages(void)
{
    LONG offCurrent, offNext;
    FILEIMAGES *pfiNew, *pfiLoop;
    USHORT usType;
    ULONG  ulData[2];
    USHORT usData[3];
    BOOL fSkip;
    
    offCurrent = 0L;
    cImages = 0;

/*  Continue looping while we have images in the file.
    ================================================== */

    while (offCurrent >= 0L) {

/*      Get the image type for the current image.
        ========================================= */

        if (fseek(InFile, offCurrent, SEEK_SET) != 0 ||
                fread(&usType, sizeof(USHORT), 1, InFile) == 0) {
			fprintf(stderr, "Error reading PM icon file type info.\n");
            exit(1);
        }

/*      Perform processing based on type of image header.
        ================================================= */

        fSkip = FALSE;
        switch (usType) {

/*      Handle a Bitmap Array, which might have anything in it.
        ======================================================= */

        case BFT_BITMAPARRAY:
		    if (fread(&ulData[0], sizeof(ULONG), 2, InFile) == 0) {
			    fprintf(stderr, "Error reading BmpArray info.\n");
                exit(1);
            }
            offNext = ulData[1];
            if (offNext == 0L) offNext = -1L;

		    if (fread(&usData[0], sizeof(USHORT), 3, InFile) == 0) {
			    fprintf(stderr, "Error reading BmpArray info.\n");
                exit(1);
            }
            switch (usData[2]) {
            case BFT_BMAP:
                if (FileType != FT_BMP) {
                    fSkip = TRUE;
                    printf("Skipping Bitmap in BITMAPARRAY.\n");
                } else {
                    ++cImages;
                    printf("Image %2d:  BITMAPARRAY (BMP).\n", cImages);
                }
                break;
            case BFT_ICON:
                fSkip = TRUE;
                printf("Skipping Mono Icon in BITMAPARRAY (unsupported).\n");
                break;
            case BFT_COLORICON:
                if (FileType != FT_ICO) {
                    fSkip = TRUE;
                    printf("Skipping Icon in BITMAPARRAY.\n");
                } else {
                    ++cImages;
                    printf("Image %2d:  BITMAPARRAY (ICO).\n", cImages);
                }
                break;
            case BFT_POINTER:
                if (FileType != FT_CUR) {
                    fSkip = TRUE;
                    printf("Skipping Cursor in BITMAPARRAY.\n");
                } else {
                    ++cImages;
                    printf("Image %2d:  BITMAPARRAY (CUR).\n", cImages);
                }
                break;
            case BFT_COLORPOINTER:
                fSkip = TRUE;
                printf("Skipping Color Cursor in BITMAPARRAY (unsupported).\n");
                break;
            default:
                fSkip = TRUE;
                printf("Skipping unrecognized element in BITMAPARRAY.\n");
                break;
            }
            break;

/*      Handle a Bitmap File.
        ===================== */

        case BFT_BMAP:
            if (FileType != FT_BMP) {
                printf("Error: Found a bitmap image (expected %s).\n", typenames[FileType]);
                return(-1);
            }
            goto common;

/*      Handle an Icon file.
        ==================== */

        case BFT_COLORICON:
        case BFT_ICON:
            if (FileType != FT_ICO) {
                printf("Error: Found an icon image (expected %s).\n", typenames[FileType]);
                return(-1);
            }
            goto common;

/*      Handle a Cursor file.
        ===================== */

        case BFT_POINTER:
            if (FileType != FT_CUR) {
                printf("Error: Found a cursor image (expected %s).\n", typenames[FileType]);
                return(-1);
            }
            goto common;

        case BFT_COLORPOINTER:
            printf("Can't handle color cursors yet.\n");
            return(-1);

/*      Common code for bitmap, icon and cursor files (not Bitmap Arrays).
        ================================================================== */
common:
            if (cImages != 0) {
                fprintf(stderr, "Can't handle Single Images following BmpArray.\n");
                exit(1);
            }
            ++cImages;
            offNext = -1L;
            printf("Image %2d:  Single Image (%c%c).\n", cImages, (usType & 0x00FF), ((usType >> 8) & 0x00FF));
            break;

/*      Handle unrecognized entries in the file.
        ======================================== */

        default:
            fprintf(stderr, "Unknown PM image format.\n");
            exit(1);
        }

/*      If not skipping, add a new entry to our linked list.
        ==================================================== */

        if (!fSkip) {

/*          Allocate memory for the new entry.
            ================================== */

            if ((pfiNew = (FILEIMAGES *) malloc(sizeof(FILEIMAGES))) == NULL) {
                fprintf(stderr, "Out of memory for FILEIMAGES.\n");
                exit(1);
            }

/*          Place it at the end of the linked list.
            ======================================= */

            pfiLoop = pfiFirst;
            while (pfiLoop != NULL && pfiLoop->pfiNext != NULL) {
                pfiLoop = pfiLoop->pfiNext;
            }
        
            if (pfiLoop == NULL) pfiFirst = pfiNew;
            else                 pfiLoop->pfiNext = pfiNew;

/*          Fill in the fields of the new structure.
            ======================================== */

            pfiNew->pfiNext    = NULL;
            pfiNew->usImageNum = cImages - 1;
            pfiNew->usType     = usType;
            pfiNew->offHeader  = offCurrent;
        }

/*      Go on to the next image in the file.
        ==================================== */

        offCurrent = offNext;
    }

/*  Finished.
    ========= */

    return(cImages);
}

/****************************************************************************

    Function: ProcessImage
    
    This function processes the single image passed in the FILEIMAGES
    structure.  It returns a negative number if an error occurs and outputs
    a message.  Otherwise, a 0 is returned.

****************************************************************************/
SHORT ProcessImage(FILEIMAGES *pfi)
{
    SHORT ret;

    if (pfi->usType == BFT_BITMAPARRAY) {
        if (FileType == FT_BMP) ret = ProcessArrayBmp(pfi);
        else                    ret = ProcessArrayIco(pfi);
    } else {
        if (FileType == FT_BMP) ret = ProcessNormalBmp(pfi);
        else                    ret = ProcessNormalIco(pfi);
    }
    return(ret);
}

/****************************************************************************

    Function: ProcessArrayBmp
    
    This function processes the single image passed in the FILEIMAGES
    structure.  The FILEIMAGES structure points to a bitmap array which
    contains a bitmap.

****************************************************************************/
SHORT ProcessArrayBmp(FILEIMAGES *pfi)
{
    BITMAPARRAYFILEHEADER bafh;
    SHORT cClrClrs;
    RGB *rgb3Clr;
    USHORT width, width2, height;
    LONG offClr, sizClr;
    LONG totwrite;
    SHORT i;
    WINBITMAPFILEHEADER wbfh;
    WINBITMAPINFOHEADER wbih;
    RGB4 *rgb4;
    UCHAR *p1, *p2;

    printf("Processing bitmap from bitmap array file.\n");

/*  Read in the Bitmap Array structure.
    =================================== */

    if (fseek(InFile, pfi->offHeader, SEEK_SET) != 0 ||
            fread(&bafh, sizeof(BITMAPARRAYFILEHEADER), 1, InFile) < 1) {
        fprintf(stderr, "    Error reading bitmap file (invalid format).\n");
        return(-1);
    }

/*  Read in the RGB table for the bitmap.
    ===================================== */

    cClrClrs = 1 << bafh.bfh.bmp.cBitCount;
    if ((rgb3Clr = (RGB *) malloc(sizeof(RGB) * cClrClrs)) == NULL) {
        fprintf(stderr, "    Unable to allocate memory for PM RGB table.\n");
        return(-1);
    }

    if (fread(rgb3Clr, sizeof(RGB), cClrClrs, InFile) < (USHORT)cClrClrs) {
        fprintf(stderr, "    Error reading bitmap file (invalid format).\n");
        return(-1);
    }

/*  Get the proper offsets for the bits.
    ==================================== */

    width  = bafh.bfh.bmp.cx;
    height = bafh.bfh.bmp.cy;
    width2 = ((width % 16) == 0 ? width : (width + (16 - (width % 16))));

    offClr = bafh.bfh.offBits;
    sizClr = ((LONG)width2 * height * bafh.bfh.bmp.cBitCount) / 8;

/*  Now we have all the data.  Start filling in Windows structures.
    =============================================================== */

    totwrite = sizeof(WINBITMAPFILEHEADER) + sizeof(WINBITMAPINFOHEADER) + (sizeof(RGB4) * cClrClrs) + sizClr;

    /* Fill in the Windows BITMAPFILEHEADER structure. */

    wbfh.usType    = BFT_BMAP;
    wbfh.ulSize    = totwrite;
    wbfh.reserved1 = 0;
    wbfh.reserved2 = 0;
    wbfh.ulOffBits = totwrite - sizClr;

    /* Fill in the Windows BITMAPINFOHEADER structure. */

    wbih.cbSize    = sizeof(WINBITMAPINFOHEADER);
    wbih.ulWidth   = (ULONG)width;
    wbih.ulHeight  = (ULONG)height;
    wbih.cPlanes   = 1;
    wbih.cBitCount = bafh.bfh.bmp.cBitCount;
    for (i = 0; i < 6; ++i) {
        wbih.ulMisc[i] = 0L;
    }

    /* Convert PM's 3-byte RGB to Windows' 4-byte RGB. */

    if ((rgb4 = (RGB4 *)malloc(sizeof(RGB4) * cClrClrs)) == NULL) {
        fprintf(stderr, "    Unable to allocate memory for Windows RGB table.\n");
        free(rgb3Clr);
        return(-1);
    }

    p1 = (UCHAR *)rgb3Clr;
    p2 = (UCHAR *)rgb4;

    for (i = 0; i < cClrClrs; ++i) {
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = 0;
    }

/*  Write data.
    =========== */

    if (fseek(OutFile, offNextWrite, SEEK_SET) != 0 ||
            fwrite(&wbfh, sizeof(WINBITMAPFILEHEADER), 1, OutFile) < 1 ||
            fwrite(&wbih, sizeof(WINBITMAPINFOHEADER), 1, OutFile) < 1 ||
            fwrite(rgb4, sizeof(RGB4), cClrClrs, OutFile) < (USHORT)cClrClrs) {
        fprintf(stderr, "    Unable to write to output file.\n");
        free(rgb3Clr);
        free(rgb4);
        return(-2);
    }

    if (fseek(InFile, offClr, SEEK_SET) != 0 || CopyBytes(sizClr) != 0) {
        fprintf(stderr, "    Unable to write to output file.\n");
        free(rgb3Clr);
        free(rgb4);
        return(-2);
    }

    offNextWrite += totwrite;

/*  Finished successfully.
    ====================== */

    free(rgb3Clr);
    free(rgb4);
    return(0);
}

/****************************************************************************

    Function: ProcessArrayIco
    
    This function processes the single image passed in the FILEIMAGES
    structure.  The FILEIMAGES structure points to a bitmap array which
    contains either an icon or a cursor.

****************************************************************************/
SHORT ProcessArrayIco(FILEIMAGES *pfi)
{
    BITMAPARRAYFILEHEADER bafh;
    BITMAPFILEHEADER bfh;
    SHORT cAndClrs, cClrClrs;
    RGB *rgb3And, *rgb3Clr;
    USHORT width, height;
    LONG offXor, offAnd, sizAnd, offClr, sizClr;
    UCHAR *pXor, *pAnd, *pClr;
    LONG totwrite;
    SHORT i;
    WINBITMAPINFOHEADER wbih;
    RGB4 *rgb4;
    UCHAR *p1, *p2;

    printf("Processing icon/cursor from bitmap array file.\n");

/*  Read in the Bitmap Array structure.
    =================================== */

    fseek(InFile, pfi->offHeader, SEEK_SET);
    fread(&bafh, sizeof(BITMAPARRAYFILEHEADER), 1, InFile);

/*  Read in the RGB table for bitmap used as the ANDXOR mask.
    ========================================================= */

    cAndClrs = 1 << bafh.bfh.bmp.cBitCount;
    rgb3And = (RGB *) malloc(sizeof(RGB) * cAndClrs);
    fread(rgb3And, sizeof(RGB), cAndClrs, InFile);

/*  Read the next Bitmap File Header (for the Color mask).
    ====================================================== */

    fread(&bfh, sizeof(BITMAPFILEHEADER), 1, InFile);
    if (bfh.usType == BFT_BITMAPARRAY) {
        fseek(InFile, 14L - sizeof(BITMAPFILEHEADER), SEEK_CUR);
        fread(&bfh, sizeof(BITMAPFILEHEADER), 1, InFile);
    }

/*  Read in the RGB table for the bitmap used as the Color mask.
    ============================================================ */

    cClrClrs = (1 << bfh.bmp.cBitCount);
    rgb3Clr = (RGB *) malloc(sizeof(RGB) * cClrClrs);
    fread(rgb3Clr, sizeof(RGB), cClrClrs, InFile);

/*  Get the proper offsets for the bits.
    ==================================== */

    width  = bafh.bfh.bmp.cx;
    height = bafh.bfh.bmp.cy >> 1;

    offXor = bafh.bfh.offBits;
    sizAnd = ((LONG)width * height * bafh.bfh.bmp.cBitCount) / 8;
    offAnd = offXor + sizAnd;

    offClr = bfh.offBits;
    sizClr = ((LONG)width * height * bfh.bmp.cBitCount) / 8;

/*  Read the bits into memory.
    ========================== */

    pAnd = (UCHAR *) malloc((SHORT)sizAnd);
    pXor = (UCHAR *) malloc((SHORT)sizAnd);
    pClr = (UCHAR *) malloc((SHORT)sizClr);

    fseek(InFile, offXor, SEEK_SET);
    fread(pXor, 1, (USHORT)sizAnd, InFile);
    fread(pAnd, 1, (USHORT)sizAnd, InFile);

    fseek(InFile, offClr, SEEK_SET);
    fread(pClr, 1, (USHORT)sizClr, InFile);

/*  Convert from PM's Xor/And/Clr scheme to Windows' And/Clr scheme.
    ================================================================ */

    ConvertXorSchemes(width, height, bfh.bmp.cBitCount, pAnd, pXor, pClr);

/*  Now we have all the data.  Start filling in Windows structures.
    =============================================================== */

    /* Fill in our local FILEIMAGES structure. */

    totwrite = sizeof(WINBITMAPINFOHEADER) + (sizeof(RGB4) * cClrClrs) + sizClr + sizAnd;

    pfi->DIBWidth    = width;
    pfi->DIBHeight   = height;
    pfi->DIBColors   = cClrClrs;
    pfi->DIBxhotspot = bafh.bfh.xHotspot;
    pfi->DIByhotspot = bafh.bfh.yHotspot;
    pfi->DIBSize     = totwrite;
    pfi->DIBOffset   = offNextWrite;

    /* Fill in the Windows BITMAPINFOHEADER structure. */

    wbih.cbSize    = sizeof(WINBITMAPINFOHEADER);
    wbih.ulWidth   = (ULONG)width;
    wbih.ulHeight  = (ULONG)height * 2;
    wbih.cPlanes   = 1;
    wbih.cBitCount = bfh.bmp.cBitCount;
    for (i = 0; i < 6; ++i) {
        wbih.ulMisc[i] = 0L;
    }

    /* Convert PM's 3-byte RGB to Windows' 4-byte RGB. */

    rgb4 = (RGB4 *) malloc(sizeof(RGB4) * cClrClrs);
    p1 = (UCHAR *)rgb3Clr;
    p2 = (UCHAR *)rgb4;

    for (i = 0; i < cClrClrs; ++i) {
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = 0;
    }

/*  Write data.
    =========== */

    fseek(OutFile, offNextWrite, SEEK_SET);
    fwrite(&wbih, sizeof(WINBITMAPINFOHEADER), 1, OutFile);
    fwrite(rgb4, sizeof(RGB4), cClrClrs, OutFile);
    fwrite(pClr, (USHORT)sizClr, 1, OutFile);
    fwrite(pAnd, (USHORT)sizAnd, 1, OutFile);

    offNextWrite += totwrite;

/*  Free memory allocated during processing of this file.
    ===================================================== */

    free(rgb3And);
    free(rgb3Clr);
    free(pAnd);
    free(pXor);
    free(pClr);
    free(rgb4);

/*  Finished.
    ========= */

    return(0);
}

/****************************************************************************

    Function: ProcessNormalBmp

    This function processes the single image passed in the FILEIMAGES
    structure.  The FILEIMAGES structure points to a normal BITMAPFILEHEADER
    which describes a bitmap.

****************************************************************************/
SHORT ProcessNormalBmp(FILEIMAGES *pfi)
{
    BITMAPFILEHEADER bfh;
    SHORT cClrClrs;
    RGB *rgb3Clr;
    USHORT width, height;
    LONG offClr, sizClr;
    LONG totwrite;
    SHORT i;
    WINBITMAPFILEHEADER wbfh;
    WINBITMAPINFOHEADER wbih;
    RGB4 *rgb4;
    UCHAR *p1, *p2;

    printf("Processing bitmap from normal bitmap file.\n");

/*  Read in the Bitmap File Header structure.
    ========================================= */

    fseek(InFile, pfi->offHeader, SEEK_SET);
    fread(&bfh, sizeof(BITMAPFILEHEADER), 1, InFile);

/*  Read in the RGB table for the bitmap.
    ===================================== */

    cClrClrs = 1 << bfh.bmp.cBitCount;
    rgb3Clr = (RGB *) malloc(sizeof(RGB) * cClrClrs);
    fread(rgb3Clr, sizeof(RGB), cClrClrs, InFile);

/*  Get the proper offsets for the bits.
    ==================================== */

    width  = bfh.bmp.cx;
    height = bfh.bmp.cy;

    offClr = bfh.offBits;
    sizClr = ((LONG)width * height * bfh.bmp.cBitCount) / 8;

/*  Now we have all the data.  Start filling in Windows structures.
    =============================================================== */

    totwrite = sizeof(WINBITMAPFILEHEADER) + sizeof(WINBITMAPINFOHEADER) + (sizeof(RGB4) * cClrClrs) + sizClr;

    /* Fill in the Windows BITMAPFILEHEADER structure. */

    wbfh.usType    = BFT_BMAP;
    wbfh.ulSize    = totwrite;
    wbfh.reserved1 = 0;
    wbfh.reserved2 = 0;
    wbfh.ulOffBits = totwrite - sizClr;

    /* Fill in the Windows BITMAPINFOHEADER structure. */

    wbih.cbSize    = sizeof(WINBITMAPINFOHEADER);
    wbih.ulWidth   = (ULONG)width;
    wbih.ulHeight  = (ULONG)height;
    wbih.cPlanes   = 1;
    wbih.cBitCount = bfh.bmp.cBitCount;
    for (i = 0; i < 6; ++i) {
        wbih.ulMisc[i] = 0L;
    }

    /* Convert PM's 3-byte RGB to Windows' 4-byte RGB. */

    rgb4 = (RGB4 *) malloc(sizeof(RGB4) * cClrClrs);
    p1 = (UCHAR *)rgb3Clr;
    p2 = (UCHAR *)rgb4;

    for (i = 0; i < cClrClrs; ++i) {
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = 0;
    }

/*  Write data.
    =========== */

    fseek(OutFile, offNextWrite, SEEK_SET);
    fwrite(&wbfh, sizeof(WINBITMAPFILEHEADER), 1, OutFile);
    fwrite(&wbih, sizeof(WINBITMAPINFOHEADER), 1, OutFile);
    fwrite(rgb4, sizeof(RGB4), cClrClrs, OutFile);

    if (fseek(InFile, offClr, SEEK_SET) != 0 || CopyBytes(sizClr) != 0) {
        fprintf(stderr, "    Error copy bitmap bits (invalid format?).\n");
        free(rgb3Clr);
        free(rgb4);
        return(-1);
    }
    offNextWrite += totwrite;

/*  Finished successfully.
    ====================== */

    free(rgb3Clr);
    free(rgb4);
    return(0);
}

/****************************************************************************

    Function: ProcessNormalIco

    This function processes the single image passed in the FILEIMAGES
    structure.  The FILEIMAGES structure points to a normal BITMAPFILEHEADER
    which describes either an icon or a cursor.

****************************************************************************/
SHORT ProcessNormalIco(FILEIMAGES *pfi)
{
    BITMAPFILEHEADER bfh;

    fseek(InFile, pfi->offHeader, SEEK_SET);
    fread(&bfh, sizeof(BITMAPFILEHEADER), 1, InFile);

    if (bfh.usType == BFT_ICON || bfh.usType == BFT_POINTER) {
        return(ProcessNormalIcoMono(pfi));
    } else {
        //return(ProcessNormalIcoClr(pfi));
        printf("Can't handle color icons that aren't in a bitmap array file.\n");
        return(-1);
    }
}
    
/****************************************************************************

    Function: ProcessNormalIcoMono

    This function processes the single image passed in the FILEIMAGES
    structure.  The FILEIMAGES structure points to a normal BITMAPFILEHEADER
    which describes either an icon or a cursor.

****************************************************************************/
SHORT ProcessNormalIcoMono(FILEIMAGES *pfi)
{
    BITMAPFILEHEADER bfh;
    SHORT cClrClrs;
    RGB *rgb3Clr;
    USHORT width, height;
    LONG offClr, sizAnd, sizClr;
    UCHAR *pAnd, *pClr;
    LONG totwrite;
    SHORT i;
    WINBITMAPINFOHEADER wbih;
    RGB4 *rgb4;
    UCHAR *p1, *p2;

    printf("Processing monochrome icon/cursor from normal file.\n");

/*  Read the Bitmap File Header.
    ============================ */

    fseek(InFile, pfi->offHeader, SEEK_SET);
    fread(&bfh, sizeof(BITMAPFILEHEADER), 1, InFile);

/*  Read in the RGB table for the bitmap.
    ===================================== */

    cClrClrs = (1 << bfh.bmp.cBitCount);
    rgb3Clr = (RGB *) malloc(sizeof(RGB) * cClrClrs);
    fread(rgb3Clr, sizeof(RGB), cClrClrs, InFile);

/*  Get the proper offsets for the bits.
    ==================================== */

    width  = bfh.bmp.cx;
    height = bfh.bmp.cy >> 1;

    offClr = bfh.offBits;
    sizAnd = ((LONG)width * height) / 8;
    sizClr = sizAnd;

/*  Read the bits into memory.
    ========================== */

    pClr = (UCHAR *) malloc((SHORT)sizClr);
    pAnd = (UCHAR *) malloc((SHORT)sizAnd);

    fseek(InFile, offClr, SEEK_SET);
    fread(pClr, 1, (USHORT)sizClr, InFile);
    fread(pAnd, 1, (USHORT)sizAnd, InFile);

/*  Now we have all the data.  Start filling in Windows structures.
    =============================================================== */

    /* Fill in our local FILEIMAGES structure. */

    totwrite = sizeof(WINBITMAPINFOHEADER) + (sizeof(RGB4) * cClrClrs) + sizClr + sizAnd;

    pfi->DIBWidth    = width;
    pfi->DIBHeight   = height;
    pfi->DIBColors   = cClrClrs;
    pfi->DIBxhotspot = bfh.xHotspot;
    pfi->DIByhotspot = bfh.yHotspot;
    pfi->DIBSize     = totwrite;
    pfi->DIBOffset   = offNextWrite;

    /* Fill in the Windows BITMAPINFOHEADER structure. */

    wbih.cbSize    = sizeof(WINBITMAPINFOHEADER);
    wbih.ulWidth   = (ULONG)width;
    wbih.ulHeight  = (ULONG)height * 2;
    wbih.cPlanes   = 1;
    wbih.cBitCount = bfh.bmp.cBitCount;
    for (i = 0; i < 6; ++i) {
        wbih.ulMisc[i] = 0L;
    }

    /* Convert PM's 3-byte RGB to Windows' 4-byte RGB. */

    rgb4 = (RGB4 *) malloc(sizeof(RGB4) * cClrClrs);
    p1 = (UCHAR *)rgb3Clr;
    p2 = (UCHAR *)rgb4;

    for (i = 0; i < cClrClrs; ++i) {
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = *p1++;
        *p2++ = 0;
    }

/*  Write data.
    =========== */

    fseek(OutFile, offNextWrite, SEEK_SET);
    fwrite(&wbih, sizeof(WINBITMAPINFOHEADER), 1, OutFile);
    fwrite(rgb4, sizeof(RGB4), cClrClrs, OutFile);
    fwrite(pClr, (USHORT)sizClr, 1, OutFile);
    fwrite(pAnd, (USHORT)sizAnd, 1, OutFile);

    offNextWrite += totwrite;

/*  Free memory allocated during processing of this file.
    ===================================================== */

    free(rgb3Clr);
    free(pAnd);
    free(pClr);
    free(rgb4);

/*  Finished.
    ========= */

    return(0);
}

/****************************************************************************

    Function: WriteHeaderInfo

    This function writes the header info that goes at the top of a Windows
    icon or cursor resource file. It returns a negative number if an error
    occurs and outputs a message.  Otherwise, a 0 is returned.

****************************************************************************/
SHORT WriteHeaderInfo(void)
{
    WINICONHEADER wih;
    WINICONARRAY wia;
    FILEIMAGES *pfi;

    if (FileType != FT_ICO && FileType != FT_CUR) return(0);
    
/*  Position to the top of the output file.
    ======================================= */

    fseek(OutFile, 0L, SEEK_SET);

/*  Write the header record.
    ======================== */

    wih.reserved = 0;
    wih.resType  = (FileType == FT_ICO ? 1 : 2);
    wih.resCount = cImages;

    fwrite(&wih, sizeof(WINICONHEADER), 1, OutFile);

/*  Loop through the images, output the resource array.
    =================================================== */

    for (pfi = pfiFirst; pfi != NULL; pfi = pfi->pfiNext) {

        wia.width     = (UCHAR)pfi->DIBWidth;
        wia.height    = (UCHAR)pfi->DIBHeight;
        wia.clrcount  = (UCHAR)pfi->DIBColors;
        wia.reserved  = 0;
        wia.xHotSpot  = pfi->DIBxhotspot;
        wia.yHotSpot  = pfi->DIByhotspot;
        wia.DIBsize   = pfi->DIBSize;
        wia.DIBoffset = pfi->DIBOffset;

        fwrite(&wia, sizeof(WINICONARRAY), 1, OutFile);
    }

/*  Finished.
    ========= */

    return(0);
}

/****************************************************************************

    Function: CopyBytes

    This function copies the specified number of bytes from the input file
    to the output file.  It assumes the files are already positioned to the
    proper location.  A 0 is returned if there were no errors.

****************************************************************************/
SHORT CopyBytes(LONG count)
{
    UCHAR *pBuff;
    SHORT cbBuff;
    SHORT cnt;

/*  Allocate a copy buffer.
    ======================= */

    for (cbBuff = 8192; cbBuff >>= 1; cbBuff >= 64) {
        if ((pBuff = malloc(8192)) != NULL) break;
    }
    if (pBuff == NULL) return(-1);

/*  Copy the data.
    ============== */

    while (count > 0) {
        cnt = (SHORT)(count > cbBuff ? (SHORT)cbBuff : (SHORT)count);
        if (fread(pBuff, 1, cnt, InFile) < (USHORT)cnt || 
                fwrite(pBuff, 1, cnt, OutFile) < (USHORT)cnt) {
            free(pBuff);
            return(-1);
        }
        count -= cnt;
    }

/*  Finished.
    ========= */

    free(pBuff);
    return(0);
}

/****************************************************************************

    Function: ConvertXorSchemes

    This function converts the PM Xor plus And masks to the method used for
    Windows resources.

****************************************************************************/
void ConvertXorSchemes(SHORT width, SHORT height, SHORT cBitCount, UCHAR *pAnd, UCHAR *pXor, UCHAR *pClr)
{
    UCHAR maskAnd, maskClr;
    SHORT i;
    SHORT numPels;
    SHORT clrmod;
    static UCHAR maskClrInit[9] = { 0, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff };

/*  Set up some initial values.
    =========================== */

    numPels = width * height;
    clrmod = (8 / cBitCount);

/*  Loop through every pel.
    ======================= */

    for (i = 0; i < numPels; ++i) {

/*      Adjust the mask and pointer for the And or Xor bits.
        ==================================================== */

        if ((i % 8) == 0) {
            maskAnd = 0x80;
            if (i != 0) {
                ++pAnd;
                ++pXor;
            }
        } else {
            maskAnd >>= 1;
        }

/*      Adjust the mask and pointer for the Clr bits.
        ============================================= */

        if ((i % clrmod) == 0) {
            maskClr = maskClrInit[cBitCount];
            if (i != 0) ++pClr;
        } else {
            maskClr >>= cBitCount;
        }

/*      If the And bit is 1, we need to set the Clr bits. 
        ================================================= */

        if ((*pAnd & maskAnd) != 0) {
            *pClr |= maskClr;
            if ((*pXor & maskAnd) == 0) *pClr ^= maskClr;
        }
    }

/*  Finished.
    ========= */

    return;
}
