/************************************************************************\
* The enclosed files, "the software," is provided by 
* Microsoft Corporation "as is" without warranty of any kind. 
* MICROSOFT DISCLAIMS ALL WARRANTIES, EITHER EXPRESS OR IMPLIED, 
* INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY 
* AND FITNESS FOR A PARTICULAR PURPOSE.  You assume all risks of 
* using the software.
* 
* The software is Copyright (c) 1992 Microsoft Corporation.
* Original Author: John M. Hall, Microsoft SDE  9/1/92
*
* You are granted the right to freely distribute this software.
* You are granted the right to make changes provided this comment block
* is retained without modification and you acknowledge the changes.
* 
\************************************************************************/
/************************************************************************\
*
*  MODULE:      SHRMEM.C
*
*  PURPOSE:     Simple shared memory management.
*
*  FUNCTIONS:   ShrMemEntry() - DLL entry point
*
*               DWORD   ShrAlloc(DWORD dwSize, UCHAR uchFlags)
*               LPVOID  ShrLock( DWORD hShr)
*               DWORD   ShrUnlock( DWORD hShr)
*               DWORD   ShrFree( DWORD hShr)
*               DWORD   ShrGet(LPCTSTR lpsz)
*               BOOL    ShrName(DWORD hShr, LPCTSTR lpsz)
*        
*               VOID ShrDump()  (A debugging function)
*
*
*  COMMENTS:    
*
************************************************************************/
#define SHRMEM_DLL_C

#include <stdio.h>
#include <memory.h>
#include <windows.h>
#include <winbase.h>
#include <string.h>

BOOL WINAPI _CRT_INIT(HINSTANCE, DWORD, LPVOID);

#define DEBUG
#define DOSWIN32
#include "shrmem.h"


/************************************************************************\
*
*  FUNCTION:    ShrMemEntry
*
*  INPUTS:      hDLL       - handle of DLL
*               dwReason   - indicates why DLL called
*               lpReserved - reserved
*
*  RETURNS:     TRUE for success, FALSE on error
*
*               Note that the retuRn value is used only when
*               dwReason = DLL_PROCESS_ATTACH.
*
*  GLOBAL VARS: See Shrmem.h for list of global variables.
*
*  COMMENTS:    Modified from simple dll sample
*
\************************************************************************/


BOOL  WINAPI ShrMemEntry (HANDLE hDLL, DWORD dwReason, LPVOID lpReserved)
{


  switch (dwReason)
  { 
  case DLL_PROCESS_ATTACH:
      { 
      /******************************************************************\
      *  DLL is attaching to the address space of the current process.
      \******************************************************************/
      BOOL bFlag;
      char buf[MAX_PATH];
      LPTSTR lpBaseName, lptstr;

#ifdef _DLL
      _CRT_INIT(hDLL, dwReason, lpReserved);
#endif
      GetModuleFileName (NULL, (LPTSTR) buf, MAX_PATH);

#ifdef DEBUGGING
      MessageBox (NULL, buf, "SHRMEM_DLL: Process attaching", MB_OK);
#endif

      //
      // Obtain a pointer to the BaseName of the executeable.
      // This is used to find logical groupings of processes in
      // the registry.
      //
      lptstr = lpBaseName = buf;

      while (*lptstr)
          {
          if (*lptstr == '\\')
              {
              lptstr++;
              lpBaseName = lptstr;
              }
          else
              lptstr++;
          }

      fp_log = fopen( "hp_shr.log", "a+");

      //
      // Attach shared memory.
      //
      bFlag = attach_memory(lpBaseName);

      if (!bFlag)
          fclose(fp_log);

      AssertBox( bFlag, "Attach Memory");

      return(bFlag);
      break;
    }

    case DLL_THREAD_ATTACH:

#ifdef _DLL
    _CRT_INIT(hDLL, dwReason, lpReserved);
#endif
      /******************************************************************\
      *  A new thread is being created in the current process.
      \******************************************************************/

#ifdef DEBUGING
      MessageBox (NULL, "THE_DLL: Thread attaching", "", MB_OK);
#endif
      break;
    case DLL_THREAD_DETACH:

      /******************************************************************\
      *  A thread is exiting cleanly.
      \******************************************************************/

#ifdef DEBUGING
      MessageBox (NULL, "THE_DLL: Thread detaching", "", MB_OK);
#endif
#ifdef _DLL
      _CRT_INIT(hDLL, dwReason, lpReserved);
#endif
      break;
    case DLL_PROCESS_DETACH:

      /******************************************************************\
      *  The calling process is detaching the DLL from its address space.
      \******************************************************************/

#ifdef DEBUGING
      MessageBox (NULL, "THE_DLL: Process detaching", "", MB_OK);
#endif
      fclose(fp_log);
      UnmapViewOfFile(lpMemBase);
      UnmapViewOfFile(lpCtrlBase);
      CloseHandle(hMemory);
      CloseHandle(hControl);
#ifdef _DLL
      _CRT_INIT(hDLL, dwReason, lpReserved);
#endif
      break;
  }
  return TRUE;
  UNREFERENCED_PARAMETER(hDLL);
  UNREFERENCED_PARAMETER(lpReserved);
}

//
// Can this pointer be written to?
//
BOOL IsPtrBad( LPVOID lpv)
{
    DWORD dw;
    LPDWORD lpdw;
    BOOL b = FALSE;

    try
        {
        lpdw = (LPDWORD) lpv;
        dw = *lpdw;
        *lpdw = dw;

        }
    except (EXCEPTION_EXECUTE_HANDLER)
        {
        b = TRUE;
        }

    return(b);

}

//
// A wrapper for VirtualProtect
//
BOOL Protect(LPVOID lpv, WORD wPages, DWORD flag)
{
    DWORD dwOldAccess;
    BOOL  b;

    b = VirtualProtect(lpv, wPages * dwPageSize, flag , &dwOldAccess);

    AssertBox(b, "Virtual Protect");
    return(TRUE);
}

//
//  The uchFlags parameter is currently not used.  It was intended to
//  control multiple access and the like.
//
DWORD  ShrAlloc(DWORD dwSize, UCHAR uchFlags)
{
    WORD wPages;
    FEP  fep = (FEP) lpCtrlBase;
    FREE_ELEMENT feNext;
    WORD wLeft;
    int  ii = 0;
    int  iiNewUnused = fep[iiUnused].sNext;

    //
    // Convert allocation size to rounded up pages
    //
    wPages = (WORD) (dwSize / dwPageSize);
    if ((dwSize % dwPageSize) != 0)
        wPages++;

    WaitForSingleObject(hSemaphore, ALLFFS);


    //
    // Walk thru structures.  First Fit Algorithm.
    //
    while ((fep[ii].uchFlags & SHR_MEM_ALLOC) || (fep[ii].wLength < wPages))
        {
        if (fep[ii].sNext == -1)
            {
            ReleaseSemaphore(hSemaphore, 1, NULL);
            return(0);
            }
        else
            ii = fep[ii].sNext;
        }

    fep[ii].dwProcessID = GetCurrentProcessId(); // ID of allocating process
    wLeft = fep[ii].wLength - wPages;            // Pages left over.


    //
    // Here, we are protecting ourselves from there being no more
    // control elements to split this structure.  In this case, we
    // give this particular process more memory than it asked for.
    //
    // This shouldn't happen.  The original implementation allocated
    // 4096 control structures for 256 pages.  4096 were allocatated
    // because sizeof(FREE_ELEMENT) == 16 and the minimum mapping object
    // is 64K.
    //
    fep[ii].wLength = (iiUnused == -1) ? fep[ii].wLength : wPages;

    if (wLeft > 0 && iiUnused != -1)
        {
        feNext.dwProcessID =  0;
        feNext.wOffset     =  fep[ii].wOffset + wPages;
        feNext.wLength     =  wLeft;
        feNext.sNext       =  fep[ii].sNext;
        feNext.sPrior      =  ii;
        feNext.aName       =  0;
        feNext.uchLock     =  0;
        feNext.uchFlags    =  SHR_MEM_FREE;

        fep[iiUnused] = feNext;
        fep[ii].sNext = iiUnused;
        iiUnused = iiNewUnused;
        }

    fep[ii].uchFlags = SHR_MEM_ALLOC;

    ReleaseSemaphore(hSemaphore, 1, NULL);
    return(MUNGEIT(ii));
}

LPVOID  ShrLock( DWORD hShr)
{
    FEP fep = (FEP) lpCtrlBase;
    int ii = UNMUNGE(hShr);
    LPVOID lpv;

    WaitForSingleObject(hSemaphore, ALLFFS);

    lpv = MAKE_LPV(fep[ii].wOffset);

    //
    // A process is only allowed to lock a handle once!
    // This has been commented out because the PDC release will
    // still allow up to read pages after VirtualProtect has marked
    // them inaccessible.  Furthermore, hitting exceptions in your
    // code makes it hard to use WinDbg.
    //
#if 0
    if (IsPtrBad(lpv) == FALSE)
        {
        fprintf( fp_log,  "Warning -- Locking memory we have write access too\n");
        ReleaseSemaphore(hSemaphore, 1, NULL);
        return(NULL);
        }
#endif
    fep[ii].uchLock++;

    Protect(lpv, fep[ii].wLength, PAGE_READWRITE);
    ReleaseSemaphore(hSemaphore, 1, NULL);
    return( lpv); 

}
DWORD  ShrUnlock( DWORD hShr)
{
    FEP fep = (FEP) lpCtrlBase;
    int ii = UNMUNGE(hShr);
    LPVOID lpv;
    DWORD  dw;

    WaitForSingleObject(hSemaphore, ALLFFS);
    lpv = MAKE_LPV(fep[ii].wOffset);

    //
    // You can't unlock it if you don't have access to it!
    //
#if 0
    if (IsPtrBad(lpv) == TRUE)
        {
        fprintf( fp_log,  "Warning -- Unlocking memory we have write access too\n");
        ReleaseSemaphore(hSemaphore, 1, NULL);
        return(hShr);
        }
#endif
    fep[ii].uchLock--;
    Protect(lpv, fep[ii].wLength, PAGE_NOACCESS);

    dw =  fep[ii].uchLock;

    ReleaseSemaphore(hSemaphore, 1, NULL);
    return( dw); 

}

DWORD  ShrFree( DWORD hShr)
{
    FEP fep = (FEP) lpCtrlBase;
    int ii = UNMUNGE(hShr);
    FEP fepNext;

    WaitForSingleObject(hSemaphore, ALLFFS);

    //
    // The memory must be allocated and unlocked to be freed.
    //
    if ( ! (fep[ii].uchFlags & SHR_MEM_ALLOC) || fep[ii].uchLock != 0)
        {
        ReleaseSemaphore(hSemaphore, 1, NULL);
        return(hShr);
        }

    fep[ii].dwProcessID = 0;
    fep[ii].uchFlags       = SHR_MEM_FREE;

    if (fep[ii].aName)
        {
        GlobalDeleteAtom(fep[ii].aName);
        fep[ii].aName = 0;
        }

    //
    // Coaless Pages
    //
    while (fep[ii].sPrior != -1 && fep[fep[ii].sPrior].uchFlags == SHR_MEM_FREE)
        ii = fep[ii].sPrior;

    while (fep[ii].sNext != -1 && fep[fep[ii].sNext].uchFlags == SHR_MEM_FREE)
        {    
        SHORT sOld;
        SHORT sNext;
        fepNext = &fep[sOld = fep[ii].sNext];

        fep[ii].wLength += fepNext->wLength;
        fep[ii].sNext    = fepNext->sNext;

        sNext = fepNext->sNext;
        if (sNext != -1)
            {
            fep[sNext].sPrior = ii;
            }

        fep[sOld].sNext = iiUnused;
        fep[sOld].uchFlags = SHR_MEM_INVALID;
        iiUnused = sOld;
        }

    ReleaseSemaphore(hSemaphore, 1, NULL);
    return( 0); 

}


VOID MyAssertBox( const char *title, const char *exp,
        const char *file, int line)
        {
        char buffer[512];

        sprintf( buffer, "Assert(%s) at (%s::%d)", exp, file, line);
        MessageBox (NULL, title, buffer, MB_OK | MB_ICONHAND);
        }

BOOL attach_memory(LPTSTR lpBaseName)
    {
    BOOL bCreateHeader = FALSE;
    WORD wFree, wAlloc;


    SetErrorMode(SEM_FAILCRITICALERRORS);

    hSemaphore = CreateSemaphore( NULL, 0, 1, "JMH_SHAREMEM_SEM");

    if (hSemaphore)
        {
        //
        // This is correct, we use GetLastError() to find out how the
        // function succeeded!
        //
        if (GetLastError() == ERROR_ALREADY_EXISTS)
            {
            //
            // We didn't create it, just attach to it so we need to
            // wait till its free.
            //
            WaitForSingleObject(hSemaphore, ALLFFS);
            dwProcesses++;
            }
        else
            {
            // We must be the first process.
            //
            SYSTEM_INFO sysinfo;
            HKEY hKeyShrMem;
            HKEY hKeyMember;
            char chClass[MAX_PATH];
            char chGroup[MAX_PATH];
            DWORD dw = MAX_PATH;
            LONG lErr;
            DWORD dwType;
            LPTSTR lptstr;

            //
            // This processing depends on the registry being properly
            // initialized.  Init.exe does the proper initialization.
            //
            lErr = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                "Software\\Shareware\\ShrMem", 0, KEY_ALL_ACCESS, &hKeyShrMem);
    
            if (lErr != ERROR_SUCCESS)
                {
                CloseHandle(hSemaphore);
                return(FALSE);
                }

            lErr = RegOpenKeyEx( hKeyShrMem, "Members", 0, KEY_ALL_ACCESS,
                &hKeyMember);
    
            if (lErr != ERROR_SUCCESS)
                {
                RegCloseKey(hKeyShrMem);
                CloseHandle(hSemaphore);
                return(FALSE);
                }

            lErr = RegQueryValueEx( hKeyMember, lpBaseName, NULL, &dwType,
                (LPBYTE) chClass, &dw);

            if (lErr != ERROR_SUCCESS || chClass[0] == 0x00)
                lptstr = "Default";
            else
                lptstr = chClass;

            RegCloseKey(hKeyMember);

            strcpy( chGroup, "Groups\\");
            strcat( chGroup, lptstr);

            lErr = RegOpenKeyEx( hKeyShrMem, chGroup, 0, KEY_ALL_ACCESS,
                &hKeyMember);
    
            if (lErr != ERROR_SUCCESS)
                {
                RegCloseKey(hKeyShrMem);
                CloseHandle(hSemaphore);
                return(FALSE);
                }

            dw = 4;
            lErr = RegQueryValueEx( hKeyMember, "Base", NULL, &dwType,
                (LPBYTE) &dwBase, &dw);

            if (lErr != ERROR_SUCCESS)
                {
                dwBase = 0x01000000;
                }

            dw = 4;
            lErr = RegQueryValueEx( hKeyMember, "Length", NULL, &dwType,
                (LPBYTE) &dwLength, &dw);

            if (lErr != ERROR_SUCCESS)
                {
                dwLength = 1024*1024;
                }


            RegCloseKey(hKeyMember);
            RegCloseKey(hKeyShrMem);

            GetSystemInfo(&sysinfo);

            dwProcesses = 1;
            dwPageSize  = sysinfo.dwPageSize;
            wFree  = (WORD) (dwLength / dwPageSize);
            wAlloc = 0;
            }

        //
        // The heart of the matter, create the memory mapped files.
        //
        hMemory = CreateFileMapping(
            (HANDLE) ALLFFS,
            NULL,
            PAGE_READWRITE,
            0,
            dwLength,
            "JMH_SHAREMEM");

        hControl = CreateFileMapping(
            (HANDLE) ALLFFS,
            NULL,
            PAGE_READWRITE,
            0,
            SZ_CONTROL,
            "JMH_Control_MEM");

        if (hMemory == NULL || hControl == NULL)
            fprintf( fp_log,  "CreateFileMapping Failed\n");

        //
        // Now attach the memory to the correct addresses
        //
        lpMemBase  = MapViewOfFileEx( hMemory, FILE_MAP_WRITE, 0, 0, 0,
                            (LPVOID) dwBase);

        if (lpMemBase == NULL)
            {
            fprintf( fp_log,  "Error -- Could not attach mem at desired address\n");
            ReleaseSemaphore(hSemaphore, 1, NULL);
            CloseHandle(hControl);
            CloseHandle(hMemory);
            return(FALSE);
            }

        lpCtrlBase = MapViewOfFile( hControl, FILE_MAP_WRITE, 0, 0, 0);

        if (lpCtrlBase == NULL)
            {
            fprintf( fp_log,  "MapViewOfFile Failed\n");
            UnmapViewOfFile(lpMemBase);
            ReleaseSemaphore(hSemaphore, 1, NULL);
            CloseHandle(hControl);
            CloseHandle(hMemory);
            return(FALSE);
            }

        //
        // First time initialization.
        //
        if (dwProcesses == 1)
            {
            FEP fep = (FEP) lpCtrlBase;
            int iPages = dwLength / dwPageSize;
            int ii;

            fep[0].dwProcessID =  0;
            fep[0].wOffset     =  0;
            fep[0].wLength     = wFree;
            fep[0].sNext       = -1;
            fep[0].sPrior      = -1;
            fep[0].uchLock     =  0;
            fep[0].uchFlags    =  SHR_MEM_FREE;
            fep[0].aName       = 0;

            iiUnused = 1;
            for (ii = iiUnused; ii < iPages; ii++)
                {
                fep[ii].dwProcessID =  0;
                fep[ii].wOffset     =  0;
                fep[ii].wLength     =  0;
                fep[ii].sNext       = ii + 1;
                fep[ii].sPrior      = -1;
                fep[ii].uchLock     =  0;
                fep[ii].uchFlags    = SHR_MEM_INVALID;
                fep[ii].aName       = 0;
                }

            }

        //
        // Nobody should touch it unless allocated and locked.
        //
        Protect(lpMemBase, (WORD) (dwLength / dwPageSize), PAGE_NOACCESS);

        ReleaseSemaphore(hSemaphore, 1, NULL);

        return(TRUE);

        }
    else
        return (FALSE);

    }

VOID ShrDump()
{
    FEP fep = (FEP) lpCtrlBase;
    int ii = 0;

    WaitForSingleObject(hSemaphore, ALLFFS);

    printf( "Process ID\tOffset\tLength\tNext\tPrior\tLocks\tFlags\tAtom\n");

    while (ii != -1)
        {
        printf( "%8lx\t%6d\t%6d\t%+4d\t%+4d\t%6d\t%4x\t%#6x\n",
            fep[ii].dwProcessID,
            fep[ii].wOffset,
            fep[ii].wLength,
            fep[ii].sNext,
            fep[ii].sPrior,
            0x0000ffff & ((int) fep[ii].uchLock),
            0x0000ffff & ((int) fep[ii].uchFlags),
            0x0000ffff & ((int) fep[ii].aName));
        ii = fep[ii].sNext;
        }

    ReleaseSemaphore(hSemaphore, 1, NULL);

    return; 


}
BOOL    ShrName(DWORD hShr, LPCTSTR lpsz)
{
    ATOM aName;
    FEP fep = (FEP) lpCtrlBase;
    int ii = UNMUNGE(hShr);
    int jj = 0;

    WaitForSingleObject(hSemaphore, ALLFFS);

    //
    // You can only name an allocated segment with a unique
    // name that is a valid atom.
    //
    if ( ! (fep[ii].uchFlags & SHR_MEM_ALLOC) )
        {
        ReleaseSemaphore(hSemaphore, 1, NULL);
        return(FALSE);
        }

    if ((aName = GlobalAddAtom(lpsz)) == 0)
        {
        ReleaseSemaphore(hSemaphore, 1, NULL);
        return(FALSE);
        }

    // Look for the Atom already in the list.
    //
    while (fep[jj].sNext != -1 )
        {
        if (fep[jj].aName == aName)
            {
            GlobalDeleteAtom(aName);
            ReleaseSemaphore(hSemaphore, 1, NULL);
            return(FALSE);
            }
        else
            jj = fep[jj].sNext;
        }

    fep[ii].aName = aName;

    ReleaseSemaphore(hSemaphore, 1, NULL);

    return(TRUE);
}
DWORD   ShrGet(LPCTSTR lpsz)
{
    FEP fep = (FEP) lpCtrlBase;
    int ii = 0;
    ATOM aName;

    //
    // Atom must be defined for this to work.
    //
    if ((aName = GlobalFindAtom(lpsz)) == 0)
        return(0);

    WaitForSingleObject(hSemaphore, ALLFFS);

    //
    // Walk thru structures looking for an Atom Match
    //
    do
        {
        if (fep[ii].aName == aName)
            {
            ReleaseSemaphore(hSemaphore, 1, NULL);
            return(MUNGEIT(ii));
            }
        } while ((ii = fep[ii].sNext) != -1);

    ReleaseSemaphore(hSemaphore, 1, NULL);
    return(0);
}


