// ---------------------------------------------------------------------

// MODULE:      MAPDEBUG

// PURPOSE:     Contains the debug portions of MAPCON


// ---------------------------------------------------------------------

#include <Windows.H>
#include <Malloc.H>
#include <StdIO.H>
#include <StdLib.H>
#include <WinCon.H>
#include <String.H>
#include "MAPDEBUG.H"
#include "MAPCON.H"

	PDEBUGMAP		pDebugMap;
	UINT			iNumMaps;


// ---------------------------------------------------------------------

// FUNCTION:    OpenDebug

// PURPOSE:     Load the debug map file

// COMMENTS:    Return FALSE if error

// ---------------------------------------------------------------------


BOOL OpenDebug( VOID )
{
HANDLE		hMap;
DWORD		dwRead;
LPVOID		lpMap;
DWORD		MapSize;

	hMap = CreateFile( "MAPCON.COD", GENERIC_READ, FILE_SHARE_READ, 0L,
								 OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0L);
	if ( hMap == (HANDLE) -1 )
		return FALSE;
	ReadFile( hMap, &iNumMaps, sizeof( UINT ), &dwRead, NULL );
	if ( !iNumMaps )
		{
		 CloseHandle( hMap );
		 return FALSE;
		}
	MapSize = iNumMaps * sizeof( DEBUGMAP );
	lpMap = VirtualAlloc( NULL, MapSize, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE );
	if ( !lpMap )
		{
		 iNumMaps = 0;
		 CloseHandle( hMap );
		 return FALSE;
		}

	ReadFile( hMap, lpMap, MapSize, &dwRead, NULL );
 CloseHandle( hMap );
	if ( dwRead != MapSize )
		 iNumMaps = 0;
	pDebugMap = (PDEBUGMAP)lpMap;

	return TRUE;
}


// ---------------------------------------------------------------------

// FUNCTION:    HandleException

// FUNCTION:		An exception has occurred


// PURPOSE:     This function is an exception filter that handles exceptions
//              that occurr while executing.

// ARGUMENTS:
//              ExceptionCode - Reason for the exception.
//              ExceptionInfo - Information about the exception and the
//                              context that it occurred in.

// RETURNS:     Exception disposition code that tells the exception dispatcher
//              what to do with this exception. One of three values is returned:

//              EXCEPTION_EXECUTE_HANDLER - execute the exception handler
//              associated with the exception clause that called this filter
//              procedure.

//              EXCEPTION_CONTINUE_SEARCH - Continue searching for an exception
//              handler to handle this exception.

//              EXCEPTION_CONTINUE_EXECUTION - Dismiss this exception and
//              return control to the instruction that caused the exception.

// COMMENTS:
//              We always go to the EXECUTE_HANDLER clause, returning
//              to clean up and exit.

// ---------------------------------------------------------------------

int HandleException( DWORD ExceptionCode, PEXCEPTION_RECORD ExceptionInfo )
{
	UINT			FaultAddress[EXCEPTION_MAXIMUM_PARAMETERS+1];
	DWORD			dwIx;
	char			szMsg[256];
	char			szWhere[128];
	PDEBUGMAP		pDebugCurr;
	UINT			iDebugIdx;

	FaultAddress[0] = (UINT)ExceptionInfo->ExceptionAddress;
	for ( dwIx = 0;	dwIx < ExceptionInfo->NumberParameters;	dwIx++ )
		FaultAddress[dwIx+1] = (UINT)(ExceptionInfo->ExceptionInformation[dwIx]);

	switch ( ExceptionCode )
		{
		 case EXCEPTION_ACCESS_VIOLATION:
				sprintf( szMsg, "Access error. Ref to locn 0x%08X",	FaultAddress[2]);
				break;
		 case EXCEPTION_DATATYPE_MISALIGNMENT:
				strcpy( szMsg, "Data type misalignment" );
				break;
		 case EXCEPTION_BREAKPOINT:
				strcpy( szMsg, "Breakpoint instruction" );
				break;
		 case EXCEPTION_SINGLE_STEP:
				strcpy( szMsg, "Single step breakpoint" );
				break;
		 case EXCEPTION_ARRAY_BOUNDS_EXCEEDED:
				strcpy( szMsg, "Array bounds exceeded" );
				break;
		 case EXCEPTION_FLT_DENORMAL_OPERAND:
				strcpy( szMsg, "FLOAT: Denormalized operand" );
				break;
		 case EXCEPTION_FLT_DIVIDE_BY_ZERO:
				strcpy( szMsg, "FLOAT: Divide by zero" );
				break;
		 case EXCEPTION_FLT_INEXACT_RESULT:
				strcpy( szMsg, "FLOAT: Inexact result" );
				break;
		 case EXCEPTION_FLT_INVALID_OPERATION:
				strcpy( szMsg, "FLOAT: Invalid operation" );
				break;
		 case EXCEPTION_FLT_OVERFLOW:
				strcpy( szMsg, "FLOAT: Result overflow" );
				break;
		 case EXCEPTION_FLT_STACK_CHECK:
				strcpy( szMsg, "FLOAT: FP Stack check" );
				break;
		 case EXCEPTION_FLT_UNDERFLOW:
				strcpy( szMsg, "FLOAT: Result underflow" );
				break;
		 case EXCEPTION_INT_DIVIDE_BY_ZERO:
				strcpy( szMsg, "Integer divide by zero" );
				break;
		 case EXCEPTION_INT_OVERFLOW:
				strcpy( szMsg, "Integer result overflow" );
				break;
		 case EXCEPTION_PRIV_INSTRUCTION:
				strcpy( szMsg, "Privileged instruction" );
				break;
		 case CONTROL_C_EXIT:
				strcpy( szMsg, "Control-C interrupt" );
				break;
		 default:
				sprintf(szMsg,"Unknown code : 0x%08X", ExceptionCode );
		}
	pDebugCurr = pDebugMap;
	for ( iDebugIdx = 0; iDebugIdx < iNumMaps; iDebugIdx++ )
		{
		 if ( FaultAddress[0] < pDebugCurr->SymbAddr )
			break;
		 pDebugCurr++;
		}
	if ( ( iDebugIdx < iNumMaps ) && ( iDebugIdx > 0 ) )
		{
		 pDebugCurr--;
		 sprintf( szWhere, "%s(%s)+0x%X", pDebugCurr->Routine, pDebugCurr->Module,
												FaultAddress[0] - pDebugCurr->SymbAddr );
		}
	else
		sprintf( szWhere, "0x%08X", FaultAddress[0] );
	printf( "EXCEPTION : %s at %s.", szMsg, szWhere );

	return EXCEPTION_EXECUTE_HANDLER;

}


// ---------------------------------------------------------------------

// FUNCTION:    CloseDebug

// PURPOSE:     Free the debug map entries

// COMMENTS:    Return FALSE if error

// ---------------------------------------------------------------------


BOOL CloseDebug( VOID )
{
	if ( !pDebugMap )
		return FALSE;

	VirtualFree( pDebugMap, 0, MEM_DECOMMIT );
	VirtualFree( pDebugMap, 0, MEM_RELEASE );

	return TRUE;

}
