/************************************************************************\
* The enclosed files, "the software," is provided by 
* Microsoft Corporation "as is" without warranty of any kind. 
* MICROSOFT DISCLAIMS ALL WARRANTIES, EITHER EXPRESS OR IMPLIED, 
* INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY 
* AND FITNESS FOR A PARTICULAR PURPOSE.  You assume all risks of 
* using the software.
* 
* The software is Copyright (c) 1992 Microsoft Corporation.
* Original Author: John M. Hall, Microsoft SDE  9/1/92
*
* You are granted the right to freely distribute this software.
* You are granted the right to make changes provided this comment block
* is retained without modification and you acknowledge the changes.
* 
\************************************************************************/

//
// Hint: To read this source code skip down to main() at the end of
//       the file and read that first.
//
#define DEBUG
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <windows.h>
#include <string.h>
#include <memory.h>
#include <process.h>
#include "list.h"

#define SZ_DUMMY 10
#define SZ_INCREMENT  4096

COUNTER_INFO counter_info[MAX_NDX];
PPERFOBJECT pProcess;
PPERFOBJECT pThread;



DWORD      dwPerfDataLen = 0;
PPERFDATA  pPerfData;
HKEY       hKey;


//
//  Obtain the Performance data from the registry.
//  Note that the buffer size necessary is not known so an iterative
//  solution is chosen.
//
VOID GetSystemPerfData ()
   {  // GetSystemPerfData
   LONG           lError = ERROR_MORE_DATA;
   BOOL           bProcessing = TRUE;
   DWORD          dwSize = dwPerfDataLen;

   while (lError == ERROR_MORE_DATA)
       {
       lError = RegQueryValueEx (HKEY_PERFORMANCE_DATA, "Global", NULL, NULL,
                                 (LPSTR) pPerfData, &dwSize) ;

       if (lError == ERROR_MORE_DATA)
           {
           dwPerfDataLen += SZ_INCREMENT;
           dwSize = dwPerfDataLen;
           pPerfData = (PPERFDATA) realloc(pPerfData, dwPerfDataLen);
           }
       }
   assert(lError == ERROR_SUCCESS);

}  // GetSystemPerfData

//
// Record information on the counters we are interested in.
// counter_info has fields for counter type and offset.
// It also has an area for accumulating counter values.
//
VOID Do_Counters(PPERFCOUNTERDEF pCounter, int NumCounters)
{
    int ii;
    int iIndex;

    for (ii = 0; ii < NumCounters ; ii++)
        {
        switch (pCounter->CounterNameTitleIndex)
            {
            case P_PROCESSOR_TIME:  iIndex = NDX_PROCTIME; break;
            case T_PROCESSOR_TIME:  iIndex = NDX_THRDTIME; break;
            case P_USER_TIME     :  iIndex = NDX_PROCUSER; break;
            case P_KERNEL_TIME   :  iIndex = NDX_PROCPRIV; break;
            default:        iIndex = -1;           break;
            }

        if (iIndex != -1)
            {
            if (counter_info[iIndex].bFilledIn)
                printf( "Already filled in! %d\n", iIndex);

            counter_info[iIndex].bFilledIn = TRUE;
            counter_info[iIndex].dwLow = 0;
            counter_info[iIndex].dwHi = 0;
            counter_info[iIndex].CounterType = pCounter->CounterType;
            counter_info[iIndex].CounterOffset = pCounter->CounterOffset;
            }
        pCounter = NextCounter(pCounter);
        }
}


//
//  This function takes the instance number of the process
//  (PInstance) and the type of counter we are looking for
//  (NDX_THRDTIME in this example).
//
//  counter_info contains offset information about the counter
//  as well as a place to accumulate counter information.
//
//  The function then loops through all threads in the system
//  looking for a thread which has a parent object == PInstance.
//  [0 means none, so we protect against PInstance == 0].
//
//  On finding such a thread, the information is accumulated
//  in counter_info.
//
//  The return is a pointer to the data in counter_info[Index]
//
LPDWORD GetThreadData( DWORD PInstance, int iIndex)
{
    PPERFINSTANCEDEF pThreadInstance = FirstInstance(pThread);
    DWORD            NumThreads      = pThread->NumInstances;
    int              ii;
    LPDWORD          lpdw;

    // Clear the counter.
    counter_info[iIndex].dwLow = 0;
    counter_info[iIndex].dwHi  = 0;

    // ParentObject == 0 is none, so PInstance == 0 is not allowed.
    if (PInstance == 0)
        return(&counter_info[iIndex].dwLow);

    // For all threads
    for (ii = 0; ii < (int) NumThreads; ii++)
        {
        // Does it match the parent we are looking for?
        if (pThreadInstance->ParentObjectInstance == PInstance)
            {
            // Get offset of low 32bits of counter using information
            // stored in counter_info.
            //
            lpdw = (LPDWORD) ((PCHAR)GetCounterBlock(pThreadInstance)
                               + counter_info[iIndex].CounterOffset);

            counter_info[iIndex].dwLow += *lpdw;
            lpdw++;
            counter_info[iIndex].dwHi  += *lpdw;

            }
        pThreadInstance = NextInstance(pThreadInstance);
        }

    return(&counter_info[iIndex].dwLow);

}


//
// This example places a 64 bit counter (each counter == 100 nsec)
// into a string of 15 characters in either seconds (sec) milliseconds
// (msec) or microseconds (mmsec).
//
// All of the counters in this example were of this counter type.
//
#define MEG (1024.0*1024.0)
VOID FormatTime( LPDWORD lpdw, LPTSTR lptstr)
{
    FLOAT f;
    FLOAT fhi;

    if (*(lpdw +1) != 0)
        {
        fhi = ((FLOAT) *(lpdw + 1)) * MEG * 4096.0 / 10000000.0; //sec
        f   = ((FLOAT) *lpdw) / 10000000.0;  // seconds
        f  += fhi;
        sprintf( lptstr, " %8.2f sec  ", f);
        return;
        }

    f = ((FLOAT) *lpdw) / 10.0;  // micro seconds

    if (f < 1000.0 )
        sprintf( lptstr, " %8.2f mmsec", f);
    else
        {
        f /= 1000.0;   // milli seconds
        if (f < 1000.0)
            sprintf( lptstr, " %8.2f msec ", f);
        else
            {
            f /= 1000.0;   // seconds
            sprintf( lptstr, " %8.2f sec  ", f);
            }
        }
}

//
// Print information on all processes
//
VOID Do_Processes(PPERFINSTANCEDEF pInstance, int NumInstances)
{
    int ii;
    char chBuffer[32];
    char chName[72];
    LPDWORD lpdw;

    //
    // Print header
    //
    printf( "Instance Name");
    printf( "%15s", "Process Time");
    printf( "%15s", "User Time");
    printf( "%15s", "Kernel Time");
    printf( "%15s\n", "Thread Time");

    //
    // For all processes in system
    //
    for (ii = 0; ii < NumInstances; ii++)
        {
        //
        // Name is in Unicode! Convert to Ascii
        //
        wcstombs( chName, InstanceName(pInstance),
                pInstance->NameLength / sizeof(WCHAR));

        printf( "%13s", chName);

        //
        // This section gets the counter value, has FormatTime
        // format it, then prints it in the next column.
        //
        lpdw = (LPDWORD) ((PCHAR)GetCounterBlock(pInstance)
                               + counter_info[NDX_PROCTIME].CounterOffset);

        FormatTime(lpdw, chBuffer);
        printf( "%15s", chBuffer);

        lpdw = (LPDWORD) ((PCHAR)GetCounterBlock(pInstance)
                               + counter_info[NDX_PROCUSER].CounterOffset);

        FormatTime(lpdw, chBuffer);
        printf( "%15s", chBuffer);

        lpdw = (LPDWORD) ((PCHAR)GetCounterBlock(pInstance)
                               + counter_info[NDX_PROCPRIV].CounterOffset);
        FormatTime(lpdw, chBuffer);
        printf( "%15s", chBuffer);

        if (ii == 0)
            printf( "%15s", "None");
        else
            {
            lpdw = GetThreadData( ii, NDX_THRDTIME);
            FormatTime(lpdw, chBuffer);
            printf( "%15s", chBuffer);
            }

        printf("\n");
        pInstance = NextInstance(pInstance);
        }

}

//
// Main routine
//

main(int argc, char *argv[])
{
    LONG lErr;
    PPERFOBJECT pObject;
    int ii;
    int iNumObjects;


    //
    // Clear counter_info
    //
    for (ii = 0; ii < MAX_NDX; ii++)
        counter_info[ii].bFilledIn = FALSE;


    //
    // Obtain the system's performance data
    //
    dwPerfDataLen = SZ_INCREMENT;
    pPerfData = (PPERFDATA) malloc(dwPerfDataLen);

    GetSystemPerfData ();

    //
    // For all objects, find the ones of interest (PROCESS and THREAD)
    //
    iNumObjects = pPerfData->NumObjectTypes;

    ii = 0;
    pObject = FirstObject(pPerfData);

    while (ii < iNumObjects)
        {
        switch( pObject->ObjectNameTitleIndex)
            {
            default:
                break;

            case PROCESS :
                //
                // Record information about counters for this object
                //
                Do_Counters(FirstCounter(pObject), pObject->NumCounters);
                pProcess = pObject;
                break;

            case THREAD:
                Do_Counters(FirstCounter(pObject), pObject->NumCounters);
                pThread = pObject;
                break;
            }

        ii++;
        pObject = NextObject(pObject);
        }

    //
    // Print information
    //
    Do_Processes( FirstInstance(pProcess), pProcess->NumInstances);

    free(pPerfData);

    //
    // Close the key -- It helps performance.
    //
    lErr = RegCloseKey(HKEY_PERFORMANCE_DATA);
    assert( lErr == ERROR_SUCCESS);
    return(0);
}
