; bsacc.asm
; Helper routines for manipulation of master boot record
; and bootstrap loader code.

; From C, these routines appear thusly:
;
;   int ReadBootSector( char *sbuf, int hdrive, int size )
;   /* Read sbuf from sector 0 of hdrive, return error or 0. */
;
;   int WriteBootSector( char *sbuf, int hdrive, int size )
;   /* Write sbuf to sector 0 of hdrive, return error or 0. */
;
;   int CopyBootCode( char *sbuf )
;   /* Copy booter to sbuf and return its size. (= table start addr) */
;
;   int BootRunOffset( void )
;   /* Return offset of relocated booter relative to start of boot record */
;
;   void SetBootDrive( int hdrive )
;   /* Set which drive boot loads from. ( 0x80 by default ) */
;
;   int GetDiskStatus( void )
;   /* Return disk status byte from most recent operation. */


	TITLE   bsacc.asm

	.MODEL small, c

PBYTE   TYPEDEF      PTR BYTE   ; Type for pointer to bytes
PWORD   TYPEDEF      PTR WORD   ; Type for pointer to words
PSWORD  TYPEDEF      PTR SWORD  ; Type for pointer to integers
PDWORD  TYPEDEF      PTR DWORD  ; Type for pointer to integers
NPBYTE  TYPEDEF NEAR PTR BYTE   ; Type for near pointer to bytes
FPBYTE  TYPEDEF FAR  PTR BYTE   ; Type for far pointer to bytes
FPVOID  TYPEDEF FAR  PTR        ; Type for far pointer to void
PSEG    TYPEDEF WORD            ; Type for segment value

ReadBootSector PROTO C sbuf:PBYTE, hdrv:WORD, bufsz:WORD
WriteBootSector PROTO C sbuf:PBYTE, hdrv:WORD, bufsz:WORD
CopyBootCode PROTO C sbuf:PBYTE
BootRunOffset PROTO C

; timer hardware constants
TIMER_0 = 040h		; timer 0 data register I/O addr (lsb and/or msb)
TIMER_M = 043h		; timer mode register I/O addr
TM_READ_LATCH = 0	; mode value to latch counter 0 "on the fly"
T0_ST_LATCH = 0E2h	; command to latch counter 0 status only
T0_CNT_LATCH = 0D2h	; command to latch counter 0 count only

WAIT_TIME = 5	; Wait this many seconds before booting active partition

; Some ASCII codes
CR = 0Dh
LF = 0Ah

; Define structure of a boot menu item.
key_os struct
key		db 0	; what key selects this item
ose		db -1	; item enable flag or partition index
osname	dw offset mom + run_off	; pto Zstring OS name
key_os ends

; Define structure of a partition table entry.
part_spec struct
boot_flag db ?  ; This doubles as the drive, per IBM PC convention
beg_head  db ?
beg_sect  db ?  ; Sector (bits 543210) and hi cylinder (bits 76)
beg_cyl   db ?  ; lo cylinder
sys_type  db ?  ; aka OS signature
end_head  db ?
end_sect  db ?  ; Sector (bits 543210) and hi cylinder (bits 76)
end_cyl   db ?  ; lo cylinder
block_beg dd ?
block_cnt dd ?
part_spec ends

SECTOR_SIZE = 0200h
BS_MAGIC = 0AA55h

abs0    segment at 0

; Define malarky allowing image to be relocated, then address itself.
rloc_addr = 0600h	; where partition boot code is relocated
brun_addr = 0000h   ; where it appears (is assembled) to run
boot_addr = 7C00h	; where x86 PC's load 1st stuff from disk

; Compute offset of run address versus assembled address.  This is
; necessary because the booter image is relocated, then run.
run_off = rloc_addr - brun_addr

; Define where partition table is in boot sector , per PC convention.
part_off = SECTOR_SIZE - sizeof word - 4 * sizeof part_spec

  assume cs:abs0	; Give assembler a CS value, even tho no code here.

    org rloc_addr	; This is where sector 0 boot image is relocated
loadpt proc far		; so that code from partition to be booted can be
loadpt endp			; loaded into the original location.

    org boot_addr	; This is where PC's load sector 0 of a disk that
boot_entry proc far	; is to be booted.  When a partition is booted,
boot_entry endp		; its sector 0 must also be loaded and run here.

	org 0441h
disk_status db ?	; status from most recent disk operation

abs0    ends

	.CODE

GetDiskStatus  PROC C USES ds
	mov ax, seg disk_status
	mov ds, ax
  assume ds:abs0
	mov al, disk_status
	xor ah,ah
	ret
  assume ds:nothing
GetDiskStatus  ENDP

ReadBootSector PROC C USES ds es, sbuf:PBYTE, hdrv:WORD, bufsz:WORD
;|*** /*  */
;|*** {
;|*** 	if (bufsz != SECTOR_SIZE)
	cmp	bufsz,SECTOR_SIZE
	je rbs_1
;|*** 		return -1;
	mov	ax,-1
	jmp	rbs_x
rbs_1:
;|*** 	if (!diskread( 0, hdrv, sbuf ));
	; read in current partition table
	mov dx, hdrv
	mov dh, 0	; head
	mov cl, 1	; sector
	mov ch, 0	; cylinder
	mov al, 1	; xfr cnt
	mov bx, sbuf
	push ds
	pop es
	mov ah, 2  ; disk read
	int 13h   ; if BIOS call fails here, it's OK since booter also assumes
	jnc rbs_2  ; int 13h does disk I/O.  (This cannot be portable, anyway.)
;|*** 		return GetDiskStatus();
	invoke GetDiskStatus
	jmp rbs_x
;|*** else
rbs_2:
;|*** 		return 0;
	mov	ax,0
;|*** }
rbs_x:
	ret	
ReadBootSector	ENDP

WriteBootSector	PROC C USES ds es, sbuf:PBYTE, hdrv:WORD, bufsz:WORD
;|*** /*  */
;|*** {
;|*** 	if (bufsz != SECTOR_SIZE)
	cmp	bufsz,SECTOR_SIZE
	je	wbs_1
;|*** 		return -1;
	mov	ax,-1
	jmp	wbs_x
wbs_1:
;|*** 	if (!diskwrite( 0, hdrv, sbuf ))
	; write out new partition boot sector
	mov dx, hdrv
	mov dh, 0	; head
	mov cl, 1	; sector
	mov ch, 0	; cylinder
	mov al, 1	; xfr cnt
	mov bx, sbuf
	push ds
	pop es
	mov ah, 3  ; disk write
	int 13h
	jnc wbs_2
;|***		return GetDiskStatus();
	invoke GetDiskStatus
	jmp wbs_x
;|*** else
wbs_2:
;|*** 		return 0;
	mov	ax,0
;|*** }
wbs_x:
	ret	
WriteBootSector	ENDP

CopyBootCode	PROC C USES si di ds es, sbuf:PBYTE
;|*** /* Copy bootstrap loader code to sbuf and return its length */
;|*** {
	LOCAL n:SWORD
;|***	n =  &boot_code_end - &boot_entry;
	mov	cx, offset boot_code_end_ - offset boot_entry_
	mov n, cx
;|*** 	movmem( &booter, sbuf, n);
	push ds
	pop es
	mov di, sbuf
	mov ax, SEG boot_entry_
	mov ds, ax
	mov si, OFFSET boot_entry_
	cld
	rep movsb
;|*** 	return n;
	mov	ax,n
;|*** }
	ret	
CopyBootCode	ENDP

BootRunOffset	PROC C
;|*** /* Return offset of boot_sector[0] when relocated. */
;|*** {
;|*** 	return run_off - brun_addr;
	mov	ax, run_off - brun_addr;
;|*** }
	ret	
BootRunOffset	ENDP

SetBootDrive	PROC C USES ds, hdrv:SWORD
;|*** /* Set which drive booter loads from. */
	mov ax, seg hdrive
	mov ds, ax
  assume ds:_BOOTLD
	mov ax, hdrv
	mov hdrive, al
	ret
  assume ds:nothing
SetBootDrive	ENDP

; -------------- Sector 0 bootstrap loader ---------------

_BOOTLD  SEGMENT
	assume	cs:_BOOTLD, ds:_BOOTLD

	org brun_addr

boot_entry_:
; This is the entry point, and beginning, of the boot loader code.

; Set segment registers and stack pointer.
    xor ax, ax
    mov ss, ax	; (This disables ints for 1 instruction.)
    mov sp, offset boot_entry	; Set stack below this image.
  STI		; Allow ints, normally, from now on.
	mov ds, ax
	mov es, ax
; Move this code/data away from here, (where successor loader will load).
    cld
    mov si, sp
    mov di, offset loadpt
    mov cx, SECTOR_SIZE / 2
    rep movsw
	jmp far ptr abs_0:boot_con
boot_con_:
cont_off = offset boot_con_ - offset boot_entry_

; This digression sets where loader jumps to continue after relocating itself.
abs_0    segment at 0
  assume cs:abs_0
  org rloc_addr+cont_off
boot_con proc far
boot_con endp
abs_0    ends

	test byte ptr kos[run_off + sizeof key_os],0FFh
	je x0		   ; skip choices if only 1
    call put_menu  ; put the OS v Key menu
    call key_wto   ; get a key with timeout
    jz x0          ; timed out, take default path
    call put_bing  ; map key to partition, put message "Booting X ..."
    jnz x5         ; go use key if good, else treat as no key

x0:	
; Absent or invalid key input or single OS, so just boot active partition.
	; Prepare to scan partition table
    mov si, offset part_tab + run_off
    mov cx, length part_tab
x1:
    mov dl, byte ptr [si]
	neg dl	;  test for 0x80, 0 or other
	jo x5
	jnz x2
    add si, sizeof part_spec
	loop x1
    int 018h                  ; If none active, try to run ROM BASICK.
x2:
    mov si, offset ipm + run_off	; indicate invalid partition table
x3:
    call bios_msg_out
    jmp $	; loop tightly until reset

; Enter here with partition to boot addressed by si
x5:
    mov bp, si	; needed for DOS, somehow
    mov di, 4	; load boot sectors, 4 retries
x6:
    mov bx, offset boot_entry
    mov dh, [si+1]  ; DH = head
	mov dl, hdrive[run_off]	; DL = drive
    mov cx, [si+2]	; get CX = sector / cylinder
    mov ax, 0201h	; read sector, count = 1
    push di
    int 13h
    pop di
    jnb x7
    xor ax, ax	; disk reset
    int 13h
    dec di
    jnz x6		; retry
    mov si, offset lem + run_off
    jmp x3		; indicate system load error and wait for reset
x7:
    mov si, offset mom + run_off
    mov di, offset chk_val - offset boot_entry_ + offset boot_entry
    cmp word ptr [DI], BS_MAGIC
    jnz x3
    mov si, bp
    jmp far ptr boot_entry

chr_out proc near
; Put the char in AL to the screen, via BIOS video SWI
    push si
    push bx
    mov bx, 7
    mov ah, 0Eh
    int 10h
    pop bx
    pop si
    ret
chr_out endp

mo_prefix:	; part of bios_msg_out
	call near ptr chr_out
	; fall back into bios_msg_out

bios_msg_out proc near
; Put Zstring at SI to the screen, via BIOS video SWI
    lodsb
    cmp al, 0
    jnz  mo_prefix
    ret
bios_msg_out endp

rd_t0 proc near
; Return AL = MS byte of timer0.  Preserves all other registers.
	mov al, T0_CNT_LATCH		; tell timer to latch counter 0 value.
	out TIMER_M, al
	call near ptr rd_t0a	; call in(), then fall into it to effect return.
rd_t0a:
	in al, TIMER_0			; get LSB on 1st call, then MSB on fall thru
	ret
rd_t0 endp

key_wto proc near
; Get a key with timeout.  Return Z if timed out, else AL = keyin()
; Note that extended key codes (0,scancode) also return Z.
  PUSHF
    mov cx, (36 * WAIT_TIME) + (WAIT_TIME / 12)  ; about 2 * 18.15 Hz
kw_1:
  CLI	; holdoff ints for sake of timer read sequence
	call rd_t0
	mov ah, al
kw_2:	call rd_t0
	xor al, ah
	jns kw_2	; Wait for counter sign to change (twice per wrap)
  STI	; resume ints so key input can occur
    mov ah, 1
    int 16h		; BIOS call, keyboard hit?
    jz kw_3
    mov ah, 0	; yes, so get the key
    int 16h
    jmp kw_x
kw_3:
    loop kw_1
    xor ax, ax	; return Z thru common exit path
kw_x:
  POPF
    and al, not('a'xor'A')	; toupper(), assuming ASCII coding
    ret
key_wto endp

put_menu proc near
; Put menu of whatever OS's were enabled during setup.
    mov si, offset bmen1 + run_off
    mov bx, offset kos + run_off
pm_1:
	mov ax, word ptr (key_os ptr [bx]).key ; get key and flags members
	or ax, ax	; check for sentinel, enabled
    jz pm_x		; exit on sentinel
	js pm_2		; skip on not enabled
	; is enabled, put menu portion for this OS
	push ax
    call bios_msg_out	; separator or leader
	pop ax
    call chr_out		; say what key for this choice
    mov si, offset bmen2 + run_off
    call bios_msg_out	; say " to boot "
    mov si, (key_os ptr [bx]).osname
    call bios_msg_out	; name the OS for the key
    mov si, offset bmen3 + run_off	; set separator
pm_2:
    add bx, 4
    jmp pm_1
pm_x:
	call put_nl
    ret
put_menu endp

put_bing proc near
; Given key in AL, tell what will boot and return si pointing
; to selected partion table entry, or return Z if no match.
    mov bx, offset kos + run_off
pb_1:
    mov dx,  word ptr (key_os ptr [bx]).key	; get active flag (or sentinel)
    or dx, dx		; test for 0
    jz pb_x			; exit if 0
	js pb_1a		; if not active, not a choice
    xor dl, al		; compare select key with AL
    je pb_2			; if match, use to select partition
pb_1a:
    add bx, sizeof key_os	; else examine next entry
    jmp pb_1
pb_2:
    mov si, offset bing + run_off
    call bios_msg_out	; Put message "Will boot "
    mov si, (key_os ptr [bx]).osname
    call bios_msg_out	; Put OS name
    call put_nl		; and newline
    mov al, (key_os ptr [bx]).ose	; get selected partition index
    mov ah, sizeof part_spec
	mul ah
    add ax, offset part_tab + run_off
    mov si, ax
    or (part_spec ptr [si]).boot_flag, 080h	; Fake it was active.
pb_x:
    ret
put_bing endp

put_nl proc near
; Put a newline
	push si
	mov si, offset crlf + run_off
	call bios_msg_out
	pop si
	ret
put_nl endp

; Stingy messages.  See above modification tips before improving these.
bmen1 db 'Press ',0
bmen2 db ' to boot ',0
bmen3 db ', ',0
crlf  db CR,LF,0
bing  db 'Will boot ',0
ipm db 'Invalid partition table',0
lem db 'System load failed',0
mom db 'Missing system',0

hdrive db 080h	; Drive from which to load selected partition loader.

boot_code_end_:  ; Data from here to partition table is setup by mboot.

kos:
; This table specifies, for each OS, what key selects the OS, what the
; OS is to be named (for menu purposes only) and whether that OS is a
; choice at boot time.  (Whether it was found during multi-boot setup.)
; The values given are typical.  This table is overwritten by mboot.
	key_os < 'D', -1, offset os_dos + run_off   >
	key_os < 'U', -1, offset os_unix + run_off  >
	key_os < 'W', -1, offset os_winnt + run_off >
	key_os < 'O', -1, offset os_ostoo + run_off >
	dw 0  ; sentinel, MUST BE 0 and at end of array

; Names of OS's, for menu and 'Will boot X' msg
os_ostoo db 'OS/2',0
os_winnt db 'WinNT',0
os_unix db 'Unix',0
os_dos db 'DOS',0

bcodelimit:	;Mark upper limit of boot code.

    org brun_addr+part_off	;Set partition table at fixed location.

; Assert that pseudo-PC does not encroach upon partition table.
 .ERRE bcodelimit LE $, <Boot code tromples partition table.>

part_tab part_spec 4 dup(<>)

chk_val dw BS_MAGIC	; magic number that must be at end of boot sector

; Assert that pseudo-PC is exactly one sector past booter entry point.
 .ERRE $ EQ (offset boot_entry_+SECTOR_SIZE), <Incorrect boot sector image size.>

_BOOTLD  ENDS
; ----------------------------------------------------------

END
