#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "parttab.h"

char *Usage = "\
mboot [<option> ...]\n\
  where <option> is 0 or more of:\n\
  -d[1|2]  => specifys drive to be accessed (1 by default)\n\
  -i  => install selectable OS bootstrap loader\n\
  -v[<filename>]  => display partition table contents (1)\n\
  -s[<filename>]  => save master boot record (sector 0) to a file (2)\n\
  -r[<filename>]  => restore master boot record from a file (dangerous!)\n\
  -x[<filename>]  => same as -s and -r together (swaps file with sector 0)\n\
  Notes:\n\
  1. -v displays (filename given) THEN from file ELSE current boot record.\n\
  2. -s, -r and -x access old_ptab.bin if no filename is given.\n\
";

/* routines from bsacc.asm for manipulating master boot record */
extern int ReadBootSector( char *sbuf, int hdrv, int bufsz);
extern int WriteBootSector( char *sbuf, int hdrv, int bufsz);
extern int CopyBootCode( char *sbuf);
extern int BootRunOffset( void );
extern void SetBootDrive( int hdrv );

/* type of table built into booter to govern boot selection */
typedef struct {
	char key;	/* what key will select this partition */
	signed char ptx; /* partition entry number (index) */
	char *osname;    /* pto name of this selectable OS */
} key_os;

/* type of table built into this pgm to determine above build */
typedef struct {
	unsigned char ossig;
	key_os oskpn;
} sig_key_os;

/* for known OS's: signature value, select key, (ptx), and brief name */
sig_key_os known_os[] = {
	{ DOS_TYPE,   { 'D', -1, "DOS" } },
	{ UNIX_TYPE,  { 'U', -1, "Unix" } },
	{ WINNT_TYPE, { 'W', -1, "WinNT" } },
	{ OSTOO_TYPE, { 'O', -1, "OS/2" } },
	{ 0, { ' ', -1, NULL } }
};

struct {
	unsigned char part_sig;
	char *part_name;
} pt_signame_map[] = {
	/* From Andrew Schulman's Undocumented DOS */
	{ 0x01, "DOS, FAT12" },
	{ 0x02, "XENIX filesys" },
	{ 0x04, "DOS, FAT16" },
	{ 0x05, "DOS extended" },
	{ 0x06, "DOS Large filesys" },
	{ 0x07, "QNX" },
	{ 0x08, "AIX bootable" },
	{ 0x09, "AIX data" },
	{ 0x51, "Disk Manager" },
	{ 0x61, "SpeedStor" },
	{ 0x63, "Unix SystemV" },
	{ 0x64, "Novell NetWare" },
	{ 0x75, "PC/IX" },
	{ 0x80, "Minix v1.3-" },
	{ 0x81, "Minix v1.5+" },
	{ 0xDB, "CP/M" },
	{ 0xE1, "SpeedStor FAT12x" },
	{ 0xE4, "SpeedStor FAT16x" },
	{ WINNT_TYPE, "Windows NT" },
	{ OSTOO_TYPE, "OS/2" },
	{ 0, NULL }
};

char *ptName( unsigned int systyp )
{
	int n, stv;

	if (systyp == 0)
		return "Unused";

	for (n = 0; stv = pt_signame_map[n].part_sig; ++n)
		if (stv == systyp)
			return pt_signame_map[n].part_name;

	return "Unknown";
}

/* read/write buffers for partition table / boot sector */
char pbs[SECTOR_SIZE], xpbs[SECTOR_SIZE];

void ShowPartTable( part_table ptab )
{
	int n, k;
	char *title = "\
Beg(cyl hd sect) End(cyl hd sect) At_block N_blocks   Paritition type\n\
";
	char *content = "\
   %4d %2d %3d      %4d %2d %3d   %8ld %8ld %s%s (0x%02x)\n\
";

	printf( "%s", title );
	for (n = 0; n < 4; ++ptab, ++n) {
		k = ptab->sys_type;
		printf( content,
		  cylinder(ptab->beg_arc), ptab->beg_head, sector(ptab->beg_arc),
		  cylinder(ptab->end_arc), ptab->end_head, sector(ptab->end_arc),
		  ptab->block_beg, ptab->block_cnt,
		  (ptab->boot_flag)? "A " : "  ", ptName(k), k
		);
	}
}

int main(int an, char *av[])
{
	part_spec *ptab = (part_spec *)(pbs + (PART_OFF));

	int bootCodeSize, n, k, status;
	signed char osCount;
	FILE *fo, *fi;
	char *ofnm = 0, *ifnm = 0, *dfnm, *vfnm = 0;
	int showPT = 0, selBoot = 0, modBoot = 0, drive = 1;
	unsigned char sv;
	char *osNameSpace, *osn, opt, *osNS;
	key_os *osKeyTable;
	char fpn[_MAX_PATH];

	if (an < 2 || av[1][0] != '-') {
		printf( "Usage:\n%s", Usage );
		exit(1);
	}
	while (an > 1) {
		if (av[1][0] != '-')
parmYap:
			fprintf( stderr, "Skipping bad parameter: %s\n", av[1] );
		else
			switch ( opt = av[1][1] ) {
				case 'v':
					if (av[1][2] != 0)
						vfnm = &av[1][2];
					else if (an > 2 && av[2][0] != '-') {
						--an;
						++av;
						vfnm = av[1];
					}
					showPT = 1;
					break;
				case 'i':
					selBoot = 1;
					modBoot = 1;
					break;
				case 'd':
					drive = 1;
					if (sscanf( &av[1][2], "%d", &drive) != 1) {
						return 2;
					}
					break;
				case 'r': case 's': case 'x':
					if (av[1][2] != 0)
						dfnm = &av[1][2];
					else if (an > 2 && av[2][0] != '-') {
						--an;
						++av;
						dfnm = av[1];
					}
					else
						dfnm = "old_ptab.bin";
					switch( opt ) {
						case 'r':
							ifnm = dfnm;
							modBoot = 1;
							break;
						case 's':
							ofnm = dfnm;
							break;
						case 'x':
							ifnm = dfnm;
							ofnm = dfnm;
							modBoot = 1;
							break;
					}
					break;
				default: goto parmYap;
			}
		++av;
		--an;
	}
	if (ifnm && !_fullpath(fpn, ifnm, sizeof fpn)) {
		fprintf( stderr, "Invalid input filename: %s\n", ifnm );
		exit(2);
	}
	if (ofnm && !_fullpath(fpn, ofnm, sizeof fpn)) {
		fprintf( stderr, "Invalid output filename: %s\n", ofnm );
		exit(2);
	}
	if (vfnm && !_fullpath(fpn, vfnm, sizeof fpn)) {
		fprintf( stderr, "Invalid output filename: %s\n", vfnm );
		exit(2);
	}
	if (drive < 1 || drive > 2) {
		fprintf( stderr, "Invalid drive. (must be 1 or 2)\n" );
		exit(2);
	}
	SetBootDrive( 0x80 + (drive-1) );
	if (status = ReadBootSector( pbs, 0x80 + (drive-1), SECTOR_SIZE )) {
		fprintf( stderr, "Error %d reading master boot record.\n", status );
		return 1;
	}
	if (*(int *)(pbs+SECTOR_SIZE - sizeof(int)) != BS_MAGIC) {
		fprintf( stderr, "Not a master boot record. (wrong magic number).\n" );
		return 2;
	}
	if (showPT) {
		if (vfnm) {
			fi = fopen(vfnm, "rb");
			if (!fi) {
				fprintf(stderr, "Cannot open %s.\n", vfnm);
				return (1);
			}
			if (fread( xpbs, sizeof(char), SECTOR_SIZE, fi ) != SECTOR_SIZE) {
				fprintf(stderr, "Cannot read %s.\n", ifnm);
				return (2);
			}
			fclose(fi);
			ShowPartTable( (part_spec *)(xpbs + (PART_OFF)) );
		}
		else
			ShowPartTable( ptab );
	}
	if (ifnm) {
		fi = fopen(ifnm, "rb");
		if (!fi) {
			fprintf(stderr, "Cannot open %s.\n", ifnm);
			return (1);
		}
		if (fread( xpbs, sizeof(char), SECTOR_SIZE, fi ) != SECTOR_SIZE) {
			fprintf(stderr, "Cannot read %s.\n", ifnm);
			return (2);
		}
		fclose(fi);
	}
	if (ofnm) {
		fo = fopen(ofnm, "wb");
		if (!fo) {
			fprintf(stderr, "Cannot open %s for write.\n", ofnm);
			return (1);
		}
		if (fwrite( pbs, sizeof(char), SECTOR_SIZE, fo ) != SECTOR_SIZE) {
			fprintf(stderr, "Cannot write %s.\n", ofnm);
			return (2);
		}
		fclose(fo);
	}
	if (ifnm)
		for (n = 0; n < sizeof pbs; ++n)
			pbs[n] = xpbs[n];
	if (selBoot) {
		bootCodeSize = CopyBootCode( pbs );
		for (n = bootCodeSize; n < PART_OFF; ++n)
			pbs[n] = 0;

		osCount = 0;
		osKeyTable = (key_os *) &pbs[bootCodeSize];
		osKeyTable->key = 0;
		osKeyTable->ptx = -1;
		osNameSpace = &pbs[PART_OFF];
		printf("Bootable systems:");
		for (n = 0; n < PART_TAB_LEN; ++n) {
			/* see if this partition can boot a recognized OS */
			for (k = 0; sv = known_os[k].ossig; ++k)
				if (sv == ptab[n].sys_type)
					break;
			if (sv) {  /* known, so build choice into select table */
				osn = known_os[k].oskpn.osname;
				printf("  %s", osn);
				osNS = osNameSpace - (strlen(osn) + 1);
				/* Check for adequate space to add new entry. */
				if ( (char *)(&osKeyTable[osCount+1].osname) > osNameSpace ) {
					fprintf( stderr, "\nSelect table full. OS dropped.\n" );
					continue;
				}
				osNameSpace = osNS;
				strcpy( osNameSpace, osn );
				osKeyTable[osCount] = known_os[k].oskpn;
				osKeyTable[osCount].ptx = n;
				osKeyTable[osCount].osname = \
				  (char*)( osNameSpace - pbs + BootRunOffset() );
				++osCount;
				osKeyTable[osCount].key = 0;
				osKeyTable[osCount].ptx = -1;
			}
		}
		printf( "%s\n", (osCount > 0)? "" : "None" );
	}
	if (modBoot) {
		if (status = WriteBootSector( pbs, 0x80 + (drive-1), SECTOR_SIZE )) {
			fprintf(stderr, "Error %d writing master boot record.\n", status);
			return 3;
		}
		printf(
  "Selectable OS master boot record written to drive 0x%02x, sector 0.\n",
		  0x80 + (drive-1)
		);
	}
	return 0;
}
