; pickboot.asm -- A multiple OS boot utility for 80x86 PC's.

; Credits:
;   Modeled after bootstrap loaders provided by Microsoft (R) for
;   MSDOS (TM) v5 and Digital Research (R) for DR DOS (TM) v6
;   by Larry Brasfield.  (Actually, it just loads compatibly.)
;   Suggestions can be mailed to Compuserve, user # 70413,3332.

; Rights:
;   Placed into the public domain by Larry Brasfield, 6 Dec 1992,
;   except the right to be credited is retained.  No warranty
;   as to fitness for use is given or implied.  The user
;   assumes all risk arising from use of this program.

; Purpose:
;   This utility is designed to permit selection, at boot time, 
; of which OS partition boot sector to load.  This selectability
; appears at boot time in the form of a short (one line) menu,
; indicating what OS's may be booted and what keys will do so.
; If one of the offered choices is taken before a timeout, the
; selected OS is booted.  Otherwise, the active OS is booted.

; Method:
;   The multi-boot setup utility ("pickboot") operates by reading
; the current sector 0 on drive 0x80 to get the partition table,
; then scanning it for known OS signature bytes to determine which
; menu selections to enable.  The results of this scan are used to
; modify a boot loader which is appended to the program.  Finally,
; (if any OS is recognized), the (slightly) modified loader and a
; true copy of the original partition table are written back out.
;   The contained physical hard disk boot sector boots the active
; partition by default after displaying a boot selection menu for
; several seconds. * If a menu key is pressed before the timeout,
; the selected boot partition is booted.  Because the space left
; for code in the partition sector is strictly limited, there is
; no error check or retry when a wrong boot select key is hit.
; The penalty for pressing a wrong key is merely that the active
; (default) partition will be booted, just as if nothing was
; typed before the timeout.
;   * If only 1 bootable OS is recognized during setup, no menu
; is displayed and the active partition is booted without delay.
; If no bootable OS is recognized, sector 0 is left unmodified.

; Usage:
;   The pickboot utility is designed to run under MS-DOS.  It
; accepts no parameters.  It just goes to work and indicates
; completion status.  So, be brave (or backed-up) and just run it.

; Bugs, Caveats, Regrets:
; - The booter code embeds the assumption that a 8254 timer is at
; port address 0x40.  (This is true for PC-AT's and later clones.)
; It might be smarter to verify this assumption at setup time.

; Generation:
; This file can be assembled by Microsoft's Macro Assembler v6.0
; It will be necessary to assemble without /AT, using exe2bin to
; generate a .COM file, because the assembler gets upset about
; segments references in a "Tiny" program, even though all the
; addresses resolve to constants.  A makefile for this is:
;
; pickboot.com : pickboot.exe
;         exe2bin pickboot pickboot.com
;         del pickboot.exe
;
; pickboot.exe : pickboot.asm
;         ml pickboot.asm
;         del pickboot.obj


; Modification tips:
;   Right now, pickboot only "knows" about Unix and DOS OS's.
; This knowledge resides in 2 tables which are readily extended. 
; See "kos_init" and "kos" (and be sure they always track).  If
; these tables grow, it may be necessary to shorten some message(s)
; to keep the booter code from crowding the partition table.

WAIT_TIME = 5	; Wait this many seconds before booting active partition

; partition table OS signature values (sys_type)
DOS_TYPE = 6
UNIX_TYPE = 063h
WINNT_TYPE = 0 ; ?? This is wrong and cannot work.
OSTOO_TYPE = 0 ; ?? Ditto for this.

; timer hardware constants
TIMER_0 = 040h		; timer 0 data register I/O addr (lsb and/or msb)
TIMER_M = 043h		; timer mode register I/O addr
TM_READ_LATCH = 0	; mode value to latch counter 0 "on the fly"
T0_ST_LATCH = 0E2h	; command to latch counter 0 status only
T0_CNT_LATCH = 0D2h	; command to latch counter 0 count only

; Define structure of a boot menu item.
key_os struct
key		db 0	; what key selects this item
ose		db -1	; item enable flag or partition index
osname	dw offset mom + run_off	; pto Zstring OS name
key_os ends

; Define structure of a partition table entry.
part_spec struct
boot_flag db ?  ; This doubles as the drive, per IBM PC convention
beg_head  db ?
beg_sect  db ?
beg_cyl   db ?
sys_type  db ?
end_head  db ?
end_sect  db ?
end_cyl   db ?
block_beg dd ?
block_cnt dd ?
part_spec ends

; Some ASCII codes
CR = 0Dh
LF = 0Ah

abs0    segment at 0

; Define malarky allowing image to be relocated, then address itself.
rloc_addr = 0600h	; where partition boot code is relocated
brun_addr = 02C0h   ; where it appears (is assembled) to run
comm_addr = 0100h	; where loader read/modify/write code runs

SECTOR_SIZE = 0200h
BS_MAGIC = 0AA55h

; Compute offset of run address versus assembled address.  This is
; necessary because the booter image is relocated, then run.
run_off = rloc_addr - brun_addr

; Define where partition table is in boot sector , per PC convention.
part_off = SECTOR_SIZE - sizeof word - 4 * sizeof part_spec

boot_addr = 7C00h	; where x86 PC's load 1st stuff from disk

  assume cs:abs0	; Give assembler a CS value, even tho no code here.

    org rloc_addr	; This is where sector 0 boot image is relocated
loadpt proc far		; so that code from partition to be booted can be
loadpt endp			; loaded into the original location.

    org boot_addr	; This is where PC's load sector 0 of a disk that
boot_entry proc far	; is to be booted.  When a partition is booted,
boot_entry endp		; its sector 0 must also be loaded and run here.

abs0    ends

bseg segment

  assume cs:bseg, ds:bseg

	org comm_addr

; Entry point for setup utility (100h for .COM programs)
mbsetup:
	; read in current partition table
	mov dl, 080h	; drive
	mov dh, 0	; head
	mov cl, 1	; sector
	mov ch, 0	; cylinder
	mov al, 1	; xfr cnt
	mov bx, offset bsbuffer
	mov ah, 2  ; disk read
	int 13h   ; if BIOS call fails here, it's OK since booter also assumes
	jnc bg_1  ; int 13h does disk I/O.  (This cannot be portable, anyway.)
	mov si, offset dam  ; Disk Abort Message
	call dos_msg_out
	mov ax, 04C03h  ; DOS exit with error (3)
	int 21h
bg_1:
	; copy current partition table into new image
	mov cx, size part_tab / sizeof word
	mov si, offset bsbuffer + part_off
	mov di, offset part_tab
	cld
    rep movsw

	; setup boot selection menu
	mov si, offset eom  ; Say which OS's enabled.
	call dos_msg_out
	mov bx, offset part_tab
	mov cx, length part_tab
bg_2:
	mov ah, (part_spec ptr[bx]).sys_type
	mov si, offset kos_init	; pto known system type table
bg_2a:
	lodsb
	or al, al
	jz bg_2b	; check for type table sentinel
	cmp ah, al	; does it match this partition?
	jne bg_2a	; no, next type
	mov ax, si
	sub ax, (1 + offset kos_init)	; = offset of match
	mov ah, sizeof key_os
	mul ah
	add ax, offset kos ; = &kos[n]
	mov si, ax
	mov ah, length part_tab
	sub ah, cl		; compute which partition was matched here
	mov  (key_os ptr [si]).ose, ah	; tell which partition to boot
	inc byte ptr os_cnt
	mov si, (key_os ptr [si]).osname;
	sub si, run_off
	call dos_msg_out
	mov si, offset ssm
	call dos_msg_out
bg_2b:
	add bx, size part_tab / length part_tab
	loop bg_2

	; Check how many bootable OS's were found.
	cmp byte ptr os_cnt, 1
	ja bg_4
	je bg_3
	mov si, offset nos	; none found, say so and forego writeout
	jmp bg_5
bg_3:
	; just 1 found, disable selection process (and wait)
	mov si, offset nmm	; say single OS, no menu
	call dos_msg_out
bg_4:
	; write out new partition boot sector
	mov dl, 080h	; drive
	mov dh, 0	; head
	mov cl, 1	; sector
	mov ch, 0	; cylinder
	mov al, 1	; xfr cnt
	mov bx, offset bentry
	mov ah, 3  ; disk write
	int 13h
	jnc bg_6
	mov si, offset dam  ; Disk Abort Message
bg_5:
	call dos_msg_out
	mov ax, 04C03h		; DOS error exit
	int 21h
bg_6:
	mov si, offset okm  ; OK Message
	call dos_msg_out
	mov ax, 04C00h		; DOS OK exit
	int 21h

eom db 'OS types enabled: ',0
nos db 'None',CR,LF
    db 'Sector 0 (booter and partition table) unchanged.',CR,LF,0
ssm db ' ',0
nmm db CR,LF,'Only 1 bootable OS found; Selectable boot is disabled.',0
okm db CR,LF,'Selectable partition booter written to sector 0.',CR,LF,0
dam db 'Cannot access partition boot sector -- aborting.',CR,LF,0

dos_msg_out proc near
; Put message at si, using DOS calls.  Preserve regs.
	push ax
	push dx
	cld
dmo_1:
	lodsb
	or al, al
	jz dmo_x
	mov dl, al
	mov ah, 2
	int 21h
	jmp dmo_1
dmo_x:
	pop dx
	pop ax
	ret
dos_msg_out endp

kos_init:
; NOTE: This table must be ordered identically to kos, below.
	db DOS_TYPE
	db UNIX_TYPE
	db WINNT_TYPE
	db OSTOO_TYPE
	db 0 ; sentinel MUST BE 0 at end of OS signature list

; Assert that setup code does not collide with booter.
; (This would be unnecessary if the assembler knew of relocation.)
 .ERRE ($-mbsetup) LE (brun_addr-comm_addr), <Booter image too low.>

; -------------- Sector 0 bootstrap loader ---------------

    org brun_addr
bentry:
; This is the entry point, and beginning, of the boot loader code.

; Set segment registers and stack pointer.
    xor ax, ax
    mov ss, ax	; (This disables ints for 1 instruction.)
    mov sp, offset boot_entry	; Set stack below this image.
  STI		; Allow ints, normally, from now on.
	mov ds, ax
	mov es, ax
; Move this code/data away from here, (where successor loader will load).
    cld
    mov si, sp
    mov di, offset loadpt
    mov cx, SECTOR_SIZE / 2
    rep movsw
    jmp far ptr loadpt[offset bcontinue - offset bentry]

bcontinue:
	dec byte ptr os_cnt[run_off]
	jle x0		; skip choices if only 1
    call put_menu  ; put the OS v Key menu
    call key_wto   ; get a key with timeout
    jz x0          ; timed out, take default path
    call put_bing  ; map key to partition, put message "Booting X ..."
    jnz x5         ; go use key if good, else treat as no key

x0:	
; Absent or invalid key input or single OS, so just boot active partition.
	; Prepare to scan partition table
    mov si, offset part_tab + run_off
    mov cx, length part_tab
x1:
    mov dl, byte ptr [si]
	neg dl	;  test for 0x80, 0 or other
	jo x5
	jnz x2
    add si, sizeof part_spec
	loop x1
    int 018h                  ; If none active, try to run ROM BASICK.
x2:
    mov si, offset ipm + run_off	; indicate invalid partition table
x3:
    call bios_msg_out
    jmp $	; loop tightly until reset

; Enter here with partition to boot addressed by si
x5:
    mov bp, si	; needed for DOS, somehow
    mov di, 4	; load boot sectors, 4 retries
x6:
    mov bx, offset boot_entry
    mov dx, [si]  ; get DL = drive, DH = head
    mov cx, [si+2] ; get CX = sector / cylinder
    mov ax, 0201h	; read sector, count = 1
    push di
    int 13h
    pop di
    jnb x7
    xor ax, ax	; disk reset
    int 13h
    dec di
    jnz x6		; retry
    mov si, offset lem + run_off
    jmp x3		; indicate system load error and wait for reset
x7:
    mov si, offset mom + run_off
    mov di, offset chk_val - offset bentry + offset boot_entry
    cmp word ptr [DI], BS_MAGIC
    jnz x3
    mov si, bp
    jmp far ptr boot_entry

chr_out proc near
; Put the char in AL to the screen, via BIOS video SWI
    push si
    push bx
    mov bx, 7
    mov ah, 0Eh
    int 10h
    pop bx
    pop si
    ret
chr_out endp

mo_prefix:	; part of bios_msg_out
	call near ptr chr_out
	; fall back into bios_msg_out

bios_msg_out proc near
; Put Zstring at SI to the screen, via BIOS video SWI
    lodsb
    cmp al, 0
    jnz  mo_prefix
    ret
bios_msg_out endp

rd_t0 proc near
; Return AL = MS byte of timer0.  Preserves all other registers.
	mov al, T0_CNT_LATCH		; tell timer to latch counter 0 value.
	out TIMER_M, al
	call near ptr rd_t0a	; call in(), then fall into it to effect return.
rd_t0a:
	in al, TIMER_0			; get LSB on 1st call, then MSB on fall thru
	ret
rd_t0 endp

key_wto proc near
; Get a key with timeout.  Return Z if timed out, else AL = keyin()
; Note that extended key codes (0,scancode) also return Z.
  PUSHF
    mov cx, (36 * WAIT_TIME) + (WAIT_TIME / 12)  ; about 2 * 18.15 Hz
kw_1:
  CLI	; holdoff ints for sake of timer read sequence
	call rd_t0
	mov ah, al
kw_2:	call rd_t0
	xor al, ah
	jns kw_2	; Wait for counter sign to change (twice per wrap)
  STI	; resume ints so key input can occur
    mov ah, 1
    int 16h		; BIOS call, keyboard hit?
    jz kw_3
    mov ah, 0	; yes, so get the key
    int 16h
    jmp kw_x
kw_3:
    loop kw_1
    xor ax, ax	; return Z thru common exit path
kw_x:
  POPF
    and al, not('a'xor'A')	; toupper(), assuming ASCII coding
    ret
key_wto endp

put_menu proc near
; Put menu of whatever OS's were enabled by pickboot.
    mov si, offset bmen1 + run_off
    mov bx, offset kos + run_off
pm_1:
	mov ax, word ptr (key_os ptr [bx]).key ; get key and flags members
	or ax, ax	; check for sentinel, enabled
    jz pm_x		; exit on sentinel
	js pm_2		; skip on not enabled
	; is enabled, put menu portion for this OS
	push ax
    call bios_msg_out	; separator or leader
	pop ax
    call chr_out		; say what key for this choice
    mov si, offset bmen2 + run_off
    call bios_msg_out	; say " to boot "
    mov si, (key_os ptr [bx]).osname
    call bios_msg_out	; name the OS for the key
    mov si, offset bmen3 + run_off	; set separator
pm_2:
    add bx, 4
    jmp pm_1
pm_x:
	call put_nl
    ret
put_menu endp

put_bing proc near
; Given key in AL, tell what will boot and return si pointing
; to selected partion table entry, or return Z if no match.
    mov bx, offset kos + run_off
pb_1:
    mov dx,  word ptr (key_os ptr [bx]).key	; get active flag (or sentinel)
    or dx, dx		; test for 0
    jz pb_x			; exit if 0
	js pb_1a		; if not active, not a choice
    xor dl, al		; compare select key with AL
    je pb_2			; if match, use to select partition
pb_1a:
    add bx, sizeof key_os	; else examine next entry
    jmp pb_1
pb_2:
    mov si, offset bing + run_off
    call bios_msg_out	; Put message "Will boot "
    mov si, (key_os ptr [bx]).osname
    call bios_msg_out	; Put OS name
    call put_nl		; and newline
    mov al, (key_os ptr [bx]).ose	; get selected partition index
    mov ah, sizeof part_spec
	mul ah
    add ax, offset part_tab + run_off
    mov si, ax
    or (part_spec ptr [si]).boot_flag, 080h	; Fake it was active.
pb_x:
    ret
put_bing endp

put_nl proc near
; Put a newline
	push si
	mov si, offset crlf + run_off
	call bios_msg_out
	pop si
	ret
put_nl endp

os_cnt db 0	; count of OS's found during setup (used to skip menu)

; Names of OS's, for menu and 'Will boot X' msg
os_dos db 'DOS',0
os_unix db 'Unix',0
os_winnt db 'WinNT',0
os_ostoo db 'OS/2',0

kos:
; This table specifies, for each OS, what key selects the OS, what the
; OS is to be named (for menu purposes only) and whether that OS is a
; choice at boot time.  (Whether it was found when pickboot was run.)
; NOTE: This table must be ordered identically to kos_init, above.
	key_os < 'D', -1, offset os_dos + run_off   >
	key_os < 'U', -1, offset os_unix + run_off  >
	key_os < 'W', -1, offset os_winnt + run_off >
	key_os < 'O', -1, offset os_ostoo + run_off >
	dw 0  ; sentinel, MUST BE 0 and at end of array

; Stingy messages.  See above modification tips before improving these.
bmen1 db 'Press ',0
bmen2 db ' to boot ',0
bmen3 db ', ',0
crlf  db CR,LF,0
bing  db 'Will boot ',0
ipm db 'Invalid partition table',0
lem db 'System load failed',0
mom db 'Missing system',0

bcodelimit:	;Mark upper limit of boot code.

    org brun_addr+part_off	;Set partition table at fixed location.

; Assert that pseudo-PC does not encroach upon partition table.
 .ERRE bcodelimit LE $, <Boot code tromples partition table.>

part_tab part_spec 4 dup(<>)

chk_val dw BS_MAGIC	; magic number that must be at end of boot sector

; Assert that pseudo-PC is exactly one sector past booter entry point.
 .ERRE $ EQ (offset bentry+SECTOR_SIZE), <Incorrect boot sector image size.>

bsend:	; mark end of partition boot sector

bsbuffer db SECTOR_SIZE dup(?) ; buffer for partition sector during setup

bseg    ends

END mbsetup
