/*
 *  netd.c -- 
 *      Net daemon for PC NFS file server.  It sets up and
 *      arbitrates the port mapper daemon, mount daemon and
 *      the NFS server.
 *
 *  Author:
 *      See-Mong Tan, 6/11/88
 *  Modified by:
 *      Rich Braun, 3/29/91
 *
 *  Revision history:
 *  
 * $Log: netd.c_v $
 * Revision 1.5  1991/05/13  17:44:55  richb
 * Add the -b command line option to specify read/write request size.
 *
 * Revision 1.4  1991/04/11  20:39:04  richb
 * Add -t option and use standard parser (getopt).
 *
 */

#ifdef RCSID
static char _rcsid_ = "$Id: netd.c_v 1.5 1991/05/13 17:44:55 richb Exp $";
#endif

#include "common.h"

extern int getopt();
extern int sock_start();
extern void svc_init();
extern long coreleft();

bool_t NFS_VERBOSE = FALSE;	/* flag for nfs verbose mode */
bool_t NFS_READONLYFS = FALSE;	/* true if started as a read only filesystem */
bool_t NFS_TRUNCATENAMES = FALSE; /* true if long names should be truncated */
int    nfsrd_size = RD_SIZ;	/* I/O read default size */
int    nfswr_size = WR_SIZ;	/* I/O write default size */

main(argc, argv)
	int argc;
	char *argv[];
{
time_t now;

	netd_init(argc, argv);		/* initialize netd */

	time (&now);
	(void) printf("SOSS v%s %s (compiled %s)\n\nStarting net services at %s",
		      VERSION_NUM, VERSION_DATE, __DATE__, ctime (&now));

	sock_start();
	svc_init();

	/* initialize and register services */
	if (! pmap_init())
		netd_Punt("portmapper");
	DBGPRT0 (nfsdebug, "netd: port mapper created");

	if (! mountd_init())
		netd_Punt("mount daemon");
	DBGPRT0 (nfsdebug, "netd: mount daemon created");

	if (! nfs_init())
		netd_Punt("nfs server");
	DBGPRT0 (nfsdebug, "netd: nfs server created");

	if (! inode_init())
		netd_Punt("inode interface");

	if (! dtime_init())
		netd_Punt("dos time interface");

	(void) printf("netd: port mapper, mountd and nfs server running\n\n");
	DBGPRT1 (nfsdebug, "Memory available = %ld\n", coreleft());
	(void) svc_run();       /* wait for and service net requests */
	netd_Punt("net daemon returned");
	return 0;
}

long
coreleft(void)
{
	MEMORYSTATUS m;
	GlobalMemoryStatus(&m);
	return (long) m.dwAvailPageFile;
}


/*
 *  void netd_Punt(char *s) --
 *      Prints net daemon error message and exits.
 *      For irrecoverable errors.
 */
void netd_Punt(s)
	char *s;
{
	int i;

	(void) fprintf(stderr, "net daemon error: %s\n", s);
	/* Does this really work? */
	for(i = 0; i < 3; i++)
		closesocket(i);
	exit(1);
}

int
netd_daemon(void)
{
	SECURITY_ATTRIBUTES saPipe; 
	PROCESS_INFORMATION pi;
	STARTUPINFO si;  /* for CreateProcess call */
	int r;
	char *cmd = "soss";

	saPipe.nLength = sizeof(SECURITY_ATTRIBUTES);
	saPipe.lpSecurityDescriptor = NULL;
	saPipe.bInheritHandle = FALSE;

	memset(&si, 0, sizeof(si));
	si.cb = sizeof(si);
	r = CreateProcess(NULL,  /* filename */
		cmd,  /* full command line for child */
		NULL,  /* process security descriptor */
		NULL,  /* thread security descriptor */
		FALSE,  /* inherit handles? */
		0,  /* creation flags */
		NULL,  /* inherited environment address */
		NULL,  /* startup dir; NULL = start in current */
		&si,  /* pointer to startup info (input) */
		&pi);  /* pointer to process info (output) */
	if (!r) {
		fprintf(stderr,"CreateProcess() failed!\n");
		return 1;
	}
	CloseHandle(pi.hThread);
	CloseHandle(pi.hProcess);
	return 0;
}

/*
 *  void netd_init(int argc, char **argv) --
 *      Initializes the net daemon.  Should be called before any other
 *      routine in the server.
 */
void netd_init(argc, argv)
	int argc;
	char *argv[];
{
        dbg_init ();
	netd_parsecmdln(argc, argv);	/* parse command line */
	signal(SIGINT, netd_break);	/* break handler */
}

/*
 *  void netd_break() --
 *      Break handler.  Closes all sockets and exits.
 */
void netd_break(int x)
{
	int i;

	(void) fprintf(stderr, "Netd:  break caught... exiting\n");
	for(i = 0; i < 3; i++)
		closesocket(i);

	exit(1);
}

/*
 *  void netd_parsecmdln(int argc, char **argv) --
 *      Parse command line arguments.
 */
void netd_parsecmdln(argc, argv)
	int argc;
	char *argv[];
{
    int c;
    extern int optind;
    extern char *optarg;
    int err = FALSE;

    while ((c = getopt (argc, argv, "b:drtv")) != -1)
      switch (c) {
	case 'b':       if (sscanf (optarg, "%d", &nfsrd_size) == 1)
	  		   nfswr_size = nfsrd_size;
	  		else
			   err = TRUE;
	  		break;
	case 'd':	printf("NETD_DAEMON!\n"); netd_daemon(); exit(0);
	case 'r':	NFS_READONLYFS = TRUE;    break;
	case 't':	NFS_TRUNCATENAMES = TRUE; break;
	case 'v':	NFS_VERBOSE = TRUE;	  break;
	default:        err = TRUE;
      }
      if (err || optind < argc)
	netd_Punt("Usage: soss [ -b blocksize -rtv ]");
}		
