{**************************************************}
{                     Org 0.6                      }
{                    Written in                    }
{             Turbo Pascal for Windows             }
{                       by                         }
{                  Zack Urlocker                   }
{                    08/13/92                      }
{**************************************************}

program Org;

{ This is a simple implementation of an org charting program written
  in Turbo Pascal for Windows using the ObjectWindows application
  framework and the ObjectGraphics library from The Whitewater Group. 
  The program is divided into several object types:

  TOrgApplication      --creates and shows the main window
  TMainWindow          --main window for toolbar, drawing area
  TOrgWindow           --the drawing area
  TEmpRect             --employee graphic object
  TDataDialog          --dialog for adding employee info
  TRetEdit             --edit control that traps Enter key
}

{$R Org1.res}          { Link in resources }

uses OGL1, OGL2, OGL3, RGBDialog,   { ObjectGraphics }
     WObjects, StdDlgs, Strings, WinTypes, WinProcs, WinDos,
     ToolBar, OrgWin;

const
 cm_New    = 501;       { File menu items }
 cm_Open   = 502;
 cm_Save   = 503;
 cm_SaveAs = 504;
 cm_Print  = 505;
 cm_PrChnge= 506;
 cm_PrSetup= 507;
 cm_Exit   = 598;
 cm_About  = 599;
 cm_Cut    = 201;       { Edit menu items }     
 cm_Copy   = 202;
 cm_Paste  = 203;
 cm_Clear  = 204;
 cm_Grid   = 100;       { Tool menu items }
 cm_Width  = 101;                    
 cm_Color  = 102;
 cm_Select = 109;
 cm_Line   = 110;
 cm_DotLine= 111;
 cm_EmpRect = 112;
 cm_Rect   = 113;
 cm_RndRect= 114;
 cm_Text   = 115;
 cm_ViewGrid=300;       { View menu items }
 cm_Data   = 309;      
 cm_Help   = 699;

type
  { Main application object }
  TOrgApp = object(TApplication)
    procedure InitInstance; virtual;
    procedure InitMainWindow; virtual;
  end;

  { Main window contains toolbar and drawing area }
  PMainWindow = ^TMainWindow;
  TMainWindow = object(TGWindow)
    FileName    : array [0..fsPathName] of Char;
    Printer     : PGPrinter;
    ToolBar     : PToolBar; 
    OrgWindow   : POrgWindow; { drawing area }
    Saved: Boolean;           { has work been saved?  }
    ViewGrid: Boolean;        { Is grid displayed?    }
    constructor Init(AParent: PWindowsObject; ATitle: PChar);
    procedure SetSpace(NewSpace: PGraphSpace);virtual;
    procedure SetupWindow;virtual;
    function  CanClose: Boolean;virtual;
    { File menu items }
    procedure NewFile(var Msg: TMessage); virtual cm_First + cm_New;
    procedure Open(var Msg: TMessage); virtual cm_First + cm_Open;
    procedure Save(var Msg: TMessage); virtual cm_First + cm_Save;
    procedure SaveAs(var Msg: TMessage); virtual cm_First + cm_SaveAs;
    procedure ChoosePrinter(var Msg: TMessage); virtual cm_First + cm_PrChnge;
    procedure PrinterConfig(var Msg: TMessage); virtual cm_First + cm_PrSetup;
    procedure Print(var Msg: TMessage); virtual cm_First + cm_Print;
    procedure Exit(var Msg: TMessage); virtual cm_First + cm_Exit;
    procedure About(var Msg: TMessage); virtual cm_First + cm_About;
    procedure Help(var Msg: TMessage); virtual cm_First + cm_Help;
    { Tool selection menu items}
    procedure ChangeGrid(var Msg: TMessage);virtual cm_First + cm_Grid;
    procedure Color(var Msg: TMessage);virtual cm_First + cm_Color;
    procedure Width(var Msg: TMessage);virtual cm_First + cm_Width;
    procedure DefCommandProc(var Msg: TMessage);virtual;
    { View menu items }
    procedure ToggleGrid(var Msg: TMessage);virtual cm_First + cm_ViewGrid;
    procedure Data(var Msg: TMessage);virtual cm_First + cm_Data;
    {Window Messages}
    procedure WMSize(var Msg: TMessage); virtual wm_First + wm_Size;
    procedure WMSetFocus(var Msg: TMessage); virtual WM_SETFOCUS;
    {Utility Methods}
    procedure SetCaption;virtual;
    procedure SizeKids;virtual;
  end;

{ Initialize main window cursor, icon, toolbar, style etc }
constructor TMainWindow.Init(AParent: PWindowsObject; ATitle: PChar);
var
  AppIcon: TIcon;
const
  { constants for setting up toolbar bitmaps to commands }
  Bitmaps: BitmapArray = ('Open', 'Save',
                          'Select', 'EmpRect', 'Rect', 'RndRect',
                          'Line', 'DotLine', 'Text',
                          'Grid', 'Width', 'Color');

  IDs : IDArray = (cm_Open, cm_Save,
                   cm_Select, cm_EmpRect, cm_Rect, cm_RndRect,
                   cm_Line, cm_DotLine, cm_Text,
                   cm_Grid, cm_width, cm_color);              

begin
  { Register object storage facilities }
  RegisterWObjects;
  RegisterOGL;             
  TGWindow.Init(AParent, ATitle);
  { Set menu, icon, cursor, style }
  Attr.Menu := LoadMenu(HInstance, 'commands');
  AppIcon.Init(0,0, 'orgIco');
  SetIcon(@AppIcon);
  AppIcon.Done;
  SetGCursor(cs_ThinCross);
  Attr.Style:= ws_OverlappedWindow or ws_Maximize;
  New(Brush, InitDefault);
  Brush^.Pattern := bp_Invisible;
  { Initialize other data }
  Saved := True;
  ViewGrid := True;
  StrCopy(FileName, '(Untitled)');
  New(Printer,    Init(@Self));
  New(ToolBar,    Init(@Self));
  Toolbar^.setIDs(IDs);
  Toolbar^.setBitmaps(BitMaps);
  New(OrgWindow, Init(@Self, 'Org'));
end;

{ Ensure proper size of child windows etc after init }
procedure TMainWindow.SetupWindow;
var
  CRect: TMathRect;
begin
  CRect.Init(20,20, 600,440);
  SetClientRect(@CRect);
  CRect.Done;
  SizeKids;
  TGWindow.SetupWindow;
  SetCaption;
end;

{ Resize main window by resizing child windows }
procedure TMainWindow.WMSize(var Msg: TMessage);
begin
  SizeKids;
  OrgWindow^.Invalidate;
end;

{ Compute size of child windows when resizing }
procedure TMainWindow.SizeKids;
var
  CRect: TMathRect;
begin
  CRect.InitDefault;
  GetDisplayRect(CRect);
  CRect.Origin^.Build(30, 0);
  OrgWindow^.SetClientRect(@CRect);
  CRect.Build(0, 0, 30, CRect.Bottom);
  ToolBar^.SetClientRect(@CRect);
  CRect.Done;
end;

{ Update graphspace of child windows }
procedure TMainWindow.SetSpace(NewSpace: PGraphSpace);
begin
  if NewSpace <> nil then
  begin
    TGWindow.SetSpace(NewSpace);
    OrgWindow^.Invalidate;
  end;
end;

{ set the window caption }
procedure TMainWindow.SetCaption;
var
  TmpTitle: array [0..80] of Char;
begin
  StrCopy(TmpTitle, Attr.Title);
  StrCat(TmpTitle, ' - ');
  SetWindowText(HWindow, StrLCat(TmpTitle, FileName, 80));
end;


{ File menu response methods }

{ Erase the current drawing }
procedure TMainWindow.NewFile(var Msg: TMessage);
begin
  if (not Saved) or
     (MessageBox(HWindow, 'Are you sure?',
      'File New', mb_YesNo or mb_IconQuestion)= id_Yes)
  then
  begin
    StrCopy(FileName, '(Untitled)');
    SetCaption;
    Saved := True;
    OrgWindow^.ClearAll;
  end;
end;

{ Open a new graphics file streamed out on disk }
procedure TMainWindow.Open(var Msg: TMessage);
var
  TmpFile   : array [0..fsPathName] of Char;
  APicture  : PPicture;
  ASpace    : PGraphSpace;
  FileStream: TGStream;
  Header    : TFileHeader;
begin
  if Application^.ExecDialog(New(PFileDialog, Init(@Self,
                                 PChar(sd_FileOpen),
                                 StrCopy(TmpFile, '*.org'))))
   = id_Ok
  then
  begin
    StrCopy(FileName, TmpFile);
    FileStream.Init(FileName, stOpen, 10000);
    if ReadOGLHeader(FileStream, Header) then
    begin
      ASpace  := PGraphSpace(FileStream.Get);
      APicture:= PPicture(FileStream.Get);
    end;
    FileStream.Done;
    OrgWindow^.SetSpace(ASpace);
    OrgWindow^.SetPicture(APicture);
    OrgWindow^.invalidate;
    SetCaption;
    Saved := True;
  end;
end;

{ Save the current drawing to disk as stream }
procedure TMainWindow.Save(var Msg: TMessage);
var
  FileStream: TGStream;
  FileSize  : LongInt;
begin
  if StrIComp(FileName, '(Untitled)') = 0 then
    SaveAs(Msg)
  else
  begin
    FileSize := SizeOf(TFileHeader)
              + StoreSize(OrgWindow^.Space)
              + StoreSize(OrgWindow^.Picture);
    FileStream.Init(FileName, stCreate, FileSize);
    WriteOGLHeader(FileStream, ff_Standard);
    FileStream.Put(OrgWindow^.Space);
    FileStream.Put(OrgWindow^.Picture);
    FileStream.Done;
    Saved := True;
  end;
end;

{ Save the current drawing under a new name }
procedure TMainWindow.SaveAs(var Msg: TMessage);
var
  TmpFile   : array [0..fsPathName] of Char;
  FileStream: TGStream;
  FileSize  : LongInt;
begin
  if StrIComp(FileName, '(Untitled)') = 0 then
    StrCopy(TmpFile, '*.org')
  else
    StrCopy(TmpFile, FileName);
  if Application^.ExecDialog(New(PFileDialog, Init(@Self,
                                 PChar(sd_FileSave), TmpFile)))
   = id_Ok
  then
  begin
    StrCopy(FileName, TmpFile);
    FileSize := SizeOf(TFileHeader)
              + StoreSize(OrgWindow^.Space)
              + StoreSize(OrgWindow^.Picture);
    FileStream.Init(FileName, stCreate, FileSize);
    WriteOGLHeader(FileStream, ff_Standard);
    FileStream.Put(OrgWindow^.Space);
    FileStream.Put(OrgWindow^.Picture);
    FileStream.Done;
    SetCaption;
    Saved := True;
  end;
end;

{ Printing related response methods }
procedure TMainWindow.PrinterConfig(var Msg: TMessage);
begin
  Printer^.Configure;
end;

procedure TMainWindow.ChoosePrinter(var Msg: TMessage);
begin
  Printer^.Choose;
end;

{ Do the actual printing }
procedure TMainWindow.Print(var Msg: TMessage);
var
  Page     : Integer;
  CopiesTxt: array [0..3] of Char;
  Copies   : Integer;
  NumCode  : Integer;
  Continue : Boolean;
begin
  Str(1, CopiesTxt);
  if Application^.ExecDialog(New(PInputDialog, Init(@Self, 'Print',
                  'Number of Copies?', CopiesTxt,
                   SizeOf(CopiesTxt))))
   = id_Ok
  then
  begin
    Val(CopiesTxt, Copies, NumCode);
    if NumCode <> 0 then Copies := 1;

    if Printer^.Start(FileName) > 0 then
    begin
      Page     := 1;
      Continue := True;
      while (Page <= Copies) and Continue do
      begin
        OrgWindow^.Print(Printer);
        if Page < Copies then  { Printer^.Finish does final page}
          Continue := (Printer^.NextFrame > 0);
        Inc(Page);
      end;
    end;
    Printer^.Finish;
  end;
end;

{ Display About box }
procedure TMainWindow.About(var Msg: TMessage);
var  Dlg: TDialog;
begin
  Dlg.Init(@Self, 'AboutDlg');
  Dlg.Execute;
  Dlg.Done;
end;

{ Display Help dialog }
procedure TMainWindow.Help(var Msg: TMessage);
var  Dlg: TDialog;
begin
  Dlg.Init(@Self, 'HelpDlg');
  Dlg.Execute;
  Dlg.Done;
end;

{ Exit the program if its ok }
procedure TMainWindow.Exit(var Msg: TMessage);
begin
  if CanClose then
     postQuitMessage(0);
end;

{ Make sure the user has saved his work before closing }
function TMainWindow.CanClose: Boolean;
var Reply : Integer;
    Msg : TMessage;
begin
  if not Saved then
  begin
    Reply := MessageBox(HWindow, 'File has not been saved. '+#13+
             'Save file before closing?','Warning',
             mb_IconStop or mb_YesNoCancel);
    if Reply = id_Yes then
      Save(Msg);
  end;
  CanClose := Saved or (Reply <> id_Cancel);
end;

{ Tool selection methods }
procedure TMainWindow.Color(var Msg: TMessage);
begin
  Application^.ExecDialog(New(PRGBDialog, Init(@Self,
    OrgWindow^.CurrPen^.Color)));
end;

{ Set the pen width }
procedure TMainWindow.Width(var Msg: TMessage);
var
  Dlg: PInputDialog;
  Txt: array[0..3] of Char;
  Size: Integer;
  NumCode: Integer;
begin
  Str(OrgWindow^.CurrPen^.Width, Txt);
  Dlg := New(PInputDialog, Init(@Self, Application^.Name,
    'Enter new pen width:', Txt, SizeOf(Txt)));
  if Application^.ExecDialog(Dlg) = id_Ok then
  begin
    Val(Txt, Size, NumCode);
    if NumCode = 0 then
      OrgWindow^.SetWidth(Size);
  end;
end;

{ Change the grid size }
procedure TMainWindow.ChangeGrid(var Msg: TMessage);
var
  Dlg: PInputDialog;
  Txt: array[0..3] of Char;
  Size: Integer;
  NumCode: Integer;
  GridSize : TGPoint;
begin
  Str(OrgWindow^.Space^.Grid^.x, Txt);
  Dlg := New(PInputDialog, Init(@Self, Application^.Name,
    'Enter new grid size (0 to disable):', Txt, SizeOf(Txt)));
  if Application^.ExecDialog(Dlg) = id_Ok then
  begin
    Val(Txt, Size, NumCode);
    if NumCode = 0 then
    begin
      GridSize.init(Size, Size);
      OrgWindow^.Space^.SetGrid(@GridSize);
      OrgWindow^.invalidate;
      GridSize.done;
    end;
  end;
end;

{ Trap all menu items related to Tool selection, Edit menu }
procedure TMainWindow.DefCommandProc(var Msg: TMessage);
var i : word;
begin
  { Tool selection }
  if (Msg.WParam >= cm_Select) and (Msg.WParam <= cm_Text) then
  begin
    { Add a check mark to selected menu item }
    for i := cm_Select to cm_Text do
      checkMenuItem(GetMenu(HWindow), i, mf_Unchecked);
    checkMenuItem(GetMenu(HWindow), Msg.WParam, mf_Checked);
    drawMenuBar(HWindow);
    Saved := False;
    OrgWindow^.setTool(Msg.wParam)
  end
  else             
  { Edit menu }
  if (Msg.WParam >= cm_Cut) and (Msg.WParam <= cm_Clear) then
  begin
    messageBeep(0);
    case Msg.WParam of 
      cm_EditCut  : OrgWindow^.CutChoice;
      cm_EditCopy : OrgWindow^.CopyChoice;
      cm_EditPaste: OrgWindow^.PasteChoice;
      cm_EditClear: OrgWindow^.ClearChoice;
    end;
  end
  else
  { Command handled through normal message routing to object } 
    TGWindow.DefCommandProc(Msg);
end;

{ View menu response methods }
procedure TMainWindow.ToggleGrid(var Msg: TMessage);
var  style : word;
begin
  ViewGrid := not ViewGrid;
  if ViewGrid then
    style := mf_Checked
  else
    style := mf_Unchecked;
  checkMenuItem(GetMenu(HWindow), cm_ViewGrid, style);
  drawMenuBar(HWindow);
  OrgWindow^.ViewGrid := ViewGrid;
  OrgWindow^.Invalidate;
end;

{ Show menu response methods }
procedure TMainWindow.Data(var Msg: TMessage);
begin
  OrgWindow^.WMLButtonDblClick(Msg);
end;

{ Make sure the drawing window always gets the focus }
procedure TMainWindow.WMSetFocus(var Msg: TMessage);
begin
  OrgWindow^.WMSetFocus(Msg);
end;

{ Application methods }
procedure TOrgApp.InitInstance;
begin
  TApplication.InitInstance;
  HAccTable := LoadAccelerators(HInstance, 'Keys');
end;

{ Start the application by creating main window }
procedure TOrgApp.InitMainWindow;
begin
  MainWindow := New(PMainWindow, Init(nil, Application^.Name));
end;

var
  App: TOrgApp;

begin
  App.Init('Org');
  App.Run;
  App.Done;
end.