/*****************************************************************************

	copy.c
	------

	Routines in this module demonstrate how the TVReturnData function is
	used, by copying the trace window contents to another TextView window

	This source is Copyright (c) Alan Phillips 1991. It may be freely used
	and adapted for non-commercial applications. Commercial and ShareWare
	authors should first obtain the written permission of the author.

	The source is edited with a tab size of 4.

*****************************************************************************/

#include	"stdhead.h"


/*****************************************************************************

	Local Data
	----------

*****************************************************************************/

static	HWND	hCopyWnd;				/* handle to copy window */
static	FARPROC	lpCopyMenuHandler;		/* instance of menu handler function */

/*****************************************************************************

	copy_trace_window
	-----------------

	This routine is called from the 'copy' option of the main window's
	'Trace' menu. It creates a second TextView window (or, if it already
	exists, empties it) and uses the TVReturnData function to copy the
	current contents of the trace window into it

	copy_trace_window(hWnd)

	HWND	hWnd;					Handle to main window

*****************************************************************************/

void	copy_trace_window(HWND hWnd)

{
	char	buffer[256];				/* buffer for reading lines from the
										*  trace window into
										*/
	FARPROC	lpNotifyProc;				/* ptr to notification proc */
	
	/* If we don't have a copy window at the moment, we'll create one using the
	*  same window class as the trace window. If we do have one already we
	*  empty it and make sure it's visible
	*/

	if ( hCopyWnd != NULL )
	{
		/* We already have a copy window, so we'll reset it and make sure
		*  that it's visible
		*/

		TVResetWindow(hCopyWnd);
		if ( IsIconic(hCopyWnd) )
			ShowWindow(hCopyWnd,SW_SHOWNORMAL);
		else
			BringWindowToTop(hCopyWnd);
	}
	else
	{
		/* We don't have a copy window, so we'll create one. Since we don't
		*  have an option in the application's menu to close this window,
		*  we need to give it a system menu and leave the 'Close' option
		*  active. This means that we have to have a callback function so that
		*  TextView can notify us when the window closes.
		*
		*  Note that we don't include the TVS_TIMESTAMP option for this
		*  window. The lines we read back from the trace window will include
		*  the timestamp value added by TextView when they were written.
		*
		*  We make the window's storage buffer 2 lines larger than the trace
		*  window's, so we can add a header and trailer line. If the trace
		*  window is at the maximum size, this request will be rounded down.
		*/

		lpCopyMenuHandler	= MakeProcInstance(copy_menu_handler,hInst);

		hCopyWnd	= TVCreateWindow("TRACE_WINDOW",
									 "Trace Copy",
									 CW_USEDEFAULT,
									 CW_USEDEFAULT,
									 300,
									 200,
									 hInst,
									 NULL,
									 TVS_VSCROLL | TVS_HSCROLL |
										TVS_SYSMENU | TVS_SCROLLMENU,
									 4,
									 0,
									 TRW_SIZE+2,
									 lpCopyMenuHandler);

		if ( hCopyWnd == NULL )
		{
			/* The handle we got back was NULL, so something has gone wrong and
			*  we don't have the window
			*/

			message("Cannot create copy window",NULL,MB_ICONSTOP);
			return;
		}
	}

	/* The window is created, so we can write a message to head it, writing
	*  it in blue
	*/

	TVSetTextColor(hCopyWnd,RGB(0,0,255));
	TVOutputText(hCopyWnd,"*** Copy of trace window ***",0);

	/* Then put the colour for the rest of the text to be black */

	TVSetTextColor(hCopyWnd,RGB(0,0,0));

	/* Since we may have to write a large number of lines to the copy window,
	*  we'll now mark it so that the screen isn't updated every time we write a
	*  line. That way we'll finish much faster, since screen output and
	*  scrolling accounts for most of the time taken
	*/

	TVSetRedraw(hCopyWnd,FALSE);

	/* And now we can do the real work of reading back the lines in the
	*  trace window. We first need to set up a procedure instance of the
	*  notify function that TextView will use to tell us every time it
	*  copies a line into our buffer
	*/

	lpNotifyProc	= MakeProcInstance(notify_handler,hInst);

	/* And then we ask TextView for the data. The work is then done inside
	*  the notification handler, and we won't return from TVReturnData until
	*  it's all done.
	*/

	TVReturnData(hTraceWnd,buffer,sizeof(buffer),lpNotifyProc);

	/* When we get to here the read out process has ended, so we can free off
	*  the procedure instance to the notification handler
	*/

	FreeProcInstance(lpNotifyProc);

	/* We can now allow the output window to be redrawn again. The change from
	*  redrawing off to redrawing on will make TextView repaint the window so
	*  we'll see that last screen-full of lines
	*/

	TVSetRedraw(hCopyWnd,TRUE);

	/* Write a message to the copy window to show that that's the end; we
	*  do this in blue
	*/

	TVSetTextColor(hCopyWnd,RGB(0,0,255));
	TVOutputText(hCopyWnd,"*** Copy Completed ***",0);

	/* Then put the copy window into manual scroll mode and scroll it back so
	*  that the first line is visible
	*/

	TVSetScrollState(hCopyWnd,TV_SCR_MANUAL);
	TVSetVPosition(hCopyWnd,TVSP_START);

}


/*****************************************************************************

	copy_menu_handler
	-----------------

	This is the menu handler callback function specified in the call to
	TVCreateWindow that creates the copy of the trace window. Here we're
	interested only in monitoring when the user closes the window so that
	we can clear out our copy of the handle and release resources.

	This routine must be exported in the appliation's module definition file.
	It must use the Pascal calling convention, and be declared FAR.

	copy_menu_handler(hWnd,nMenuItem)

	HWND	hWnd;					Handle to window
	WORD	nMenuItem;				Code indicating the menu item

*****************************************************************************/

void	FAR	PASCAL	copy_menu_handler(HWND hWnd,WORD nMenuItem)

{
	/* We look only for the system menu close option being clicked */

	if ( nMenuItem == TVMI_CLOSE )
	{
		/* The user has closed the window through the system menu. Lose our
		*  copy of the handle
		*/

		hCopyWnd	= NULL;

		/* And free the procedure instance of this function */

		FreeProcInstance(lpCopyMenuHandler);
	}

}


/*****************************************************************************

	notify_handler
	--------------

	This is the notification handler specified in the call to TVReturnData.
	TextView will call this routine every time it places the text of a line
	from the trace window into our buffer; we can then process the buffer as
	we wish. TextView will keep calling back into this routine until either
	all the text has been passed back, or we return a zero value.

	This routine must be exported in the appliation's module definition file.
	It must use the Pascal calling convention, and be declared FAR.

	reply = notify_handler(hWnd,lpBuffer,nCount,nTruncated)

	int		reply;			0 to stop the callback process
	HWND	hWnd;			Handle to the window whose data is being read
	LPSTR	lpBuffer;		Ptr to our buffer
	int		nCount;			Number of this line in the window (the first line
							is numbered zero)
	BOOL	nTruncated;		TRUE if text was truncated to fit our buffer

*****************************************************************************/

int		FAR	PASCAL	notify_handler(HWND hWnd,LPSTR lpBuffer,int nCount,
								   BOOL nTruncated)

{
	/* All we do here is to write the current contents of the buffer back out
	*  to the copy of the trace window. We don't care what's in it or whether
	*  it's been truncated
	*/

	TVOutputText(hCopyWnd,lpBuffer,lstrlen(lpBuffer));

	/* And return a non-zero value to continue with the next line */

	return(1);

}
