/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** PROJECT NAME: Abstract Data Types Library
**
** MODULE  NAME: HEAP.C 
**	  Implementation of Class HEAP which is responsible for malloc-style
**   sub-segment allocation routines.
**
** $Header:   E:/ADT.LIB/VCS/HEAP.C_V   1.2   27 May 1992 15:24:56   GenKiyooka  $
**
**   AUTHOR.: Gen Kiyooka 
**
**   NAME DESCRIBES CLASS - LOCATION:  The naming convention for functions
**     attempts to describe the location and the class of the function. 
**       ie. BOX_Display is a method of Class BOX (in BOX.C)
**
** $Log:   E:/ADT.LIB/VCS/LHEAP.C_V  $
 * 
 *    Rev 1.2   27 May 1992 15:24:56   GenKiyooka
 * Fixed bug again.
 * 
 *    Rev 1.1   27 May 1992 15:21:48   GenKiyooka
 * Removed PRECONDITION on HeapAlloc because it occurs during normal
 * program execution.
 * 
 *    Rev 1.0   22 May 1992 06:58:44   GenKiyooka
 * Initial revision.
*/
#include <WINDOWS.H>
#include <WINDOWSX.H>

#include "HEAP.H"

//
// Thanks to Bertrand Meyer for the improved ASSERTION 
//
#define PRECONDITION( expression, failValue ) 			\
if (!(expression))												\
	{																	\
	MessageBox( (HWND)NULL,										\
		(LPSTR)__LINE__,											\
		(LPSTR)"HEAP Precondition Failed",					\
		MB_OK );														\
	return (failValue);											\
	}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** NAME: HeapCreate 
**	  Create a new 'Local' heap and return the handle of this
**   new heap to the caller.
**
**   PARAMETERS:
**     IN: MinSize - the minimum size of the new heap
**
**   ATTRIBUTES: EXPORT QUALIFIED
**
**   RETURNS: HANDLE of new heap if successful
*/ 
HANDLE HeapCreate( UINT fdwHeap, UINT cbInitial, UINT cbMaximum )
{
LPSTR pNewHeap = GlobalAllocPtr (GPTR, cbInitial );
	PRECONDITION( pNewHeap, (HANDLE)NULL );
	{
HANDLE 	hNewHeap = GlobalPtrHandle( pNewHeap );
WORD  	hHeap		= (WORD)SELECTOROF(pNewHeap);
WORD  	wSize 	= (WORD)GlobalSize(hNewHeap) - 16;

	PRECONDITION( LocalInit (hHeap, 0, wSize), (HANDLE)NULL );

	GlobalUnlock (hNewHeap);  //  Undo LocalInit's GlobalLock.

	return (HANDLE)hHeap;

	// These two parameters are unused 
	cbMaximum;		// maximum heap size
	fdwHeap;			// flags determine heap behaviour under Win32
	}
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** NAME: HeapAlloc 
**	  Do a regular style malloc against the heap, and return
**   a long pointer to the result of the allocation.  Since
**   we're in protect mode and we can assume things about the
**   heap/memory block, we can do this.  Trust me.
**
**   PARAMETERS:
**     IN: hHeap - segment of the heap
**     IN: cbAlloc - size of the object to allocate in bytes
**
**   ATTRIBUTES: EXPORT QUALIFIED
**
**   RETURNS: pointer to new object or NULL
*/ 
LPSTR HeapAlloc( HANDLE hHeap, UINT cbAlloc )
	{
LPSTR lpMem = (LPSTR)NULL;
WORD  wSeg = (WORD) hHeap;
WORD  hLocal;

   _asm 	push    ds						// Save our DS value
   _asm 	mov     ax, wSeg				// swap in a new data segment
   _asm 	mov     ds, ax

   hLocal = (WORD)LocalAlloc ( LPTR, cbAlloc );

   _asm 	pop     ds						// Put the original DS back

	if (hLocal)	lpMem = (LPSTR) MAKELP( wSeg, hLocal );

	return lpMem;
   }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** NAME: HeapFree 
**	  Do a malloc style free on the memory object which was
**   previously allocated from the local heap given by hHeap.
**
**   PARAMETERS:
**     IN:	 	hHeap - selector (segment) of Local Heap
**     IN/OUT: lpMem - points to the object to free
**
**   ATTRIBUTES: EXPORT QUALIFIED
**
**   RETURNS: TRUE if successful
*/ 
BOOL  HeapFree( HANDLE hHeap, LPSTR lpMem )
	{
WORD  wSeg = (WORD) hHeap;
WORD  hLocal = OFFSETOF( lpMem );

	PRECONDITION( (wSeg==(WORD)SELECTOROF(lpMem)), FALSE );

	_asm 	push    ds
	_asm 	mov     ax, wSeg
	_asm 	mov     ds, ax

	hLocal = (WORD)LocalFree( (HLOCAL)hLocal );

	_asm 	pop     ds

	PRECONDITION( (!hLocal), FALSE );
	return TRUE;
	}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** NAME: HeapSize 
**	  Determine the size of an object allocated .
**
**   PARAMETERS:
**     IN: hHeap - selector or segment of Heap.
**
**   ATTRIBUTES: EXPORT QUALIFIED
**
**   RETURNS: TRUE if successful
*/ 
UINT HeapSize( HANDLE hHeap, LPSTR lpMem )
{
WORD  wSeg = (WORD) hHeap;
WORD  hLocal = OFFSETOF( lpMem );
UINT	cbSize = 0;

	PRECONDITION( (wSeg==(WORD)SELECTOROF(lpMem)), cbSize );

	_asm 	push    ds
	_asm 	mov     ax, wSeg
	_asm 	mov     ds, ax

	cbSize = (WORD)LocalSize( (HLOCAL)hLocal );

	_asm 	pop     ds

	return cbSize;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
** NAME: HeapDestroy 
**	  Destroy the Local Heap given by hHeap.
**
**   PARAMETERS:
**     IN: hHeap - selector or segment of Heap.
**
**   ATTRIBUTES: EXPORT QUALIFIED
**
**   RETURNS: TRUE if successful
*/ 
BOOL  HeapDestroy( HANDLE hHeap )
{
LPSTR phHeap = MAKELP( ((WORD)hHeap), 0);
	GlobalFreePtr( phHeap );
	return TRUE;
}
