/*
 * File:     wx_dc.h
 * Purpose:  wxDC device context declaration
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */


#ifndef wx_dch
#define wx_dch

#include <fstream.h>
#include "common.h"
#include "wx_frame.h"
#include "wx_gdi.h"

#ifdef wx_xview
#include "xview/scrollbar.h"
#endif

#define wxDEVICE_CANVAS  1  // X canvas
#define wxDEVICE_EPS     2  // Encapsulated PostScript on any platform
#define wxDEVICE_WINDOWS 3  // MS Windows device (canvas, printer)
#define wxDEVICE_PIXMAP  4  // X pixmap

#ifdef wx_x
#define MM_TEXT        1
#define MM_ISOTROPIC   2
#define MM_ANISOTROPIC 3
#define MM_LOMETRIC    4
#define MM_HIMETRIC    5
#define MM_TWIPS       6
#endif

#define MM_POINTS      7
#define MM_METRIC      8


class wxCanvas;
class wxDC: public wxObject
{
 public:
  int device;
  Bool ok;
  Bool clipping;
  Bool wx_interactive;

  // Coordinate system variables
  float logical_origin_x;
  float logical_origin_y;

  float device_origin_x;
  float device_origin_y;

  float logical_scale_x;
  float logical_scale_y;

  float user_scale_x;
  float user_scale_y;

  int mapping_mode;

  int yorigin;          // For EPS
  ofstream *pstream;    // PostScript output stream
  float min_x;          // EPS bounding box
  float min_y;
  float max_x;
  float max_y;
  char *title;
  char *filename;
#ifdef wx_x
  // Every time a callback happens, these are set to point to the right values
  // for drawing calls to work
  GC gc;
#ifdef wx_motif
  GC gcBacking;
#endif
  int background_pixel;
  wxColour *current_colour;
  wxBitmap *selected_pixmap;
#endif
#ifdef wx_msw
  int clip_x1;
  int clip_y1;
  int clip_x2;
  int clip_y2;
  HDC cdc;
  Bool dont_delete;
  int window_ext_x;
  int window_ext_y;

  // Store all old GDI objects when do a SelectObject,
  // so we can select them back in (this unselecting user's
  // objects) so we can safely delete the DC.
  HBITMAP old_bitmap;
  HPEN    old_pen;
  HBRUSH  old_brush;
  HFONT   old_font;
#endif
  wxCanvas *canvas;
  Bool Colour;

  int current_logical_function;

  wxPen *current_pen;
  wxBrush *current_brush;
  wxBrush *current_background_brush;
  wxColour *current_text_foreground;
  wxColour *current_text_background;
  wxFont *font;

  wxDC(void);

  // Create a compatible DC from an existing one
  wxDC(wxDC *old_dc);

  // Create a DC corresponding to a canvas
  wxDC(wxCanvas *canvas);

  // Create a printer DC
  wxDC(char *driver, char *device, char *output, Bool interactive = TRUE);

  ~wxDC(void);

  virtual void DrawLine(float x1, float y1, float x2, float y2);
  virtual void DrawPoint(float x, float y);
  virtual void DrawLines(int n, wxPoint points[], float xoffset = 0, float yoffset = 0);
  virtual void DrawLines(wxList *list, float xoffset = 0, float yoffset = 0);
  virtual void DrawPolygon(int n, wxPoint points[], float xoffset = 0, float yoffset = 0);
  virtual void DrawPolygon(wxList *list, float xoffset = 0, float yoffset = 0);
  virtual void DrawRectangle(float x, float y, float width, float height);
  virtual void DrawRoundedRectangle(float x, float y, float width, float height, float radius = 20);
  virtual void DrawEllipse(float x, float y, float width, float height);

  // Splines
  // 3-point spline
  virtual void DrawSpline(float x1, float y1, float x2, float y2, float x3, float y3);
  // Any number of control points - a list of pointers to wxPoints
  virtual void DrawSpline(wxList *points);

  virtual void DrawIcon(wxIcon *icon, float x, float y);

  virtual void Clear(void);
  virtual void SetFont(wxFont *font);
  virtual void SetPen(wxPen *pen);
  virtual void SetBrush(wxBrush *brush);
  virtual void SetLogicalFunction(int function);
  virtual void SetBackground(wxBrush *brush);
  virtual void SetTextForeground(wxColour *colour);
  virtual void SetTextBackground(wxColour *colour);
  virtual void SetClippingRegion(float x, float y, float width, float height);
  virtual void DestroyClippingRegion(void);
  virtual void DrawText(char *text, float x, float y);

  virtual int GetTextFamily(void);
  virtual int GetTextStyle(void);
  virtual int GetTextWeight(void);
  virtual float GetTextHeight(void);
  virtual float GetTextWidth(void);
  virtual void GetTextExtent(char *string, float *x, float *y);
  virtual Bool Ok(void);
  virtual Bool StartDoc(char *message);
  virtual void EndDoc(void);
  virtual void StartPage(void);
  virtual void EndPage(void);
  virtual void SetMapMode(int mode);
  virtual int  GetMapMode(void);
  virtual void SetLogicalOrigin(float x, float y);
  virtual void SetDeviceOrigin(float x, float y);
  virtual void SetLogicalScale(float x, float y);
  virtual void SetUserScale(float x, float y);
  virtual float DeviceToLogicalX(int x);
  virtual float DeviceToLogicalY(int y);
  virtual int LogicalToDeviceX(float x);
  virtual int LogicalToDeviceY(float y);

  // Only works for PostScript *after* you've printed an image.
  // Gives width and height of image.
  virtual void GetSize(float *width, float *height);

  void CalcBoundingBox(float x, float y);

#ifdef wx_msw
  void SetRop(HDC cdc);
  void DoClipping(HDC cdc);
  void SelectOldObjects(HDC dc);
#endif

  wxDC *CreateCompatibleDC(void);
  void SelectObject(wxBitmap *bitmap);
  Bool Blit(float xdest, float ydest, float width, float height,
            wxDC *source, float xsrc, float ysrc, int rop = wxCOPY);
};

/*
 * Metafile and metafile device context classes - work in Windows 3.1 only
 *
 */

class wxMetaFile: public wxObject
{
 public:
    
#ifdef wx_msw
  HANDLE metafile;
#endif
  wxMetaFile(void);
  ~wxMetaFile(void);

  // After this is called, the metafile cannot be used for anything
  // since it is now owned by the clipboard.
  Bool SetClipboard(int width = 0, int height = 0);
};

class wxMetaFileDC: public wxDC
{
 public:
#ifdef wx_msw
  wxMetaFile *metafile;
#endif

  wxMetaFileDC(char *file = NULL);
  ~wxMetaFileDC(void);

  // Should be called at end of drawing
  wxMetaFile *Close(void);
  void SetMapMode(int mode);
};

extern char *wx_printer_command;
extern char *wx_printer_flags;
extern Bool wx_preview;
extern char *wx_preview_command;
extern Bool wx_portrait;
extern float wx_printer_scale_x;
extern float wx_printer_scale_y;
extern float wx_printer_translate_x;
extern float wx_printer_translate_y;
extern Bool wx_print_to_file;

/*
 * XView
 * Ok, so all this isn't neat, but it about works.
 */
#ifdef wx_x

// Logical to device
// Absolute
#define XLOG2DEV(x) (int)(((x) - logical_origin_x)*logical_scale_x*user_scale_x + device_origin_x)
#define YLOG2DEV(y) (int)(((y) - logical_origin_y)*logical_scale_y*user_scale_y + device_origin_y)

// Logical to device without the device translation
#define XLOG2DEV_2(x) (int)(((x) - logical_origin_x)*logical_scale_x*user_scale_x)
#define YLOG2DEV_2(y) (int)(((y) - logical_origin_y)*logical_scale_y*user_scale_y)

// Relative
#define XLOG2DEVREL(x) (int)((x)*logical_scale_x*user_scale_x)
#define YLOG2DEVREL(y) (int)((y)*logical_scale_y*user_scale_y)

// Device to logical
// Absolute
#define XDEV2LOG(x) (int)(((x) - device_origin_x)/(logical_scale_x*user_scale_x) + logical_origin_x)

#define YDEV2LOG(y) (int)(((y) - device_origin_y)/(logical_scale_y*user_scale_y) + logical_origin_y)

// Relative
#define XDEV2LOGREL(x) (int)((x)/(logical_scale_x*user_scale_x))
#define YDEV2LOGREL(y) (int)((y)/(logical_scale_y*user_scale_y))

#endif

/*
 * MS Windows
 */
#ifdef wx_msw

// Logical to device
// Absolute
#define XLOG2DEV(x) (int)((x))

#define YLOG2DEV(y) (int)((y))

// Relative
#define XLOG2DEVREL(x) (int)((x))
#define YLOG2DEVREL(y) (int)((y))

// Device to logical
// Absolute
#define XDEV2LOG(x) (int)((x))

#define YDEV2LOG(y) (int)((y))

// Relative
#define XDEV2LOGREL(x) (int)((x))
#define YDEV2LOGREL(y) (int)((y))

/*
 * Have the same macros as for XView but not for every operation:
 * just for calculating window/viewport extent (a better way of scaling).
 */

// Logical to device
// Absolute
#define MS_XLOG2DEV(x) (int)(((x) - logical_origin_x)*logical_scale_x*user_scale_x + device_origin_x)

#define MS_YLOG2DEV(y) (int)(((y) - logical_origin_y)*logical_scale_y*user_scale_y + device_origin_y)

// Relative
#define MS_XLOG2DEVREL(x) (int)((x)*logical_scale_x*user_scale_x)
#define MS_YLOG2DEVREL(y) (int)((y)*logical_scale_y*user_scale_y)

// Device to logical
// Absolute
#define MS_XDEV2LOG(x) (int)(((x) - device_origin_x)/(logical_scale_x*user_scale_x) + logical_origin_x)

#define MS_YDEV2LOG(y) (int)(((y) - device_origin_y)/(logical_scale_y*user_scale_y) + logical_origin_y)

// Relative
#define MS_XDEV2LOGREL(x) (int)((x)/(logical_scale_x*user_scale_x))
#define MS_YDEV2LOGREL(y) (int)((y)/(logical_scale_y*user_scale_y))

#define VIEWPORT_EXTENT 1000

#endif

// Conversion
#define METRIC_CONVERSION_CONSTANT  0.0393700787

// Scaling factors for various unit conversions
#define mm2inches (METRIC_CONVERSION_CONSTANT)
#define inches2mm (1/METRIC_CONVERSION_CONSTANT)

#define mm2twips (METRIC_CONVERSION_CONSTANT*1440)
#define twips2mm (1/(METRIC_CONVERSION_CONSTANT*1440))

#define mm2pt (METRIC_CONVERSION_CONSTANT*72)
#define pt2mm (1/(METRIC_CONVERSION_CONSTANT*72))

class wxSpline: public wxObject
{
 public:
  int type;
  wxList *points;

  wxSpline(wxList *list);
  void DeletePoints(void);

  // Doesn't delete points
  ~wxSpline(void);
};

#endif
