/*
 * File:     wx_form.cc
 * Purpose:  Form handling code
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>
#include <stdio.h>
#include <stdarg.h>
#include "common.h"
#include "wx_utils.h"
#include "wx_dialg.h"
#include "wx_form.h"

// Helper function
wxFormItemConstraint *wxFormConstraintsContains(int type, wxList *constraints);

// Display items and callback functions
class wxFormButton: public wxButton
{
 public:
  wxForm *form;
  wxFormButton(wxPanel *panel, wxFunction func, char *label,
               int x = -1, int y = -1, int width = -1, int height = -1);
};

void wxFormOk(wxFormButton& button, wxEvent& event);
void wxFormRevert(wxFormButton& button, wxEvent& event);
void wxFormCancel(wxFormButton& button, wxEvent& event);
void wxFormUpdate(wxFormButton& button, wxEvent& event);

/*
 * Range
 *
 */

wxRealRange::wxRealRange(float the_lo, float the_hi)
{
  lo = the_lo;
  hi = the_hi;
}

/*
 * Form item constraint
 *
 */

wxFormItemConstraint::wxFormItemConstraint(int type)
{
  Type = type;
}

wxFormItemConstraint::~wxFormItemConstraint(void)
{
  switch (Type)
  {
    case wxFORM_CONSTRAINT_ONE_OF:
    {
      wxNode *node = Constraint.OneOf->First();
      while (node)
      {
        char *s = (char *)node->Data();
        wxNode *next = node->Next();
        delete s;
        delete node;
        node = next;
      }
      break;
    }
    default:
      break;
  }
}

/*
 * Form item
 *
 */

wxFormItem::wxFormItem(int type, int item_type)
{
  Id = 0;
  Type = type;
  ItemType = item_type;
  PanelItem = NULL;
  Constraints = NULL;
  CustomEditor = NULL;
  HelpString = NULL;
  Label = NULL;
  Height = -1;
  Width = -1;
}

wxFormItem::~wxFormItem(void)
{
  if (HelpString)
    delete HelpString;
  if (Label)
    delete Label;

  Form->FormItems.DeleteObject(this);

  if (Constraints)
  {
    wxNode *node = Constraints->First();
    while (node)
    {
      wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
      delete constraint;
      node = node->Next();
    }
    delete Constraints;
  }
}

void wxFormItem::MakePanelItem(wxPanel *panel)
{
  switch (Type)
  {
    case wxFORM_STRING:
    {
      switch (ItemType)
      {
        case wxFORM_SINGLE_LIST:
        {
          PanelItem = new wxListBox(panel, (wxFunction)NULL, Label, wxSINGLE,
                                    -1, -1, Width, Height);
          wxListBox *listbox = (wxListBox *)PanelItem;
          wxFormItemConstraint *constraint =
            wxFormConstraintsContains(wxFORM_CONSTRAINT_ONE_OF, Constraints);
          if (constraint)
          {
            wxNode *node = constraint->Constraint.OneOf->First();
            while (node)
            {
              char *s = (char *)node->Data();
              listbox->Append(s);
              node = node->Next();
            }
          }
          break;
        }
        case wxFORM_CHOICE:
        {
          PanelItem = new wxChoice(panel, (wxFunction)NULL, Label,
                                    -1, -1, Width, Height);
          wxChoice *choice = (wxChoice *)PanelItem;
          wxFormItemConstraint *constraint =
            wxFormConstraintsContains(wxFORM_CONSTRAINT_ONE_OF, Constraints);
          if (constraint)
          {
            wxNode *node = constraint->Constraint.OneOf->First();
            while (node)
            {
              char *s = (char *)node->Data();
              choice->Append(s);
              node = node->Next();
            }
          }
          break;
        }
        case wxFORM_TEXT:
        {
          PanelItem = new wxText(panel, (wxFunction)NULL, Label, "", -1, -1,
                                 Width, Height);
          break;
        }
        default:
          break;
      }
      break;
    }
    case wxFORM_DUMB_MESSAGE:
    {
      PanelItem = new wxMessage(panel, Label);
      break;
    }
    case wxFORM_BUTTON:
    {
      PanelItem = new wxButton(panel, ButtonFunc, Label, -1, -1, Width, Height);
      break;
    }
    case wxFORM_NEWLINE:
    {
      panel->NewLine();
      break;
    }
    case wxFORM_SHORT:
    case wxFORM_LONG:
    {
      switch (ItemType)
      {
        case wxFORM_SLIDER:
        {
          int lo = -100;
          int hi = 100;
          int width = 230;
          wxFormItemConstraint *constraint =
            wxFormConstraintsContains(wxFORM_CONSTRAINT_RANGE, Constraints);
          if (constraint)
          {
            wxRealRange *range = constraint->Constraint.Range;
            lo = (int)range->lo;
            hi = (int)range->hi;
          }
          int default_value;
          if (Type == wxFORM_SHORT)
            default_value = *(ValuePtr.ShortIntValuePtr);
          else
            default_value = (int)*(ValuePtr.LongIntValuePtr);

          PanelItem = new wxSlider(panel, (wxFunction)NULL, Label,
                                   default_value, lo, hi, width);
          break;
        }
        case wxFORM_TEXT:
        default:
        {
          PanelItem = new wxText(panel, (wxFunction)NULL, Label, "", -1, -1,
                                 Width, Height);
          break;
        }
      }
      break;
    }

    case wxFORM_BOOL:
    {
      switch (ItemType)
      {
        case wxFORM_CHECKBOX:
        default:
        {
          PanelItem = new wxCheckBox(panel, (wxFunction)NULL, Label, -1, -1,
                                 Width, Height);
          break;
        }
      }
      break;
    }
    case wxFORM_FLOAT:
    {
      switch (ItemType)
      {
        case wxFORM_TEXT:
        default:
        {
          PanelItem = new wxText(panel, (wxFunction)NULL, Label, "", -1, -1,
                                 Width, Height);
          break;
        }
      }
      break;
    }
    case wxFORM_LIST_OF_STRINGS:
    {
      break;
    }
    case wxFORM_LIST_OF_LONGS:
    {
      break;
    }

    default:
      break;
  }
}

Bool wxFormItem::CheckLongValue(long val)
{
  char msg_buf[300];
  if (Constraints)
  {
    wxNode *node = Constraints->First();
    Bool ok = TRUE;
    while (node && ok)
    {
      wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
      switch (constraint->Type)
      {
        case wxFORM_CONSTRAINT_RANGE:
        {
          wxRealRange *range = constraint->Constraint.Range;
          if ((range->lo > (float)val) || (range->hi < (float)val))
          {
            sprintf(msg_buf, "%s should be in range %.0f to %.0f",
                    Label, range->lo, range->hi);
            ok = FALSE;
          }
          break;
        }
        case wxFORM_CONSTRAINT_FUNCTION:
        {
          ok = (*(constraint->Constraint.ConstraintFunc))(Type, (char *)&val, Label, msg_buf);
          break;
        }
        default:
          break;
      }
      if (!ok)
      {
        (void)wxMessageBox(msg_buf, "Constraint violation", wxOK);
      }
      node = node->Next();
    }
    return ok;
  }
  else return TRUE;
}

Bool wxFormItem::CheckBoolValue(Bool val)
{
  char msg_buf[300];
  if (Constraints)
  {
    wxNode *node = Constraints->First();
    Bool ok = TRUE;
    while (node && ok)
    {
      wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
      switch (constraint->Type)
      {
        case wxFORM_CONSTRAINT_FUNCTION:
        {
          ok = (*(constraint->Constraint.ConstraintFunc))(Type, (char *)&val, Label, msg_buf);
          break;
        }
        default:
          break;
      }
      if (!ok)
      {
        (void)wxMessageBox(msg_buf, "Constraint violation", wxOK);
      }
      node = node->Next();
    }
    return ok;
  }
  else return TRUE;
}

Bool wxFormItem::CheckStringValue(char *val)
{
  char msg_buf[300];
  if (Constraints)
  {
    wxNode *node = Constraints->First();
    Bool ok = TRUE;
    while (node && ok)
    {
      wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
      switch (constraint->Type)
      {
        case wxFORM_CONSTRAINT_FUNCTION:
        {
          ok = (*(constraint->Constraint.ConstraintFunc))(Type, val, Label, msg_buf);
          break;
        }
        default:
          break;
      }
      if (!ok)
      {
        (void)wxMessageBox(msg_buf, "Constraint violation", wxOK);
      }
      node = node->Next();
    }
    return ok;
  }
  else return TRUE;
}

Bool wxFormItem::CheckFloatValue(float val)
{
  char msg_buf[300];
  if (Constraints)
  {
    wxNode *node = Constraints->First();
    Bool ok = TRUE;
    while (node && ok)
    {
      wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
      switch (constraint->Type)
      {
        case wxFORM_CONSTRAINT_RANGE:
        {
          wxRealRange *range = constraint->Constraint.Range;
          if ((range->lo > val) || (range->hi < val))
          {
            sprintf(msg_buf, "%s should be in range %.2f to %.2f",
                    Label, range->lo, range->hi);
            ok = FALSE;
          }
          break;
        }
        case wxFORM_CONSTRAINT_FUNCTION:
        {
          ok = (*(constraint->Constraint.ConstraintFunc))(Type, (char *)&val, Label, msg_buf);
          break;
        }
        default:
          break;
      }
      if (!ok)
      {
        (void)wxMessageBox(msg_buf, "Constraint error", wxOK);
      }
      node = node->Next();
    }
    return ok;
  }
  else return TRUE;
}

void wxFormItem::RevertValue(void)
{
  switch (Type)
  {
    case wxFORM_SHORT:
    {
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = IntToString(*(ValuePtr.ShortIntValuePtr));
          if (svalue)
          {
            text_item->SetValue(svalue);
            delete svalue;
          }
          break;
        }
        case wxFORM_SLIDER:
        {
          wxSlider *slider_item = (wxSlider *)PanelItem;
          slider_item->SetValue(*(ValuePtr.ShortIntValuePtr));
          break;
        }
        default:
          break;
      }
      break;
    }
    case wxFORM_LONG:
    {
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = LongToString(*(ValuePtr.LongIntValuePtr));
          if (svalue)
          {
            text_item->SetValue(svalue);
            delete svalue;
          }
          break;
        }
        case wxFORM_SLIDER:
        {
          wxSlider *slider_item = (wxSlider *)PanelItem;
          slider_item->SetValue((int)*(ValuePtr.LongIntValuePtr));
          break;
        }
        default:
          break;
      }
      break;
    }
    case wxFORM_FLOAT:
    {
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = FloatToString(*(ValuePtr.FloatValuePtr));
          if (svalue)
          {
            text_item->SetValue(svalue);
            delete svalue;
          }
          break;
        }
        default:
          break;
      }
      break;
    }
    case wxFORM_BOOL:
    {
      switch (ItemType)
      {
        case wxFORM_CHECKBOX:
        {
          wxCheckBox *check_item = (wxCheckBox *)PanelItem;
          check_item->SetValue(*(ValuePtr.BoolValuePtr));
          break;
        }
        default:
          break;
      }
      break;
    }
    case wxFORM_STRING:
    {
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = *(ValuePtr.StringValuePtr);
          if (svalue)
          {
            text_item->SetValue(svalue);
          }
          break;
        }
        case wxFORM_SINGLE_LIST:
        {
          wxListBox *list_item = (wxListBox *)PanelItem;
          char *svalue = *(ValuePtr.StringValuePtr);
          if (svalue)
          {
            int pos = list_item->FindString(svalue);
            if (pos > -1)
              list_item->SetSelection(pos);
            else
              list_item->SetSelection(0);
          }
          else
            list_item->SetSelection(0);
          break;
        }
        case wxFORM_CHOICE:
        {
          wxChoice *choice_item = (wxChoice *)PanelItem;
          char *svalue = *(ValuePtr.StringValuePtr);
          if (svalue)
          {
            int pos = choice_item->FindString(svalue);
            if (pos > -1)
              choice_item->SetSelection(pos);
            else
              choice_item->SetSelection(0);
          }
          else
            choice_item->SetSelection(0);
          break;
        }
        default:
          break;
      }
      break;
    }
    default:
      break;
  }
}

Bool wxFormItem::UpdateValue(void)
{
  switch (Type)
  {
    case wxFORM_SHORT:
    {
      int short_value = 0;
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = text_item->GetValue();
          StringToInt(svalue, &short_value);
          break;
        }
        case wxFORM_SLIDER:
        {
          wxSlider *slider_item = (wxSlider *)PanelItem;
          short_value = slider_item->GetValue();
          break;
        }
        default:
          break;
      }
      if (CheckLongValue((long)short_value))
        *(ValuePtr.ShortIntValuePtr) = short_value;
      else
        return FALSE;
      break;
    }
    case wxFORM_LONG:
    {
      long long_value;
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = text_item->GetValue();
          StringToLong(svalue, &long_value);
          break;
        }
        case wxFORM_SLIDER:
        {
          wxSlider *slider_item = (wxSlider *)PanelItem;
          long_value = slider_item->GetValue();
          break;
        }
        default:
          break;
      }
      if (CheckLongValue(long_value))
        *(ValuePtr.LongIntValuePtr) = long_value;
      else
        return FALSE;
      break;
    }
    case wxFORM_FLOAT:
    {
      float float_value;
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          char *svalue = text_item->GetValue();
          StringToFloat(svalue, &float_value);
          break;
        }
        default:
          break;
      }
      if (CheckFloatValue(float_value))
        *(ValuePtr.FloatValuePtr) = float_value;
      else
        return FALSE;
      break;
    }
    case wxFORM_BOOL:
    {
      Bool bool_value;
      switch (ItemType)
      {
        case wxFORM_CHECKBOX:
        {
          wxCheckBox *check_item = (wxCheckBox *)PanelItem;
          bool_value = check_item->GetValue();
          break;
        }
        default:
          break;
      }
      if (CheckBoolValue(bool_value))
        *(ValuePtr.BoolValuePtr) = bool_value;
      else
        return FALSE;
      break;
    }
    case wxFORM_STRING:
    {
      char *string_value = NULL;
      switch (ItemType)
      {
        case wxFORM_TEXT:
        {
          wxText *text_item = (wxText *)PanelItem;
          string_value = text_item->GetValue();
          break;
        }
        case wxFORM_SINGLE_LIST:
        {
          wxListBox *list_item = (wxListBox *)PanelItem;
          string_value = list_item->GetStringSelection();
          break;
        }
        case wxFORM_CHOICE:
        {
          wxChoice *choice_item = (wxChoice *)PanelItem;
          string_value = choice_item->GetStringSelection();
          break;
        }
        default:
          break;
      }
      if (CheckStringValue(string_value))
      {
        if (*ValuePtr.StringValuePtr)
          delete (*ValuePtr.StringValuePtr);
        *(ValuePtr.StringValuePtr) = copystring(string_value);
      }
      else return FALSE;
      break;
    }
    default:
      break;
  }
  return TRUE;
}

/*
 * Form
 *
 */

wxForm::wxForm(void)
{
  wx_form_panel = NULL;
  wx_editable = TRUE;
}

wxForm::~wxForm(void)
{
  wxNode *node = FormItems.First();
  while (node)
  {
    wxFormItem *item = (wxFormItem *)node->Data();
    wxNode *next = node->Next();
    delete item;
    node = next;
  }
}

void wxForm::Add(wxFormItem *item, long id)
{
  if (id < 0)
    id = NewId();
  item->Id = id;
  item->Form = this;
  FormItems.Append(item);
}

wxNode *wxForm::FindItem(long id)
{
  wxNode *node = FormItems.First();
  wxNode *found = NULL;
  while (node && !found)
  {
    wxFormItem *item = (wxFormItem *)node->Data();
    if (item->Id == id)
      found = node;
    else node = node->Next();
  }
  return found;
}


Bool wxForm::Set(long id, wxFormItem *item)
{
  wxNode *found = FindItem(id);
  if (found)
  {
    wxNode *next = found->Next();
    wxFormItem *old_item = (wxFormItem *)found->Data();
    delete old_item;
    FormItems.Insert(next, item);
    return TRUE;
  }
  else return FALSE;
}

Bool wxForm::Delete(long id)
{
  wxNode *found = FindItem(id);
  if (found)
  {
    wxFormItem *old_item = (wxFormItem *)found->Data();
    delete old_item;
    return TRUE;
  }
  else return FALSE;
}

void wxForm::AssociatePanel(wxPanel *panel)
{
  wx_form_panel = panel;

  wxFormButton *ok_button = new wxFormButton(panel, (wxFunction)wxFormOk, "Ok");
  ok_button->form = this;

  if (wx_editable)
  {
    wxFormButton *update_button = new wxFormButton(panel, (wxFunction)wxFormUpdate, "Update");
    update_button->form = this;
  }

  wxFormButton *revert_button = new wxFormButton(panel, (wxFunction)wxFormRevert, "Revert");
  revert_button->form = this;

  if (wx_editable)
  {
    wxFormButton *cancel_button = new wxFormButton(panel, (wxFunction)wxFormCancel, "Cancel");
    cancel_button->form = this;
  }

  panel->NewLine();

  wxNode *node = FormItems.First();
  while (node)
  {
    wxFormItem *item = (wxFormItem *)node->Data();
    item->MakePanelItem(panel);
    item->RevertValue();
    node = node->Next();
  }

  panel->Fit();
}

void wxForm::DisassociatePanel(void)
{
}

Bool wxForm::UpdateValues(void)
{
  wxNode *node = FormItems.First();
  Bool success = TRUE;
  while (node && success)
  {
    wxFormItem *item = (wxFormItem *)node->Data();
    if (!(item->UpdateValue()))
      success = FALSE;
    node = node->Next();
  }
  return success;
}

void wxForm::RevertValues(void)
{
  wxNode *node = FormItems.First();
  while (node)
  {
    wxFormItem *item = (wxFormItem *)node->Data();
    item->RevertValue();
    node = node->Next();
  }
}

void wxForm::OnOk(void)
{
  if (wx_form_panel)
  {
    wx_form_panel->Show(FALSE);
    delete wx_form_panel;
  }
  delete this;
}

void wxForm::OnCancel(void)
{
  if (wx_form_panel)
  {
    wx_form_panel->Show(FALSE);
    delete wx_form_panel;
  }
  delete this;
}

void wxForm::OnRevert(void)
{
}

void wxForm::OnUpdate(void)
{
}

/*
 * Functions for creating items
 *
 */

wxFormItem *wxMakeFormString(char *label, char **var,
  int item_type, wxList *constraints,
  char *help_string, wxEditFunction editor, int width, int height)
{
  if (item_type == wxFORM_DEFAULT)
  {
    // Must decide what type of item to use.
    // If constraint list contains a OneOf, use a single-choice list box.
    // Else use a simple Text item
    if (wxFormConstraintsContains(wxFORM_CONSTRAINT_ONE_OF, constraints))
      item_type = wxFORM_SINGLE_LIST;
    else
      item_type = wxFORM_TEXT;
  }
  wxFormItem *item = new wxFormItem(wxFORM_STRING, item_type);
  item->Constraints = constraints;
  item->CustomEditor = editor;
  if (help_string)
    item->HelpString = copystring(help_string);
  item->ValuePtr.StringValuePtr = var;
  if (label)
    item->Label = copystring(label);
  item->Value.StringValue = NULL;
  item->Height = height;
  item->Width = width;
  return item;
}

wxFormItem *wxMakeFormMessage(char *label)
{
  wxFormItem *item = new wxFormItem(wxFORM_DUMB_MESSAGE, wxFORM_DUMB_MESSAGE);
  if (label)
    item->Label = copystring(label);
  return item;
}

wxFormItem *wxMakeFormButton(char *label, wxFunction func)
{
  wxFormItem *item = new wxFormItem(wxFORM_BUTTON, wxFORM_BUTTON);
  if (label)
    item->Label = copystring(label);
  item->ButtonFunc = func;
  return item;
}

wxFormItem *wxMakeFormNewLine(void)
{
  return new wxFormItem(wxFORM_NEWLINE, wxFORM_NEWLINE);
}

wxFormItem *wxMakeFormLong(char *label, long *var,
  int item_type, wxList *constraints,
  char *help_string, wxEditFunction editor,
  int width, int height)
{
  if (item_type == wxFORM_DEFAULT)
  {
    if (wxFormConstraintsContains(wxFORM_CONSTRAINT_RANGE, constraints))
      item_type = wxFORM_SLIDER;
    else item_type = wxFORM_TEXT;
  }
  wxFormItem *item = new wxFormItem(wxFORM_LONG, item_type);
  item->Constraints = constraints;
  item->CustomEditor = editor;
  if (help_string)
    item->HelpString = copystring(help_string);
  item->ValuePtr.LongIntValuePtr = var;
  if (label)
    item->Label = copystring(label);
  item->Value.LongIntValue = 0;
  item->Height = height;
  item->Width = width;
  return item;
}

wxFormItem *wxMakeFormShort(char *label, int *var,
  int item_type, wxList *constraints,
  char *help_string, wxEditFunction editor,
  int width, int height)
{
  if (item_type == wxFORM_DEFAULT)
  {
    if (wxFormConstraintsContains(wxFORM_CONSTRAINT_RANGE, constraints))
      item_type = wxFORM_SLIDER;
    else item_type = wxFORM_TEXT;
  }
  wxFormItem *item = new wxFormItem(wxFORM_SHORT, item_type);
  item->Constraints = constraints;
  item->CustomEditor = editor;
  if (help_string)
    item->HelpString = copystring(help_string);
  item->ValuePtr.ShortIntValuePtr = var;
  if (label)
    item->Label = copystring(label);
  item->Value.ShortIntValue = 0;
  item->Height = height;
  item->Width = width;
  return item;
}

wxFormItem *wxMakeFormFloat(char *label, float *var,
  int item_type, wxList *constraints,
  char *help_string, wxEditFunction editor,
  int width, int height)
{
  if (item_type == wxFORM_DEFAULT)
  {
    item_type = wxFORM_TEXT;
  }
  wxFormItem *item = new wxFormItem(wxFORM_FLOAT, item_type);
  item->Constraints = constraints;
  item->CustomEditor = editor;
  if (help_string)
    item->HelpString = copystring(help_string);
  item->ValuePtr.FloatValuePtr = var;
  if (label)
    item->Label = copystring(label);
  item->Value.FloatValue = 0.0;
  item->Height = height;
  item->Width = width;
  return item;
}

wxFormItem *wxMakeFormBool(char *label, Bool *var,
  int item_type, wxList *constraints,
  char *help_string, wxEditFunction editor,
  int width, int height)
{
  if (item_type == wxFORM_DEFAULT)
  {
    item_type = wxFORM_CHECKBOX;
  }
  wxFormItem *item = new wxFormItem(wxFORM_BOOL, item_type);
  item->Constraints = constraints;
  item->CustomEditor = editor;
  if (help_string)
    item->HelpString = copystring(help_string);
  item->ValuePtr.BoolValuePtr = var;
  if (label)
    item->Label = copystring(label);
  item->Value.BoolValue = 0;
  item->Height = height;
  item->Width = width;
  return item;
}

/*
 * Functions for creating constraints
 *
 */

wxFormItemConstraint *wxMakeConstraintStrings(wxList *list)
{
  wxFormItemConstraint *constraint =
    new wxFormItemConstraint(wxFORM_CONSTRAINT_ONE_OF);
  constraint->Constraint.OneOf = list;
  return constraint;
}

wxFormItemConstraint *wxMakeConstraintStrings(char *first ...)
{
  wxList *list = new wxList;
  va_list ap;

  va_start(ap, first);
  list->Append((wxObject *)copystring(first));

  for (;;)
  {
    char *s = va_arg(ap, char *);
    if (((int)s) == 0)
      break;
    else list->Append((wxObject *)copystring(s));
  }
  va_end(ap);

  wxFormItemConstraint *constraint =
    new wxFormItemConstraint(wxFORM_CONSTRAINT_ONE_OF);
  constraint->Constraint.OneOf = list;
  return constraint;
}

wxFormItemConstraint *wxMakeConstraintRange(float lo, float hi)
{
  wxFormItemConstraint *constraint =
    new wxFormItemConstraint(wxFORM_CONSTRAINT_RANGE);
  constraint->Constraint.Range = new wxRealRange(lo, hi);
  return constraint;
}

wxFormItemConstraint *wxMakeConstraintFunction(wxConstraintFunction func)
{
  wxFormItemConstraint *constraint =
    new wxFormItemConstraint(wxFORM_CONSTRAINT_FUNCTION);
  constraint->Constraint.ConstraintFunc = func;
  return constraint;
}

/*
 * Helper functions
 *
 */

wxFormItemConstraint *wxFormConstraintsContains(int type, wxList *constraints)
{
  if (!constraints)
    return NULL;

  wxFormItemConstraint *found = NULL;
  wxNode *node = constraints->First();
  while (node && !found)
  {
    wxFormItemConstraint *constraint = (wxFormItemConstraint *)node->Data();
    if (constraint->Type == type)
      found = constraint;
    else node = node->Next();
  }
  return found;
}

/*
 * Form display items and helper functions
 *
 */

wxFormButton::wxFormButton(wxPanel *panel, wxFunction func, char *label,
             int x, int y, int width, int height):
  wxButton(panel, func, label, x, y, width, height)
{
}

void wxFormOk(wxFormButton& button, wxEvent& event)
{
  Bool success = TRUE;

  if (button.form->wx_editable)
    success = button.form->UpdateValues();

  if (success)
    button.form->OnOk();
}

void wxFormUpdate(wxFormButton& button, wxEvent& event)
{
  Bool success = button.form->UpdateValues();
  if (success)
    button.form->OnUpdate();
}

void wxFormRevert(wxFormButton& button, wxEvent& event)
{
  button.form->RevertValues();
  button.form->OnRevert();
}

void wxFormCancel(wxFormButton& button, wxEvent& event)
{
  button.form->OnCancel();
}
