/*
 * File:     wx_gdi.cc
 * Purpose:  GDI (Graphics Device Interface) objects and functions
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>
#include "common.h"
#include "wx_list.h"
#include "wx_main.h"
#include "wx_gdi.h"

#ifdef wx_x
XFontPool   *wxFontPool = NULL;
#include <X11/cursorfont.h> // /aiai/sun4/include/X11/cursorfont.h
#endif

#ifdef wx_motif
#include "wx_frame.h"
#include "wx_utils.h"
#endif

#ifdef wx_xview
#include <xview/screen.h>
#include <xview/cursor.h>
#include <xview/svrimage.h>
extern Xv_Server xview_server;

/* These cursors courtesy of xfig
 */

static short    bull_cursor_array[16] = {
    0x0F00, 0x30C0, 0x4020, 0x4020, 0x8010, 0x8610, 0x8610, 0x8010,
    0x4020, 0x4020, 0x30C0, 0x0F00, 0x0000, 0x0000, 0x0000, 0x0000
};

static short    char_cursor_data[16] = {
    0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00,
    0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00, 0xFF00,
};

static short    crosshair_cursor_data[16] = {
    0x1000, 0x1000, 0x1000, 0xFE00, 0x1000, 0x1000, 0x1000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static short    magnifier_cursor_array[16] = {
    0x0F80, 0x3060, 0x4010, 0x4010, 0x8008, 0x8008, 0x8008, 0x8008,
    0x8008, 0x4010, 0x4010, 0x3078, 0x0F9C, 0x000E, 0x0007, 0x0003,
};

static short    pencil_cursor_array[16] = {
    0x0000, 0x0018, 0x0024, 0x0075, 0x009B, 0x0117, 0x022E, 0x045C,
    0x08B8, 0x1170, 0x22E0, 0x25C0, 0x7B80, 0x6700, 0x8600, 0x0800,
};

static short    vbar_cursor_array[16] = {
    0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
    0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
};

static short hand_cursor_array[] =
{
  0x0C00,0x1200,0x1200,0x1380,0x1240,0x7270,0x9248,0x924E,
  0x9249,0x9249,0x9009,0x8001,0x4002,0x4002,0x2004,0x2004
};

#endif

wxBrushList *wxTheBrushList = NULL;
wxPenList   *wxThePenList = NULL;
wxGDIList   *wxTheIconList = NULL;
wxGDIList   *wxTheFontList = NULL;
wxGDIList   *wxTheBitmapList = NULL;

wxColourDatabase *wxTheColourDatabase = NULL;

// Stock objects
wxFont *wxNORMAL_FONT;
wxFont *wxSMALL_FONT;
wxFont *wxITALIC_FONT;
wxFont *wxSWISS_FONT;
wxPen *wxRED_PEN;
wxPen *wxCYAN_PEN;
wxPen *wxGREEN_PEN;
wxPen *wxBLACK_PEN;
wxPen *wxTRANSPARENT_PEN;
wxPen *wxBLACK_DASHED_PEN;

wxBrush *wxBLUE_BRUSH;
wxBrush *wxGREEN_BRUSH;
wxBrush *wxWHITE_BRUSH;
wxBrush *wxBLACK_BRUSH;
wxBrush *wxTRANSPARENT_BRUSH;
wxBrush *wxCYAN_BRUSH;
wxBrush *wxRED_BRUSH;
wxBrush *wxGREY_BRUSH;
wxBrush *wxMEDIUM_GREY_BRUSH;
wxBrush *wxLIGHT_GREY_BRUSH;

wxColour *wxBLACK;
wxColour *wxWHITE;
wxColour *wxRED;
wxColour *wxBLUE;
wxColour *wxGREEN;
wxColour *wxCYAN;
wxColour *wxLIGHT_GREY;

wxCursor *wxSTANDARD_CURSOR = NULL;
wxCursor *wxHOURGLASS_CURSOR = NULL;
wxCursor *wxCROSS_CURSOR = NULL;

wxFont::wxFont(void)
{
  point_size = 0;
  temporary = TRUE;
#ifdef wx_motif
#endif
#ifdef wx_xview
  x_font = NULL;
#endif
#ifdef wx_msw
  cfont = NULL;
#endif
}

/* Constructor for a font. Note that the real construction is done
 * in wxDC::SetFont, when information is available about scaling etc.
 */
wxFont::wxFont(int PointSize, int Family, int Style, int Weight)
{
  family = Family;
  style = Style;
  weight = Weight;
  point_size = PointSize;
#ifdef wx_motif
  xFont = NULL;
#endif
#ifdef wx_xview
  temporary = TRUE;
  x_font = NULL;
#endif
#ifdef wx_msw
  temporary = FALSE;
  cfont = NULL;
#endif
  wxTheFontList->Append(this);
}

int wxFont::GetPointSize(void)
{
  return point_size;
}

wxFont::~wxFont()
{
#ifdef wx_motif
#endif
#ifdef wx_msw
  if (cfont)
    DeleteObject(cfont);
#endif
  wxTheFontList->DeleteObject(this);
}

#ifdef wx_x
// Pseudo-scaleable Font management under XView - a nightmare!
// - but it works o.k.
XFontInfo::XFontInfo(int the_family, int the_style, int the_weight, int the_point_size,
#ifdef wx_xview
                     Xv_Font the_font)
#else
                     XFontStruct *the_font)
#endif
{
  family = the_family;
  style = the_style;
  weight = the_weight;
  point_size = the_point_size;
  font = the_font;
}

XFontPool::XFontPool(void)
{
  cache = NULL;
}

void XFontPool::AddFont(int family, int style, int weight, int point_size,
#ifdef wx_xview
                        Xv_Font font)
#else
                        XFontStruct *font)
#endif
{
  XFontInfo *info = new XFontInfo(family, style, weight, point_size, font);
  Append(info);
}

#ifdef wx_xview
Xv_Font
#else
XFontStruct *
#endif
  XFontPool::FindFont(int family, int style, int weight, int point_size)
{
  if (cache && cache->family == family && cache->style == style && cache->weight == weight
        && cache->point_size == point_size)
    return cache->font;

  wxNode *node = First();
#ifdef wx_xview
  Xv_Font
#else
  XFontStruct *
#endif
    found = NULL;
  while (node && !found)
  {
    XFontInfo *info = (XFontInfo *)node->Data();
    if (info->family == family && info->style == style && info->weight == weight
        && info->point_size == point_size)
    {
      found = info->font;
      cache = info;
    }
    node = node->Next();
  }
  return found;
}

#ifdef wx_xview
Xv_Font XFontPool::FindOrCreateFont(int family, int style, int weight, int point_size, int point_size_to_store, int xres, int yres)
{
  Xv_Font font = FindFont(family, style, weight, point_size);

  if (!font)
  {
    char *xfamily;
    char *xstyle;
    switch (family)
    {
    case wxDECORATIVE: xfamily = FONT_FAMILY_LUCIDA;  // ????
                       break;
    case wxROMAN:      xfamily = FONT_FAMILY_ROMAN;
                       break;
    case wxMODERN:     xfamily = FONT_FAMILY_COUR;
                       break;
    case wxSWISS:      xfamily = FONT_FAMILY_LUCIDA;
                       break;
    case wxDEFAULT:
    default:           xfamily = FONT_FAMILY_DEFAULT;
    }

    if (style == wxNORMAL && weight == wxNORMAL)
      xstyle = FONT_STYLE_NORMAL;
    else if (style == wxNORMAL && weight == wxBOLD)
      xstyle = FONT_STYLE_BOLD;
    else if (style == wxNORMAL && weight == wxLIGHT)
      xstyle = FONT_STYLE_NORMAL;

    else if (style == wxITALIC && weight == wxNORMAL)
      xstyle = FONT_STYLE_ITALIC;
    else if (style == wxITALIC && weight == wxBOLD)
      xstyle = FONT_STYLE_BOLD_ITALIC;
    else if (style == wxITALIC && weight == wxLIGHT)
      xstyle = FONT_STYLE_ITALIC;

    else if (style == wxSLANT && weight == wxNORMAL)
      xstyle = FONT_STYLE_OBLIQUE;
    else if (style == wxSLANT && weight == wxBOLD)
      xstyle = FONT_STYLE_BOLD_OBLIQUE;
    else if (style == wxSLANT && weight == wxLIGHT)
      xstyle = FONT_STYLE_OBLIQUE;

    else xstyle = FONT_STYLE_DEFAULT;

    font = (Xv_Font)xv_find(NULL, FONT,
                              FONT_FAMILY, xfamily,
                              FONT_STYLE, xstyle,
                              FONT_SIZE, point_size,
  			      NULL);

    if (font)
    {
      AddFont(family, style, weight, point_size_to_store, font);
    }
  }
  return font;
}
#endif

#ifdef wx_motif
XFontStruct *XFontPool::FindOrCreateFont(int family, int style, int weight, int point_size, int point_size_to_store, int xres, int yres)
{
  XFontStruct *font = FindFont(family, style, weight, point_size);

  if (!font)
  {
    char *xfamily;
    char *xstyle;
    char *xweight;
    switch (family)
    {
      case wxDECORATIVE: xfamily = "lucida";
                         break;
      case wxROMAN:      xfamily = "times";
                         break;
      case wxMODERN:     xfamily = "courier";
                         break;
      case wxSWISS:      xfamily = "lucida";
                         break;
      case wxDEFAULT:
        default:           xfamily = "*";
    }
    switch (style)
    {
      case wxITALIC:     xstyle = "i";
                         break;
      case wxSLANT:      xstyle = "o";
                         break;
      case wxNORMAL:     xstyle = "r";
                         break;
      default:           xstyle = "*";
                         break;
    }
    switch (weight)
    {
      case wxBOLD:       xweight = "bold";
                         break;
      case wxLIGHT:
      case wxNORMAL:     xweight = "medium";
                         break;
      default:           xweight = "*";
                         break;
    }

    sprintf(wxBuffer, "-*-%s-%s-%s-normal-*-*-%d-*-*-*-*-*-*",
            xfamily, xweight, xstyle, point_size);

    Display *dpy = XtDisplay(wxTheApp->topLevel);
    font = XLoadQueryFont(dpy, wxBuffer);

    if (font)
    {
      AddFont(family, style, weight, point_size_to_store, font);
    }
  }
  return font;
}
#endif

#ifdef wx_xview
Xv_Font
#else
XFontStruct *
#endif
  XFontPool::FindNearestFont(int family, int style, int weight, int point_size, int xres, int yres)
{
#ifdef wx_xview
  int max_size = 50;
  int min_size = 1;
  int search_inc = 1;
#else
  int max_size = 500;
  int min_size = 1;
  int search_inc = 10;
#endif
#ifdef wx_xview
  Xv_Font
#else
  XFontStruct *
#endif
    font = FindOrCreateFont(family, style, weight, point_size, point_size, xres, yres);
  if (!font)
  {
    int i;
    i = point_size - search_inc;
    while (!font && i > min_size)
    {
      font = FindOrCreateFont(family, style, weight, i, point_size, xres, yres);
      i -= search_inc;
    }

    i = point_size + search_inc;
    while (!font && i < max_size)
    {
      font = FindOrCreateFont(family, style, weight, i, point_size, xres, yres);
      i += search_inc;
    }
    // Remember what font to use for next time
    if (font)
      AddFont(family, style, weight, point_size, font);

    if (!font)
    {
#ifdef wx_xview
      int bog_standard = 12;
#endif
#ifdef wx_motif
      int bog_standard = 120;
#endif
      // For next time this happens, remember we just use a bog standard one
      font = FindOrCreateFont(wxDEFAULT, wxNORMAL, wxNORMAL, bog_standard, point_size,
             100, 100);
    }
  }
  return font;
}

#endif



// Colour

wxColour::wxColour(char r, char g, char b)
{
  red = r; green = g; blue = b;
}

wxColour::wxColour(char *col)
{
  wxColour *the_colour = wxTheColourDatabase->FindColour(col);
  if (the_colour)
  {
    red = the_colour->Red(); green = the_colour->Green();
    blue = the_colour->Blue();
  }
  else
  {
    red = 0; green = 0; blue = 0;
  }
}


void wxColour::Set(char r, char g, char b)
{
  red = r; green = g; blue = b;
}

void wxColour::Get(char *r, char *g, char *b)
{
  *r = red; *g = green; *b = blue;
}

char wxColour::Red(void)
{
  return red;
}

char wxColour::Green(void)
{
  return green;
}

char wxColour::Blue(void)
{
  return blue;
}

wxColourDatabase::wxColourDatabase(int type):wxList(type)
{
}

// Colour database stuff
void wxColourDatabase::Initialize(void)
{
  Append("AQUAMARINE"       , new wxColour(112, 219, 147));
  Append("BLACK"            , new wxColour(0, 0, 0));
  Append("BLUE"             , new wxColour(0, 0, 255));
  Append("BLUE VIOLET"      , new wxColour(159, 95, 159));
  Append("BROWN"            , new wxColour(165, 42, 42));
  Append("CADET BLUE"       , new wxColour(95, 159, 159));
  Append("CORAL"            , new wxColour(255, 127, 0));
  Append("CORNFLOWER BLUE"  , new wxColour(66, 66, 111));
  Append("CYAN"             , new wxColour(0, 255, 255));
  Append("DARK GREY"        , new wxColour(47, 47, 47));  // ?
  Append("DARK GREEN"       , new wxColour(47, 79, 47));
  Append("DARK OLIVE GREEN" , new wxColour(79, 79, 47));
  Append("DARK ORCHID"      , new wxColour(153, 50, 204));
  Append("DARK SLATE BLUE"  , new wxColour(107, 35, 142));
  Append("DARK SLATE GREY"  , new wxColour(47, 79, 79));
  Append("DARK TURQUOISE"   , new wxColour(112, 147, 219));
  Append("DIM GREY"         , new wxColour(84, 84, 84));
  Append("FIREBRICK"        , new wxColour(142, 35, 35));
  Append("FOREST GREEN"     , new wxColour(35, 142, 35));
  Append("GOLD"             , new wxColour(204, 127, 50));
  Append("GOLDENROD"        , new wxColour(219, 219, 112));
  Append("GREY"             , new wxColour(192, 192, 192));
  Append("GREEN"            , new wxColour(0, 255, 0));
  Append("GREEN YELLOW"     , new wxColour(147, 219, 112));
  Append("INDIAN RED"       , new wxColour(79, 47, 47));
  Append("KHAKI"            , new wxColour(159, 159, 95));
  Append("LIGHT BLUE"       , new wxColour(191, 216, 216));
  Append("LIGHT GREY"       , new wxColour(168, 168, 168));
//  Append("LIGHT GREY"       , new wxColour(200, 200, 200));
  Append("LIGHT STEEL BLUE" , new wxColour(143, 143, 188));
  Append("LIME GREEN"       , new wxColour(50, 204, 50));
  Append("MAGENTA"          , new wxColour(255, 0, 255));
  Append("MAROON"           , new wxColour(142, 35, 107));
  Append("MEDIUM AQUAMARINE" , new wxColour(50, 204, 153));
  Append("MEDIUM GREY" , new wxColour(100, 100, 100));
  Append("MEDIUM BLUE"      , new wxColour(50, 50, 204));
  Append("MEDIUM FOREST GREEN" , new wxColour(107, 142, 35));
  Append("MEDIUM GOLDENROD" , new wxColour(234, 234, 173));
  Append("MEDIUM ORCHID"    , new wxColour(147, 112, 219));
  Append("MEDIUM SEA GREEN" , new wxColour(66, 111, 66));
  Append("MEDIUM SLATE BLUE" , new wxColour(127, 0, 255));
  Append("MEDIUM SPRING GREEN" , new wxColour(127, 255, 0));
  Append("MEDIUM TURQUOISE" , new wxColour(112, 219, 219));
  Append("MEDIUM VIOLET RED" , new wxColour(219, 112, 147));
  Append("MIDNIGHT BLUE"    , new wxColour(47, 47, 79));
  Append("NAVY"             , new wxColour(35, 35, 142));
  Append("ORANGE"           , new wxColour(204, 50, 50));
  Append("ORANGE RED"       , new wxColour(255, 0, 127));
  Append("ORCHID"           , new wxColour(219, 112, 219));
  Append("PALE GREEN"       , new wxColour(143, 188, 143));
  Append("PINK"             , new wxColour(188, 143, 234));
  Append("PLUM"             , new wxColour(234, 173, 234));
  Append("PURPLE"           , new wxColour(176, 0, 255));
  Append("RED"              , new wxColour(255, 0, 0));
  Append("SALMON"           , new wxColour(111, 66, 66));
  Append("SEA GREEN"        , new wxColour(35, 142, 107));
  Append("SIENNA"           , new wxColour(142, 107, 35));
  Append("SKY BLUE"         , new wxColour(50, 153, 204));
  Append("SLATE BLUE"       , new wxColour(0, 127, 255));
  Append("SPRING GREEN"     , new wxColour(0, 255, 127));
  Append("STEEL BLUE"       , new wxColour(35, 107, 142));
  Append("TAN"              , new wxColour(219, 147, 112));
  Append("THISTLE"          , new wxColour(216, 191, 216));
  Append("TURQUOISE"        , new wxColour(173, 234, 234));
  Append("VIOLET"           , new wxColour(79, 47, 79));
  Append("VIOLET RED"       , new wxColour(204, 50, 153));
  Append("WHEAT"            , new wxColour(216, 216, 191));
  Append("WHITE"            , new wxColour(255, 255, 255));
  Append("YELLOW"           , new wxColour(255, 255, 0));
  Append("YELLOW GREEN"     , new wxColour(153, 204, 50));
}

wxColour *wxColourDatabase::FindColour(char *colour)
{
  wxNode *node = Find(colour);
  if (node)
    return (wxColour *)node->Data();
  else
    return NULL;
}

char *wxColourDatabase::FindName(wxColour& colour)
{
  wxNode *node = First();
  char *found = NULL;
  char red = colour.Red();
  char green = colour.Green();
  char blue = colour.Blue();
  while (node && !found)
  {
    wxColour *col = (wxColour *)node->Data();
    if (col->Red() == red && col->Green() == green && col->Blue() == blue)
      found = node->key.string;
    else
      node = node->Next();
  }
  return found;
}


void wxInitializeStockObjects(void)
{
  wxTheBrushList = new wxBrushList;
  wxThePenList = new wxPenList;
  wxTheIconList = new wxGDIList;
  wxTheFontList =  new wxGDIList;
  wxTheBitmapList =  new wxGDIList;

#ifdef wx_motif
#endif
#ifdef wx_x
  wxFontPool = new XFontPool;
#endif

  wxNORMAL_FONT = new wxFont(12, wxMODERN, wxNORMAL, wxNORMAL);
  wxSMALL_FONT = new wxFont(10, wxSWISS, wxNORMAL, wxNORMAL);
  wxITALIC_FONT = new wxFont(12, wxROMAN, wxITALIC, wxNORMAL);
  wxSWISS_FONT = new wxFont(12, wxSWISS, wxNORMAL, wxNORMAL);
  wxRED_PEN = new wxPen("RED", 3, wxSOLID);
  wxCYAN_PEN = new wxPen("CYAN", 3, wxSOLID);
  wxGREEN_PEN = new wxPen("GREEN", 1, wxSOLID);
  wxBLACK_PEN = new wxPen("BLACK", 1, wxSOLID);
  wxTRANSPARENT_PEN = new wxPen("BLACK", 1, wxTRANSPARENT);
  wxBLACK_DASHED_PEN = new wxPen("BLACK", 1, wxSHORT_DASH);

  wxBLUE_BRUSH = new wxBrush("BLUE", wxSOLID);
  wxGREEN_BRUSH = new wxBrush("GREEN", wxSOLID);
  wxWHITE_BRUSH = new wxBrush("WHITE", wxSOLID);
  wxBLACK_BRUSH = new wxBrush("BLACK", wxSOLID);
  wxTRANSPARENT_BRUSH = new wxBrush("BLACK", wxTRANSPARENT);
  wxCYAN_BRUSH = new wxBrush("CYAN", wxSOLID);
  wxRED_BRUSH = new wxBrush("RED", wxSOLID);
  wxGREY_BRUSH = new wxBrush("GREY", wxSOLID);
  wxMEDIUM_GREY_BRUSH = new wxBrush("MEDIUM GREY", wxSOLID);
  wxLIGHT_GREY_BRUSH = new wxBrush("LIGHT GREY", wxSOLID);

  wxBLACK = new wxColour("BLACK");
  wxWHITE = new wxColour("WHITE");
  wxRED = new wxColour("RED");
  wxBLUE = new wxColour("BLUE");
  wxGREEN = new wxColour("GREEN");
  wxCYAN = new wxColour("CYAN");
  wxLIGHT_GREY = new wxColour("LIGHT GREY");

  wxSTANDARD_CURSOR = new wxCursor(wxCURSOR_ARROW);
  wxHOURGLASS_CURSOR = new wxCursor(wxCURSOR_WAIT);
  wxCROSS_CURSOR = new wxCursor(wxCURSOR_CROSS);
}

// Pens

wxPen::wxPen(void)
{
  colour = NULL;
  style = wxSOLID;
  width = 1;
#ifdef wx_msw
  cpen = NULL;
#endif
  wxThePenList->AddPen(this);
}

wxPen::~wxPen()
{
  if (colour)
    delete colour;

#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);
#endif
  wxThePenList->RemovePen(this);
}

wxPen::wxPen(wxColour& col, int Width, int Style)
{
  colour = new wxColour(col.Red(), col.Green(), col.Blue());
  width = Width;
  style = Style;
#ifdef wx_msw
  DWORD ms_colour = RGB(col.Red(), col.Green(), col.Blue());
  cpen = CreatePen(wx2msPenStyle(Style), Width, ms_colour);
#endif
  wxThePenList->AddPen(this);
}

wxPen::wxPen(char *col, int Width, int Style)
{
  colour = new wxColour(col);
  width = Width;
  style = Style;
#ifdef wx_msw
  DWORD ms_colour = RGB(colour->Red(), colour->Green(), colour->Blue());
  cpen = CreatePen(wx2msPenStyle(Style), Width, ms_colour);
#endif
  wxThePenList->AddPen(this);
}

void wxPen::SetColour(wxColour& col)
{
  if (colour)
    delete colour;
  colour = new wxColour(col.Red(), col.Green(), col. Blue());
#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);

  DWORD ms_colour = RGB(col.Red(), col.Green(), col.Blue());
  cpen = CreatePen(wx2msPenStyle(style), width, ms_colour);
#endif
}

void wxPen::SetColour(char *col)
{
  if (colour)
    delete colour;
  colour = new wxColour(col);
#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);

  DWORD ms_colour = RGB(colour->Red(), colour->Green(), colour->Blue());
  cpen = CreatePen(wx2msPenStyle(style), width, ms_colour);
#endif
}

void wxPen::SetColour(char red, char green, char blue)
{
  if (colour)
    delete colour;
  colour = new wxColour(red, green, blue);
#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);

  DWORD ms_colour = RGB(red, green, blue);
  cpen = CreatePen(wx2msPenStyle(style), width, ms_colour);
#endif
}

void wxPen::SetWidth(int Width)
{
  width = Width;
#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);

  DWORD ms_colour = 0;
  if (colour)
    ms_colour = RGB(colour->Red(), colour->Green(), colour->Blue());
  cpen = CreatePen(wx2msPenStyle(style), width, ms_colour);
#endif
}

void wxPen::SetStyle(int Style)
{
  style = Style;
#ifdef wx_msw
  if (cpen)
    DeleteObject(cpen);

  DWORD ms_colour = 0;
  if (colour)
    ms_colour = RGB(colour->Red(), colour->Green(), colour->Blue());
  cpen = CreatePen(wx2msPenStyle(style), width, ms_colour);
#endif
}

int wxPen::GetWidth(void)
{
  return width;
}

int wxPen::GetStyle(void)
{
  return style;
}

wxColour& wxPen::GetColour(void)
{
  return *colour;
}

#ifdef wx_msw
int wx2msPenStyle(int wx_style)
{
  int cstyle;
  switch (wx_style)
  {
    case wxDOT:
      cstyle = PS_DOT;
      break;
    case wxSHORT_DASH:
    case wxLONG_DASH:
      cstyle = PS_DASH;
      break;
    case wxTRANSPARENT:
      cstyle = PS_NULL;
      break;
    case wxSOLID:
    default:
      cstyle = PS_SOLID;
      break;
  }
  return cstyle;
}
#endif

// Brushes

wxBrush::wxBrush(void)
{
  colour = NULL;
  style = wxSOLID;
#ifdef wx_msw
  cbrush = NULL;
#endif
  wxTheBrushList->AddBrush(this);
}

wxBrush::~wxBrush()
{
  if (colour)
    delete colour;
#ifdef wx_msw
  if (cbrush)
    DeleteObject(cbrush);
#endif
  wxTheBrushList->RemoveBrush(this);
}

wxBrush::wxBrush(wxColour& col, int Style)
{
  colour = new wxColour(col.Red(), col.Green(), col. Blue());
  style = Style;
#ifdef wx_msw
  switch (Style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;  // Must always select a suitable background brush
                      // - could choose white always for a quick solution
      break;
    case wxSOLID:
    default:
      cbrush = CreateSolidBrush(RGB(col.Red(), col.Green(), col.Blue()));
  }
#endif
  wxTheBrushList->AddBrush(this);
}

wxBrush::wxBrush(char *col, int Style)
{
  colour = new wxColour(col);
  style = Style;
#ifdef wx_msw
  switch (Style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;  // Must always select a suitable background brush
                      // - could choose white always for a quick solution
      break;
    case wxSOLID:
    default:
      cbrush = CreateSolidBrush(RGB(colour->Red(), colour->Green(), colour->Blue()));
  }
#endif
  wxTheBrushList->AddBrush(this);
}

void wxBrush::SetColour(wxColour& col)
{
  if (colour)
    delete colour;
  colour = new wxColour(col.Red(), col.Green(), col. Blue());

#ifdef wx_msw
  if (cbrush) DeleteObject(cbrush);
  switch (style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;  // Must always select a suitable background brush
                      // - could choose white always for a quick solution
      break;
    case wxSOLID:
    default:
      cbrush = CreateSolidBrush(RGB(col.Red(), col.Green(), col.Blue()));
  }
#endif
}

void wxBrush::SetColour(char *col)
{
  if (colour)
    delete colour;
  colour = new wxColour(col);

#ifdef wx_msw
  if (cbrush) DeleteObject(cbrush);
  switch (style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;  // Must always select a suitable background brush
                      // - could choose white always for a quick solution
      break;
    case wxSOLID:
    default:
      cbrush = CreateSolidBrush(RGB(colour->Red(), colour->Green(), colour->Blue()));
  }
#endif
}

void wxBrush::SetColour(char red, char green, char blue)
{
  if (colour)
    delete colour;
  colour = new wxColour(red, green, blue);
#ifdef wx_msw
  if (cbrush) DeleteObject(cbrush);
  switch (style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;  // Must always select a suitable background brush
                      // - could choose white always for a quick solution
      break;
    case wxSOLID:
    default:
      cbrush = CreateSolidBrush(RGB(red, green, blue));
  }
#endif
}

void wxBrush::SetStyle(int Style)
{
  style = Style;
#ifdef wx_msw
  if (cbrush) DeleteObject(cbrush);
  switch (Style)
  {
    case wxTRANSPARENT:
      cbrush = NULL;     // Don't fill shapes
      break;
    case wxSOLID:
    default:
      if (colour)
      {
        cbrush = CreateSolidBrush(RGB(colour->Red(), colour->Green(), colour->Blue()));
      }
      else cbrush = NULL;
  }
#endif
}

int wxBrush::GetStyle(void)
{
  return style;
}

wxColour& wxBrush::GetColour(void)
{
  return *colour;
}

// Icons
wxIcon::wxIcon(short bits[], int width, int height)
{
#ifdef wx_motif
  iconWidth = width;
  iconHeight = height;
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  x_pixmap = XCreateBitmapFromData(dpy, RootWindow(dpy, DefaultScreen(dpy)), (char *)bits, width, height);
#endif
#ifdef wx_xview
  x_image = (Server_image)xv_create(NULL, SERVER_IMAGE,
				       XV_WIDTH,	width,
				       XV_HEIGHT,	height,
				       SERVER_IMAGE_BITS, bits,
				       NULL);
  x_icon = (Icon)xv_create(NULL, ICON, ICON_IMAGE, x_image, NULL);
#endif
  wxTheIconList->Append(this);
}

wxIcon::wxIcon(char *icon_file)
{
#ifdef wx_motif
  int hotX, hotY;
  unsigned int w, h;
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  int value = XReadBitmapFile(dpy, RootWindow(dpy, DefaultScreen(dpy)), icon_file, &w, &h, &x_pixmap, &hotX, &hotY);
  iconWidth = w;
  iconHeight = h;
  if ((value == BitmapFileInvalid) || (value == BitmapOpenFailed) || (value == BitmapNoMemory))
    x_pixmap = 0;
#endif
#ifdef wx_xview
  x_image = (Server_image)xv_create(NULL, SERVER_IMAGE,
                                       SERVER_IMAGE_BITMAP_FILE, icon_file,
				       NULL);
  x_icon = (Icon)xv_create(NULL, ICON, ICON_IMAGE, x_image, NULL);
#endif
#ifdef wx_msw
  ms_icon = LoadIcon(wxhInstance, icon_file);
#endif
  wxTheIconList->Append(this);
}

wxIcon::~wxIcon(void)
{
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  if (x_pixmap)
    XFreePixmap(dpy, x_pixmap);
#endif
#ifdef wx_xview
  xv_destroy_safe(x_icon);
#endif
#ifdef wx_msw
  if (ms_icon)
    DestroyIcon(ms_icon);
#endif
  wxTheIconList->DeleteObject(this);
}

wxGDIList::wxGDIList(void)
{
}

wxGDIList::~wxGDIList(void)
{
  wxNode *node = First();
  while (node)
  {
    wxObject *object = (wxObject *)node->Data();
    wxNode *next = node->Next();
    delete object;
    node = next;
  }
}

// Cursors
// NB Should delete loaded Windows cursors with DestroyCursor!!!
wxCursor::wxCursor(short bits[], int width, int height)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Server_image image;
  image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
				       XV_WIDTH,	width,
				       XV_HEIGHT,	height,
				       SERVER_IMAGE_BITS, bits,
				       NULL);
  x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, image, NULL);
#endif
}

wxCursor::wxCursor(char *cursor_file)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Server_image image;
  image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
                                       SERVER_IMAGE_BITMAP_FILE, cursor_file,
				       NULL);
  x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, image, NULL);
#endif
#ifdef wx_msw
  ms_cursor = LoadCursor(wxhInstance, cursor_file);
#endif
}

// Cursors by stock number
wxCursor::wxCursor(int cursor_type)
{
#ifdef wx_motif
  x_cursor = 0;
#endif
#ifdef wx_x
  use_raw_x_cursor = FALSE;
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);
  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
#endif

  int x_cursor_type;
  x_cursor_type = 0;
  switch (cursor_type)
  {
    case wxCURSOR_WAIT:
    {
#ifdef wx_xview
      x_cursor_type = OLC_BUSY_PTR;
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_SRC_CHAR, x_cursor_type, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_watch);
#endif
      break;
    }
    case wxCURSOR_CROSS:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, crosshair_cursor_data, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image,
                           CURSOR_XHOT, 3, CURSOR_YHOT, 3, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_crosshair);
#endif
      break;
    }
    case wxCURSOR_CHAR:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, char_cursor_data, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image,
                           CURSOR_XHOT, 0, CURSOR_YHOT, 13, NULL);
#endif
#ifdef wx_motif
#endif
      break;
    }
    case wxCURSOR_HAND:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, hand_cursor_array, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image,
                           CURSOR_XHOT, 5, CURSOR_YHOT, 0, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_hand1);
#endif
      break;
    }
    case wxCURSOR_BULLSEYE:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, bull_cursor_array, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image, 
                           CURSOR_XHOT, 5, CURSOR_YHOT, 5, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_target);
#endif
      break;
    }
    case wxCURSOR_PENCIL:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, pencil_cursor_array, 
              NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image, 
                           CURSOR_XHOT, 0, CURSOR_YHOT, 14, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_pencil);
#endif
      break;
    }
    case wxCURSOR_MAGNIFIER:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, magnifier_cursor_array, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image,
                           CURSOR_XHOT, 6, CURSOR_YHOT, 6, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_sizing);
#endif
      break;
    }
    case wxCURSOR_IBEAM:
    {
#ifdef wx_xview
      Server_image svr_image = (Server_image)xv_create(XV_NULL, SERVER_IMAGE,
              XV_WIDTH, 16, XV_HEIGHT, 16, SERVER_IMAGE_BITS, vbar_cursor_array, NULL);
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_IMAGE, svr_image,
                           CURSOR_XHOT, 0, CURSOR_YHOT, 13, NULL);
#endif
#ifdef wx_motif
#endif
      break;
    }
    case wxCURSOR_NO_ENTRY:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_pirate);
      break;
    }

    case wxCURSOR_LEFT_BUTTON:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_leftbutton);
      break;
    }
    case wxCURSOR_RIGHT_BUTTON:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_rightbutton);
      break;
    }
    case wxCURSOR_MIDDLE_BUTTON:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_middlebutton);
      break;
    }
    case wxCURSOR_QUESTION_ARROW:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_question_arrow);
      break;
    }
    case wxCURSOR_SIZING:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_sizing);
      break;
    }
    case wxCURSOR_WATCH:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_watch);
      break;
    }
    case wxCURSOR_SPRAYCAN:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_spraycan);
      break;
    }
    case wxCURSOR_PAINT_BRUSH:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_spraycan);
      break;
    }
    case wxCURSOR_SIZENWSE:
    case wxCURSOR_SIZENESW:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_circle);
      break;
    }
    case wxCURSOR_SIZEWE:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_sb_h_double_arrow);
      break;
    }
    case wxCURSOR_SIZENS:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_sb_v_double_arrow);
      break;
    }
    case wxCURSOR_POINT_LEFT:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_sb_left_arrow);
      break;
    }
    case wxCURSOR_POINT_RIGHT:
    {
      use_raw_x_cursor = TRUE;
      x_cursor = XCreateFontCursor(dpy, XC_sb_right_arrow);
      break;
    }
    default:
    case wxCURSOR_ARROW:
    {
#ifdef wx_xview
      x_cursor_type = OLC_BASIC_PTR;
      x_cursor = xv_create(XV_NULL, CURSOR, CURSOR_SRC_CHAR, x_cursor_type, NULL);
#endif
#ifdef wx_motif
      x_cursor = XCreateFontCursor(dpy, XC_top_left_arrow);
#endif
      break;
    }
  }
#endif
#ifdef wx_msw
  switch (cursor_type)
  {
    case wxCURSOR_WAIT:
      ms_cursor = LoadCursor(NULL, IDC_WAIT);
      break;
    case wxCURSOR_IBEAM:
      ms_cursor = LoadCursor(NULL, IDC_IBEAM);
      break;
    case wxCURSOR_CROSS:
      ms_cursor = LoadCursor(NULL, IDC_CROSS);
      break;
    case wxCURSOR_SIZENWSE:
      ms_cursor = LoadCursor(NULL, IDC_SIZENWSE);
      break;
    case wxCURSOR_SIZENESW:
      ms_cursor = LoadCursor(NULL, IDC_SIZENESW);
      break;
    case wxCURSOR_SIZEWE:
      ms_cursor = LoadCursor(NULL, IDC_SIZEWE);
      break;
    case wxCURSOR_SIZENS:
      ms_cursor = LoadCursor(NULL, IDC_SIZENS);
      break;
    case wxCURSOR_CHAR:
    {
      ms_cursor = LoadCursor(NULL, IDC_ARROW);
      break;
    }
    case wxCURSOR_HAND:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_HAND");
      break;
    }
    case wxCURSOR_BULLSEYE:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_BULLSEYE");
      break;
    }
    case wxCURSOR_PENCIL:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_PENCIL");
      break;
    }
    case wxCURSOR_MAGNIFIER:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_MAGNIFIER");
      break;
    }
    case wxCURSOR_NO_ENTRY:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_NO_ENTRY");
      break;
    }
    case wxCURSOR_LEFT_BUTTON:
    {
      ms_cursor = LoadCursor(NULL, IDC_ARROW);
      break;
    }
    case wxCURSOR_RIGHT_BUTTON:
    {
      ms_cursor = LoadCursor(NULL, IDC_ARROW);
      break;
    }
    case wxCURSOR_MIDDLE_BUTTON:
    {
      ms_cursor = LoadCursor(NULL, IDC_ARROW);
      break;
    }
    case wxCURSOR_SIZING:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_SIZING");
      break;
    }
    case wxCURSOR_WATCH:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_WATCH");
      break;
    }
    case wxCURSOR_SPRAYCAN:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_ROLLER");
      break;
    }
    case wxCURSOR_PAINT_BRUSH:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_PBRUSH");
      break;
    }
    case wxCURSOR_POINT_LEFT:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_PLEFT");
      break;
    }
    case wxCURSOR_POINT_RIGHT:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_PRIGHT");
      break;
    }
    case wxCURSOR_QUESTION_ARROW:
    {
      ms_cursor = LoadCursor(wxhInstance, "wxCURSOR_QARROW");
      break;
    }
    default:
    case wxCURSOR_ARROW:
      ms_cursor = LoadCursor(NULL, IDC_ARROW);
      break;
  }
#endif
}

wxCursor::~wxCursor(void)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  if (!use_raw_x_cursor)
    xv_destroy_safe(x_cursor);
#endif
}

// Global cursor setting
void wxSetCursor(wxCursor *cursor)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);
  if (cursor && cursor->x_cursor)
  {
    if (cursor->use_raw_x_cursor)
    {
      Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
      Window win2 = xv_get(root_window, XV_XID);

      XDefineCursor(dpy, win2, cursor->x_cursor);
    }
    else
      xv_set(root_window, WIN_CURSOR, cursor->x_cursor, NULL);
  }

#endif
#ifdef wx_msw
  if (cursor)
    ::SetCursor(cursor->ms_cursor);
#endif
}

// Pen and Brush lists
wxPenList::~wxPenList(void)
{
  wxNode *node = First();
  while (node)
  {
    wxPen *pen = (wxPen *)node->Data();
    delete pen;
    node = First();
  }
}

void wxPenList::AddPen(wxPen *pen)
{
  Append(pen);
}

void wxPenList::RemovePen(wxPen *pen)
{
  DeleteObject(pen);
}

wxPen *wxPenList::FindOrCreatePen(wxColour *colour, int width, int style)
{
  wxPen *found = NULL;
  wxNode *node = First();
  while (node && !found)
  {
    wxPen *each_pen = (wxPen *)node->Data();
    if (each_pen->width == width && each_pen->style == style &&
        each_pen->colour && each_pen->colour->Red() == colour->Red() &&
        each_pen->colour->Green() == colour->Green() &&
        each_pen->colour->Blue() == colour->Blue())
      found = each_pen;
    node = node->Next();
  }
  if (found)
    return found;
  else
    return new wxPen(*colour, width, style);
}

wxPen *wxPenList::FindOrCreatePen(char *colour, int width, int style)
{
  wxColour *the_colour = wxTheColourDatabase->FindColour(colour);
  if (colour)
    return FindOrCreatePen(the_colour, width, style);
  else return NULL;
}

wxBrushList::~wxBrushList(void)
{
  wxNode *node = First();
  while (node)
  {
    wxBrush *brush = (wxBrush *)node->Data();
    delete brush;
    node = First();
  }
}

void wxBrushList::AddBrush(wxBrush *brush)
{
  Append(brush);
}

wxBrush *wxBrushList::FindOrCreateBrush(wxColour *colour, int style)
{
  wxBrush *found = NULL;
  wxNode *node = First();
  while (node && !found)
  {
    wxBrush *each_brush = (wxBrush *)node->Data();
    if (each_brush->style == style &&
        each_brush->colour && each_brush->colour->Red() == colour->Red() &&
        each_brush->colour->Green() == colour->Green() &&
        each_brush->colour->Blue() == colour->Blue())
      found = each_brush;
    node = node->Next();
  }
  if (found)
    return found;
  else
    return new wxBrush(*colour, style);
}

wxBrush *wxBrushList::FindOrCreateBrush(char *colour, int style)
{
  wxColour *the_colour = wxTheColourDatabase->FindColour(colour);
  if (colour)
    return FindOrCreateBrush(the_colour, style);
  else return NULL;
}


void wxBrushList::RemoveBrush(wxBrush *brush)
{
  DeleteObject(brush);
}

wxPoint::wxPoint(void)
{
}

wxPoint::wxPoint(float the_x, float the_y)
{
  x = the_x;
  y = the_y;
}

// Misc. functions

// Return TRUE if we have a colour display
Bool wxColourDisplay(void)
{
#ifdef wx_x
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);

  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
#endif
  if (DefaultDepth(dpy, DefaultScreen(dpy)) < 2)
      return FALSE;
    else
      return TRUE;
#endif
#ifdef wx_msw
  HDC dc = ::GetDC(NULL);
  Bool flag;
  if (GetDeviceCaps(dc, NUMCOLORS) > 2)
    flag = TRUE;
  else
    flag = FALSE;
  ReleaseDC(NULL, dc);
  return flag;
#endif
}

// Get size of display
void wxDisplaySize(int *width, int *height)
{
#ifdef wx_motif
  if (wxTheApp->topLevel)
  {
    Display *dpy = XtDisplay(wxTheApp->topLevel);
    *width = DisplayWidth(dpy, DefaultScreen(dpy));
    *height = DisplayHeight(dpy, DefaultScreen(dpy));
  }
  else
  {
    // A good bet!
    *width = 1024;
    *height = 768;
  }
#endif
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);

  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);

  int screen_no = (int)xv_get(screen, SCREEN_NUMBER);
  *width = DisplayWidth(dpy, screen_no);
  *height = DisplayHeight(dpy, screen_no);
#endif
#ifdef wx_msw
  HDC dc = ::GetDC(NULL);
  *width = GetDeviceCaps(dc, HORZRES); *height = GetDeviceCaps(dc, VERTRES);
  ReleaseDC(NULL, dc);
#endif
}

wxBitmap::wxBitmap(short bits[], int the_width, int the_height, int no_bits)
{
  width = the_width;
  height = the_height;
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  x_pixmap = XCreateBitmapFromData(dpy, RootWindow(dpy, DefaultScreen(dpy)), (char *)bits, width, height);
#endif
#ifdef wx_xview
  Server_image x_image = (Server_image)xv_create(NULL, SERVER_IMAGE,
				       XV_WIDTH,	width,
				       XV_HEIGHT,	height,
				       SERVER_IMAGE_BITS, bits,
				       NULL);

/*

  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);

  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
  x_pixmap = XCreateBitmapFromData(dpy, RootWindow(dpy, DefaultScreen(dpy)), (char *)bits, width, height);
*/
  x_pixmap = (Pixmap)xv_get(x_image, SERVER_IMAGE_PIXMAP);
#endif
  wxTheBitmapList->Append(this);
}

wxBitmap::wxBitmap(char *bitmap_file)
{
#ifdef wx_motif
  int hotX, hotY;
  unsigned int w, h;
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  int value = XReadBitmapFile(dpy, RootWindow(dpy, DefaultScreen(dpy)), bitmap_file, &w, &h, &x_pixmap, &hotX, &hotY);
  width = w;
  height = h;
  if ((value == BitmapFileInvalid) || (value == BitmapOpenFailed) || (value == BitmapNoMemory))
    x_pixmap = 0;
#endif
#ifdef wx_xview
  Server_image x_image = (Server_image)xv_create(NULL, SERVER_IMAGE,
                                       SERVER_IMAGE_BITMAP_FILE, bitmap_file,
				       NULL);
/*
  int x_hot, y_hot;
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);

  unsigned int w, h;
  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
  if (!XReadBitmapFile(dpy, RootWindow(dpy, DefaultScreen(dpy)), bitmap_file,
                       &w, &h, &x_pixmap,
                       &x_hot, &y_hot))
    x_pixmap = 0;
  width = (int)w;
  height = (int)h;
*/
  x_pixmap = (Pixmap)xv_get(x_image, SERVER_IMAGE_PIXMAP);
#endif
#ifdef wx_msw
  ms_bitmap = LoadBitmap(wxhInstance, bitmap_file);
  BITMAP bm;
  GetObject(ms_bitmap, sizeof(BITMAP), (LPSTR) &bm);
  width = bm.bmWidth;
  height = bm.bmHeight;
#endif
  wxTheBitmapList->Append(this);
}

wxBitmap::~wxBitmap(void)
{
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
  if (x_pixmap)
    XDestroyWindow(dpy, x_pixmap);
#endif
#ifdef wx_xview
//  xv_destroy_safe(x_image);
#endif
#ifdef wx_msw
  if (ms_bitmap)
    DeleteObject(ms_bitmap);
#endif
  wxTheBitmapList->DeleteObject(this);
}
