/*
 * File:     wx_panel.cc
 * Purpose:  wxPanel class implementation
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>

#include "common.h"
#include "wx_panel.h"
#include "wx_frame.h"
#include "wx_event.h"
#include "wx_utils.h"
#include "wx_privt.h"

#ifdef wx_motif
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/Frame.h>
#endif

#ifdef wx_xview
#include <xview/panel.h>
extern void wxPanelEventProc(Panel panel, Event *event); // See wx_item.cc
#endif

// Constructor
wxPanel::wxPanel(wxFrame *frame, int x, int y, int width, int height, int style)
{
#ifdef wx_msw
  cursor_x = PANEL_LEFT_MARGIN;
  cursor_y = PANEL_TOP_MARGIN;
  max_height = 0;
  max_line_height = 0;
  max_width = 0;
  hspacing = PANEL_HSPACING;
  vspacing = PANEL_VSPACING;
#endif
  new_line = FALSE;
  label_position = wxHORIZONTAL;

  if (!frame)
    return;

  window_parent = frame;

#ifdef wx_motif
  borderWidget = 0;
  hSpacing = PANEL_HSPACING;
  vSpacing = PANEL_VSPACING;

  new_line = TRUE;
  firstRowWidget = NULL;
  lastWidget = NULL;

  if (style & wxBORDER)
  {
    borderWidget = XtVaCreateManagedWidget("main_window",
                    xmFrameWidgetClass,      frame->workArea,
                    XmNshadowType,           XmSHADOW_IN,
                    NULL);
  }

  panelWidget = XtVaCreateManagedWidget("panel",
                  xmFormWidgetClass, borderWidget ? borderWidget : frame->workArea,
                  NULL);

  handle = (char *)panelWidget;

  wxWidgetHashTable->Put((long)panelWidget, this);

  XtOverrideTranslations(panelWidget,
              XtParseTranslationTable("<Configure>: resize()"));

  SetSize(x, y, width, height);
#endif
#ifdef wx_xview
  int real_y = frame->y_offset;
  if (y > -1)
    real_y = y + frame->y_offset;  // Allow for possible menu bar

  Frame x_frame = (Frame)(frame->GetHandle());
  Panel x_panel = (Panel)xv_create(x_frame, PANEL,
                                   PANEL_BACKGROUND_PROC, wxPanelEventProc,
                                   PANEL_ACCEPT_KEYSTROKE, TRUE,
                                   WIN_CLIENT_DATA, (char *)this,
                                   PANEL_LAYOUT, PANEL_HORIZONTAL, XV_SHOW, FALSE,
                                   NULL);

  if (x > -1)
    xv_set(x_panel, XV_X, x, NULL);

  if (y > -1)
    xv_set(x_panel, XV_Y, real_y, NULL);

  // If width = -1, will extend to edge of frame
  xv_set(x_panel, XV_WIDTH, width, NULL);

  if (height > -1)
    xv_set(x_panel, XV_HEIGHT, height, NULL);

  xv_set(x_panel, XV_SHOW, TRUE, NULL);

  handle = (char *)x_panel;
#endif
#ifdef wx_msw
  wxWinType = wxTYPE_XWND;
  wxWnd *cparent = NULL;
  if (frame)
    cparent = (wxWnd *)frame->handle;

  DWORD msflags = 0;
  if (style & wxBORDER)
    msflags |= WS_BORDER;
  msflags |= WS_CHILD | WS_VISIBLE;

  wxSubWnd *wnd = new wxSubWnd(cparent, "wxPanelClass", this, x, y, width, height, msflags);

  handle = (char *)wnd;
#endif

  if (frame) frame->AddChild(this);
}

wxPanel::~wxPanel(void)
{
#ifdef wx_motif
  if (borderWidget)
    XtDestroyWidget(borderWidget);
#endif
}

void wxPanel::SetLabelPosition(int pos)  // wxHORIZONTAL or wxVERTICAL
{
  label_position = pos;
}

void wxPanel::SetSize(int x, int y, int w, int h)
{
#ifdef wx_motif
  int real_y = y;

  if (x > -1)
    XtVaSetValues(borderWidget ? borderWidget : panelWidget,
                  XmNleftAttachment, XmATTACH_SELF,
                  XmNx, x, 
                  NULL);
  if (y > -1)
    XtVaSetValues(borderWidget ? borderWidget : panelWidget,
                  XmNtopAttachment, XmATTACH_SELF,
                  XmNy, real_y,
                  NULL);
  if (w > -1)
  {
    if (borderWidget)
    {
      XtVaSetValues(borderWidget, XmNrightAttachment, XmATTACH_SELF, XmNwidth, w, NULL);
      XtVaSetValues(panelWidget, XmNwidth, w, NULL);
    }
    else
      XtVaSetValues(panelWidget, XmNrightAttachment, XmATTACH_SELF, XmNwidth, w, NULL);
  }
  if (h > -1)
  {
    if (borderWidget)
    {
      XtVaSetValues(borderWidget, XmNbottomAttachment, XmATTACH_SELF, XmNheight, h, NULL);
      XtVaSetValues(panelWidget, XmNheight, h, NULL);
    }
    else
      XtVaSetValues(panelWidget, XmNbottomAttachment, XmATTACH_SELF, XmNheight, h, NULL);
  }
#endif
#ifdef wx_xview
  int real_y = y;

  if (window_parent)
    real_y += ((wxFrame *)window_parent)->y_offset;  // Allow for possible menu bar

  Xv_opaque object = (Xv_opaque)handle;
  (void)xv_set(object, XV_X, x, XV_Y, real_y, XV_WIDTH, w, XV_HEIGHT, h, NULL);
#endif
#ifdef wx_msw
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  wxWnd *wnd = (wxWnd *)handle;
  if (wnd)
    MoveWindow(wnd->handle, x, y, w, h, TRUE);
#endif
  OnSize(w, h);
}

void wxPanel::SetClientSize(int w, int h)
{
#ifdef wx_motif
  if (w > -1)
  {
    if (borderWidget)
    {
      XtVaSetValues(borderWidget, XmNrightAttachment, XmATTACH_SELF, XmNwidth, w, NULL);
      XtVaSetValues(panelWidget, XmNwidth, w, NULL);
    }
    else
      XtVaSetValues(panelWidget, XmNrightAttachment, XmATTACH_SELF, XmNwidth, w, NULL);
  }
  if (h > -1)
  {
    if (borderWidget)
    {
      XtVaSetValues(borderWidget, XmNbottomAttachment, XmATTACH_SELF, XmNheight, h, NULL);
      XtVaSetValues(panelWidget, XmNheight, h, NULL);
    }
    else
      XtVaSetValues(panelWidget, XmNbottomAttachment, XmATTACH_SELF, XmNheight, h, NULL);
  }
  OnSize(w, h);
#endif
#ifdef wx_xview
  wxWindow::SetClientSize(w, h);
#endif
#ifdef wx_msw
  wxWindow::SetClientSize(w, h);
#endif
}

void wxPanel::GetPosition(int *x, int *y)
{
#ifdef wx_motif
  Dimension xx, yy;
  XtVaGetValues(borderWidget ? borderWidget : (Widget)handle,
                XmNx, &xx, XmNy, &yy, NULL);
  *x = xx;
  *y = yy;
#else
  wxWindow::GetPosition(x, y);
#endif
}

/*****************************************************************
 * ITEM PLACEMENT FUNCTIONS
 *****************************************************************/


// Start a new line
void wxPanel::NewLine(void)
{
#ifdef wx_motif
  new_line = TRUE;
#endif
#ifdef wx_xview
  Panel panel = (Panel)handle;
  if (new_line)
  {
      (void) xv_create(panel, PANEL_MESSAGE,
                             PANEL_LAYOUT, PANEL_HORIZONTAL,
                             PANEL_NEXT_ROW, -1, NULL);
  }
  new_line = TRUE;
#endif
#ifdef wx_msw
  cursor_x = PANEL_LEFT_MARGIN;
  if (max_line_height == 0)
  {
    cursor_y += vspacing;
  }
  else
    cursor_y += vspacing + max_line_height;
  max_line_height = 0;
#endif
}

void wxPanel::Tab(int pixels)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Panel panel = (Panel)handle;
    (void) xv_create(panel, PANEL_MESSAGE,
                           PANEL_LAYOUT, PANEL_HORIZONTAL,
                           XV_WIDTH, pixels,
                           NULL);
#endif
#ifdef wx_msw
  cursor_x += pixels;
  if (cursor_x > max_width)
    max_width = cursor_x;
#endif
}

void wxPanel::GetCursor(int *x, int *y)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  *x = (int)xv_get(object, PANEL_ITEM_X);
  *y = (int)xv_get(object, PANEL_ITEM_Y);
#endif
#ifdef wx_msw
  *x = cursor_x;
  *y = cursor_y;
#endif
}

// Set/get horizontal spacing
void wxPanel::SetHorizontalSpacing(int sp)
{
#ifdef wx_motif
  hSpacing = sp;
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  (void)xv_set(object, PANEL_ITEM_X_GAP, sp, NULL);
#endif
#ifdef wx_msw
  hspacing = sp;
#endif
}

int wxPanel::GetHorizontalSpacing(void)
{
#ifdef wx_motif
  return hSpacing;
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  return (int)xv_get(object, PANEL_ITEM_X_GAP);
#endif
#ifdef wx_msw
  return hspacing;
#endif
}

// Set/get vertical spacing
void wxPanel::SetVerticalSpacing(int sp)
{
#ifdef wx_motif
  vSpacing = sp;
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  (void)xv_set(object, PANEL_ITEM_Y_GAP, sp, NULL);
#endif
#ifdef wx_msw
  vspacing = sp;
#endif
}

int wxPanel::GetVerticalSpacing(void)
{
#ifdef wx_motif
  return vSpacing;
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  return (int)xv_get(object, PANEL_ITEM_Y_GAP);
#endif
#ifdef wx_msw
  return vspacing;
#endif
}

// Fits the panel around the items
void wxPanel::Fit(void)
{
#ifdef wx_motif
  int maxX = 0;
  int maxY = 0;
  wxNode *node = GetChildren()->First();
  while (node)
  {
    wxWindow *win = (wxWindow *)node->Data();
    int x, y, w, h;
    win->GetPosition(&x, &y);
    win->GetSize(&w, &h);
    if ((x + w) > maxX)
      maxX = x + w;
    if ((y + h) > maxY)
      maxY = y + h;
    node = node->Next();
  }
  SetClientSize(maxX + hSpacing, maxY + vSpacing);
#endif
#ifdef wx_xview
  Xv_opaque object = (Xv_opaque)handle;
  window_fit(object);
#endif
#ifdef wx_msw
  SetClientSize(max_width + PANEL_HSPACING,
                max_height + PANEL_VSPACING);
#endif
}

// Update next cursor position
void wxPanel::AdvanceCursor(wxWindow *item)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
#endif
#ifdef wx_msw
  int width, height;
  int x, y;
  item->GetSize(&width, &height);
  item->GetPosition(&x, &y);

  if ((x + width) > max_width)
    max_width = x + width;
  if ((y + height) > max_height)
    max_height = y + height;
  if (height > max_line_height)
    max_line_height = height;

  cursor_x = x + width + hspacing;
  cursor_y = y;
#endif
}

// If x or y are not specified (i.e. < 0), supply
// values based on left to right, top to bottom layout.
// Internal use only.
void wxPanel::GetValidPosition(int *x, int *y)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
#endif
#ifdef wx_msw
  if (*x < 0 || *y < 0)
  {
    *x = cursor_x;
    *y = cursor_y;
  }
#endif
}

