/*
 * File:     wx_timer.cc
 * Purpose:  wxTimer implementation
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <time.h>
#include <sys/types.h>
#include <sys/timeb.h>
#include "wx_timer.h"
#include "wx_list.h"
#include "wx_main.h"
#include "wx_frame.h"

#ifdef wx_motif
#include <Xm/Xm.h>
#endif

#ifdef wx_x // Can't find an include file with this in!!
extern "C" int ftime(struct timeb *);
#endif

#ifdef wx_motif
void wxTimerCallback(wxTimer *timer)
{
  timer->timerId = XtAppAddTimeOut(wxTheApp->appContext, timer->milli, wxTimerCallback, timer);
  timer->Notify();
}
#endif

#ifdef wx_msw
wxList wxTimerList(wxKEY_INTEGER);
UINT FAR PASCAL _export wxTimerProc(HWND hwnd, WORD, int idTimer, DWORD);
#endif

#ifdef wx_xview
Notify_value wxTimerFunc(Notify_client client, int which);
#endif

wxTimer::wxTimer(void)
{
#ifdef wx_motif
  timerId = 0;
  milli = 0;
#endif
#ifdef wx_xview
  timerval.it_value.tv_usec = 0;
  timerval.it_interval.tv_usec = 0;
#endif
#ifdef wx_msw
  id = 0;
  milli = -1;
#endif
}

wxTimer::~wxTimer(void)
{
  Stop();
#ifdef wx_msw
  wxTimerList.DeleteObject(this);
#endif
}

Bool wxTimer::Start(int milliseconds)
{
#ifdef wx_motif
  milli = milliseconds;
  timerId = XtAppAddTimeOut(wxTheApp->appContext, milliseconds, wxTimerCallback, this);
  return TRUE;
#endif
#ifdef wx_xview
  if (milliseconds > -1)
  {
    long secs = (long)(milliseconds/1000);
    long usecs = (long)(milliseconds - 1000*secs)*1000;

    timerval.it_value.tv_sec = secs;
    timerval.it_interval.tv_sec = secs;

    timerval.it_value.tv_usec = usecs;
    timerval.it_interval.tv_usec = usecs;
  }

  notify_set_itimer_func((Notify_client)this, (Notify_func)wxTimerFunc, ITIMER_REAL,
                         &timerval, NULL);
  return TRUE;
#endif
#ifdef wx_msw
  if (milliseconds < 0)
    milliseconds = milli;

  if (milliseconds < 0)
    return FALSE;

  milli = milliseconds;

  wxTimerList.DeleteObject(this);
  TIMERPROC wxTimerProcInst = (TIMERPROC) MakeProcInstance((FARPROC)wxTimerProc,
                                          wxhInstance);

  id = SetTimer(NULL, (UINT)(id ? id : 1), (UINT)milliseconds, wxTimerProcInst);
  if (id > 0)
  {
    wxTimerList.Append(id, this);
    return TRUE;
  }
  else return FALSE;
#endif
}

void wxTimer::Stop(void)
{
#ifdef wx_motif
  if (timerId > 0)
  {
    XtRemoveTimeOut(timerId);
    timerId = 0;
  }
#endif
#ifdef wx_xview
  notify_set_itimer_func((Notify_client)this, NOTIFY_FUNC_NULL, ITIMER_REAL,
                         NULL, NULL);
#endif
#ifdef wx_msw
  KillTimer(NULL, (UINT)id);
#endif
}

// Override me!
void wxTimer::Notify(void)
{
}

#ifdef wx_xview
Notify_value wxTimerFunc(Notify_client client, int which)
{
  wxTimer *timer = (wxTimer *)client;
  timer->Notify();
  return NOTIFY_DONE;
}
#endif

#ifdef wx_msw
UINT FAR PASCAL _export wxTimerProc(HWND hwnd, WORD, int idTimer, DWORD)

{
  wxNode *node = wxTimerList.Find((long)idTimer);
  if (node)
  {
    wxTimer *timer = (wxTimer *)node->Data();
    timer->Notify();
  }
  return 0;
}

#endif

/*
 * Timer functions
 *
 */

long wxStartTime = 0;
void wxStartTimer(void)
{
  struct timeb tp;
  ftime(&tp);
  wxStartTime = 1000*tp.time + tp.millitm;
}

// Returns elapsed time in milliseconds
long wxGetElapsedTime(void)
{
  struct timeb tp;
  ftime(&tp);
  long oldTime = wxStartTime;
  wxStartTime = 1000*tp.time + tp.millitm;
  return wxStartTime - oldTime;
}
