/*
 * tex2help.h
 *
 * Read in a Latex file (subset) and write it out in
 * Help format
 *
 */

#include <stdio.h>
#include "wx.h"
#include "wx_utils.h"
#include "wx_list.h"

/*
 * We have a list of macro definitions which we must define
 * in advance to enable the parsing to recognize macros.
 */

class TexMacroDef: public wxObject
{
 public:
  int no_args;
  char *name;
  Bool ignore;

  TexMacroDef(char *the_name, int n, Bool ig);
  ~TexMacroDef(void);
};

#define CHUNK_TYPE_MACRO    1
#define CHUNK_TYPE_ARG      2
#define CHUNK_TYPE_STRING   3

/*
 We have nested lists to represent the Tex document.
 Each element of a list of chunks can be one of:
  - a plain string
  - a macro with/without arguments. Arguments are lists of TexChunks.

Example (\toplevel is implicit but made explicit here):

AddMacroDef("mymat", 2);

\toplevel{The cat sat on the \mymat{very coarse and {\it cheap}}{mat}}.

Parsed as:

TexChunk: type = macro, name = toplevel, no_args = 1
  Children:

    TexChunk: type = argument

      Children:
        TexChunk: type = string, value = "The cat sat on the "
        TexChunk: type = macro, name = mymat, no_args = 2

          Children:
            TexChunk: type = argument

              Children:
                TexChunk: type = string, value = "very coarse and "
                TexChunk: type = macro, name = it, no_args = 1

                  Children:
                    TexChunk: type = argument

                      Children:
                        TexChunk: type = string, value = "cheap"

            TexChunk: type = argument

              Children:
                TexChunk: type = string, value = mat
 */

class TexChunk: public wxObject
{
 public:
  int type;
  char *name;
  char *value;
  int no_args;
  int argn;
  wxList children;
  TexChunk(int the_type);
  ~TexChunk(void);
};

Bool read_a_line(FILE *fd, char *buf);
Bool LoadFile(char *filename);
int ParseArg(wxList& children, char *buffer, int pos, char *environment = NULL);
int ParseMacroBody(char *macro_name, wxList& children, int no_args,
                   char *buffer, int pos, char *environment = NULL);
void TraverseDocument(TexChunk *chunk);
void SetCurrentOutput(FILE *fd);
void SetCurrentOutputs(FILE *fd1, FILE *fd2);
void AddMacroDef(char *name, int n, Bool ignore = FALSE);
void Initialize(void);
void Output(char *s);

/*
 * Client-defined
 *
 */

// Called on start/end of macro examination
void OnMacro(char *name, int no_args, Bool start);

// Called on start/end of argument examination
void OnArgument(char *macro_name, int arg_no, Bool start);
