/* Copyright (C) 1993 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevxxf.c */
/* External font (xfont) implementation for X11. */
#include "math_.h"
#include "memory_.h"
#include "x_.h"
#include "gx.h"
#include "gxdevice.h"
#include "gdevx.h"
#include "gsutil.h"
#include "gserrors.h"
#include "gxxfont.h"

extern gx_device_X gs_x11_device;

extern const byte gs_map_std_to_iso[256];
extern const byte gs_map_iso_to_std[256];

/* Declare the xfont procedures */
private xfont_proc_lookup_font(x_lookup_font);
private xfont_proc_char_xglyph(x_char_xglyph);
private xfont_proc_char_metrics(x_char_metrics);
private xfont_proc_render_char(x_render_char);
private xfont_proc_release(x_release);
private gx_xfont_procs x_xfont_procs = {
	x_lookup_font,
	x_char_xglyph,
	x_char_metrics,
	x_render_char,
	x_release
};

/* Return the xfont procedure record. */
gx_xfont_procs *
x_get_xfont_procs(gx_device *dev)
{
    return &x_xfont_procs;
}

/* Define a X11 xfont. */
typedef struct x_xfont_s x_xfont;
  struct x_xfont_s {
      gx_xfont_common common;
      gx_device_X *xdev;
      XFontStruct *font;
      int encoding_index;
  };

  typedef struct {
      const char *ps_name;
      const char *x11_name;
      char **stdnames;
      char **isonames;
      int stdcount, isocount;
  }
fontencoding;

private fontencoding regular_fonts[] =
{
    {"Courier",
     "-Adobe-Courier-Medium-R-Normal--",
     NULL, NULL, -1, -1},
    {"Courier-Oblique",
     "-Adobe-Courier-Medium-O-Normal--",
     NULL, NULL, -1, -1},
    {"Courier-Bold",
     "-Adobe-Courier-Bold-R-Normal--",
     NULL, NULL, -1, -1},
    {"Courier-BoldOblique",
     "-Adobe-Courier-Bold-O-Normal--",
     NULL, NULL, -1, -1},
    {"Helvetica",
     "-Adobe-Helvetica-Medium-R-Normal--",
     NULL, NULL, -1, -1},
    {"Helvetica-Oblique",
     "-Adobe-Helvetica-Medium-O-Normal--",
     NULL, NULL, -1, -1},
    {"Helvetica-Bold",
     "-Adobe-Helvetica-Bold-R-Normal--",
     NULL, NULL, -1, -1},
    {"Helvetica-BoldOblique",
     "-Adobe-Helvetica-Bold-O-Normal--",
     NULL, NULL, -1, -1},
    {"NewCenturySchlbk-Roman",
     "-Adobe-New Century Schoolbook-Medium-R-Normal--",
     NULL, NULL, -1, -1},
    {"NewCenturySchlbk-Italic",
     "-Adobe-New Century Schoolbook-Medium-I-Normal--",
     NULL, NULL, -1, -1},
    {"NewCenturySchlbk-Bold",
     "-Adobe-New Century Schoolbook-Bold-R-Normal--",
     NULL, NULL, -1, -1},
    {"NewCenturySchlbk-BoldItalic",
     "-Adobe-New Century Schoolbook-Bold-I-Normal--",
     NULL, NULL, -1, -1},
    {"Times-Roman",
     "-Adobe-Times-Medium-R-Normal--",
     NULL, NULL, -1, -1},
    {"Times-Italic",
     "-Adobe-Times-Medium-I-Normal--",
     NULL, NULL, -1, -1},
    {"Times-Bold",
     "-Adobe-Times-Bold-R-Normal--",
     NULL, NULL, -1, -1},
    {"Times-BoldItalic",
     "-Adobe-Times-Bold-I-Normal--",
     NULL, NULL, -1, -1}
};

private fontencoding symbol_fonts[] =
{
    {"Symbol",
     "-Adobe-Symbol-Medium-R-Normal--",
     NULL, NULL, -1, -1}
};

/* Look up a font. */
private gx_xfont *
x_lookup_font(gx_device *dev, const byte *fname, uint len,
	      int encoding_index, const gs_uid *puid, const gs_matrix *pmat,
	      const gs_memory_procs *mprocs)
{
    gx_device_X *xdev = (gx_device_X *) dev;
    x_xfont *xxf;
    float height;
    char x11template[100];
    char *x11fontname;
    const char *psfontname;
    XFontStruct *x11font;
    int i, j;
    float diff, d;
    int size;

    if (!xdev->useXFonts) return NULL;
    /* Only handle simple cases for now. */
    if (pmat->xy != 0 || pmat->yx != 0 || pmat->xx <= 0 || pmat->yy >= 0 ||
	fabs(pmat->yy + pmat->xx) > 0.00002)
	return NULL;

    height = pmat->yy * -1000;
    if (encoding_index < 2) {
	int tried_other_encoding = 0;

	for (i = 0; i < countof(regular_fonts); i++) {
	    if (len == strlen(regular_fonts[i].ps_name) &&
		strncmp(regular_fonts[i].ps_name, (char *)fname, len) == 0)
		break;
	}
	if (i == countof(regular_fonts))
	    return NULL;
	psfontname = regular_fonts[i].ps_name;
	while (True) {
	    if (encoding_index == 0) {
		if (regular_fonts[i].stdcount == -1) {
		    sprintf(x11template, "%s%s", regular_fonts[i].x11_name,
			    "*-*-*-*-*-*-Adobe-fontspecific");
		    regular_fonts[i].stdnames =
			XListFonts(xdev->dpy, x11template, 32,
				   &regular_fonts[i].stdcount);
		}
		if (regular_fonts[i].stdcount) {
		    diff = 999.9;
		    for (j = 0; j < regular_fonts[i].stdcount; j++) {
			char *szp = regular_fonts[i].stdnames[j] +
			strlen(regular_fonts[i].x11_name);

			size = 0;
			while (*szp >= '0' && *szp <= '9')
			    size = size * 10 + *szp++ - '0';
			if (size == 0) continue;
			d = fabs(height - size) / height;
			if (d < diff) {
			    x11fontname = regular_fonts[i].stdnames[j];
			    diff = d;
			}
		    }
		    if (diff*100 <= xdev->xFontTolerance) break;
		}
		if (tried_other_encoding) return NULL;
		encoding_index = 1;
		tried_other_encoding = 1;
	    } else if (encoding_index == 1) {
		if (regular_fonts[i].isocount == -1) {
		    sprintf(x11template, "%s%s", regular_fonts[i].x11_name,
			    "*-*-*-*-*-*-ISO8859-1");
		    regular_fonts[i].isonames =
			XListFonts(xdev->dpy, x11template, 32,
				   &regular_fonts[i].isocount);
		}
		if (regular_fonts[i].isocount) {
		    diff = 999.9;
		    for (j = 0; j < regular_fonts[i].isocount; j++) {
			char *szp = regular_fonts[i].isonames[j] +
			strlen(regular_fonts[i].x11_name);

			size = 0;
			while (*szp >= '0' && *szp <= '9')
			    size = size * 10 + *szp++ - '0';
			if (size == 0) continue;
			d = fabs(height - size) / height;
			if (d < diff) {
			    x11fontname = regular_fonts[i].isonames[j];
			    diff = d;
			}
		    }
		    if (diff*100 <= xdev->xFontTolerance) break;
		}
		if (tried_other_encoding) return NULL;
		encoding_index = 0;
		tried_other_encoding = 1;
	    }
	}
    } else if (encoding_index == 2) {
	for (i = 0; i < countof(symbol_fonts); i++) {
	    if (len == strlen(symbol_fonts[i].ps_name) &&
		strncmp(symbol_fonts[i].ps_name, (char *)fname, len) == 0)
		break;
	}
	if (i == countof(symbol_fonts))
	    return NULL;
	psfontname = symbol_fonts[i].ps_name;
	if (symbol_fonts[i].stdcount == -1) {
	    sprintf(x11template, "%s%s", symbol_fonts[i].x11_name,
		    "*-*-*-*-*-*-Adobe-fontspecific");
	    symbol_fonts[i].stdnames =
		XListFonts(xdev->dpy, x11template, 32,
			   &symbol_fonts[i].stdcount);
	}
	if (symbol_fonts[i].stdcount) {
	    diff = 999.9;
	    for (j = 0; j < symbol_fonts[i].stdcount; j++) {
		char *szp = symbol_fonts[i].stdnames[j] +
		strlen(symbol_fonts[i].x11_name);

		size = 0;
		while (*szp >= '0' && *szp <= '9')
		    size = size * 10 + *szp++ - '0';
		if (size == 0) continue;
		d = fabs(height - size) / height;
		if (d < diff) {
		    x11fontname = symbol_fonts[i].stdnames[j];
		    diff = d;
		}
	    }
	    if (diff*100 > xdev->xFontTolerance) return NULL;
	} else {
	    return NULL;
	}
    } else {
	return NULL;
    }

    x11font = XLoadQueryFont(xdev->dpy, x11fontname);
    if (x11font == NULL)
	return NULL;
    /* Don't bother with 16-bit or 2 byte fonts yet */
    if (x11font->min_byte1 || x11font->max_byte1) {
	XFreeFont(xdev->dpy, x11font);
	return NULL;
    }
    xxf = (x_xfont *) (*mprocs->alloc) (1, sizeof(x_xfont), "x_lookup_font");
    if (xxf == NULL)
	return NULL;
    xxf->common.procs = &x_xfont_procs;
    xxf->xdev = xdev;
    xxf->font = x11font;
    xxf->encoding_index = encoding_index;
    if (xdev->logXFonts) {
	fprintf(stdout, "Using %s\n", x11fontname);
	fprintf(stdout, "  for %s at %g pixels.\n", psfontname, height);
    }
    return (gx_xfont *) xxf;
}

/* Convert a character name or index to an xglyph code. */
private gx_xglyph
x_char_xglyph(gx_xfont *xf, gs_char chr, int encoding_index,
	      gs_glyph glyph, gs_proc_glyph_name_t glyph_name_proc)
{
    x_xfont *xxf = (x_xfont *) xf;

    if (chr == gs_no_char)
	return gx_no_xglyph;	/* can't look up names yet */
    if (encoding_index != xxf->encoding_index) {
	if (encoding_index == 0 && xxf->encoding_index == 1)
	    chr = gs_map_std_to_iso[chr];
	else if (encoding_index == 1 && xxf->encoding_index == 0)
	    chr = gs_map_iso_to_std[chr];
	else
	    return gx_no_xglyph;
    }
    if (chr < xxf->font->min_char_or_byte2 ||
	chr > xxf->font->max_char_or_byte2)
	return gx_no_xglyph;
    if (xxf->font->per_char) {
	int i = chr - xxf->font->min_char_or_byte2;
	XCharStruct *xc = &(xxf->font->per_char[i]);

	if ((xc->lbearing == 0) && (xc->rbearing == 0) &&
	    (xc->ascent == 0) && (xc->descent == 0))
	    return gx_no_xglyph;
    }
    return (gx_xglyph) chr;
}

/* Get the metrics for a character. */
private int
x_char_metrics(gx_xfont *xf, gx_xglyph xg, int wmode,
	       gs_int_point *pwidth, gs_int_rect *pbbox)
{
    x_xfont *xxf = (x_xfont *) xf;

    if (wmode != 0)
	return gs_error_undefined;
    if (xxf->font->per_char == NULL) {
	pwidth->x = xxf->font->max_bounds.width;
	pwidth->y = 0;
	pbbox->p.x = xxf->font->max_bounds.lbearing;
	pbbox->q.x = xxf->font->max_bounds.rbearing;
	pbbox->p.y = -xxf->font->max_bounds.ascent;
	pbbox->q.y = xxf->font->max_bounds.descent;
    } else {
	int i = xg - xxf->font->min_char_or_byte2;

	pwidth->x = xxf->font->per_char[i].width;
	pwidth->y = 0;
	pbbox->p.x = xxf->font->per_char[i].lbearing;
	pbbox->q.x = xxf->font->per_char[i].rbearing;
	pbbox->p.y = -xxf->font->per_char[i].ascent;
	pbbox->q.y = xxf->font->per_char[i].descent;
    }
    return 0;
}

/* Render a character. */
private int
x_render_char(gx_xfont *xf, gx_xglyph xg, gx_device *dev,
	      int xo, int yo, gx_color_index color, int required)
{
    x_xfont *xxf = (x_xfont *) xf;
    gx_device_X *xdev = xxf->xdev;
    char chr = (char)xg;
    gs_int_point wxy;
    gs_int_rect bbox;
    int x, y, w, h;
    int code;

    if (dev->dname == gs_x11_device.dname) {
	code = (*xf->common.procs->char_metrics) (xf, xg, 0, &wxy, &bbox);
	if (code < 0) return code;
	set_fill_style(FillSolid);
	set_fore_color(color);
	set_function(GXcopy);
	set_font(xxf->font->fid);
	XDrawString(xdev->dpy, xdev->dest, xdev->gc, xo, yo, &chr, 1);
	if (xdev->bpixmap != (Pixmap) 0) {
	    x = xo + bbox.p.x;
	    y = yo + bbox.p.y;
	    w = bbox.q.x - bbox.p.x;
	    h = bbox.q.y - bbox.p.y;
	    fit_fill(dev, x, y, w, h);
	    x_update_add(dev, x, y, w, h);
	}
	return 0;
    } else if (!required)
	return -1;	/* too hard */
    else {
	/* Display on an intermediate bitmap, then copy the bits. */
	int wbm, raster;
	int i;
	XImage *xim;
	Pixmap xpm;
	GC fgc;
	byte *bits;

	dev_proc_copy_mono((*copy_mono)) = dev->procs->copy_mono;

	code = (*xf->common.procs->char_metrics) (xf, xg, 0, &wxy, &bbox);
	if (code < 0) return code;
	w = bbox.q.x - bbox.p.x;
	h = bbox.q.y - bbox.p.y;
	wbm = round_up(w, align_bitmap_mod * 8);
	raster = wbm >> 3;
	bits = (byte *) gs_malloc(h, raster, "x_render_char");
	if (bits == 0) return gs_error_limitcheck;
	xpm = XCreatePixmap(xdev->dpy, xdev->win, w, h, 1);
	fgc = XCreateGC(xdev->dpy, xpm, NULL, 0);
	XSetForeground(xdev->dpy, fgc, 0);
	XFillRectangle(xdev->dpy, xpm, fgc, 0, 0, w, h);
	XSetForeground(xdev->dpy, fgc, 1);
	XSetFont(xdev->dpy, fgc, xxf->font->fid);
	XDrawString(xdev->dpy, xpm, fgc, -bbox.p.x, -bbox.p.y, &chr, 1);
	xim = XGetImage(xdev->dpy, xpm, 0, 0, w, h, 1, ZPixmap);
	i = 0;
	for (y = 0; y < h; y++) {
	    char b = 0;

	    for (x = 0; x < wbm; x++) {
		b = b << 1;
		if (x < w)
		    b += XGetPixel(xim, x, y);
		if ((x & 7) == 7)
		    bits[i++] = b;
	    }
	}
	code = (*copy_mono) (dev, bits, 0, raster, gx_no_bitmap_id,
			     xo + bbox.p.x, yo + bbox.p.y, w, h,
			     gx_no_color_index, color);
	gs_free((char *)bits, h, raster, "x_render_char");
	XFreePixmap(xdev->dpy, xpm);
	XFreeGC(xdev->dpy, fgc);
	XDestroyImage(xim);
	return (code < 0 ? code : 0);
    }
}

/* Release an xfont. */
private int
x_release(gx_xfont *xf, const gs_memory_procs *mprocs)
{
    x_xfont *xxf = (x_xfont *) xf;
    gx_device_X *xdev = xxf->xdev;

    XFreeFont(xxf->xdev->dpy, xxf->font);
    if (mprocs != NULL)
	(*mprocs->free) ((char *)xf, 1, sizeof(x_xfont),
			 "x_release");
    return 0;
}
