// File: print.c
//
// Purpose: Routines used for printing.
//
// Functions:
//   GetPrinterDC
//   InitPrinting
//   TermPrinting
//   PrintDlgProc
//   AbortProc
//
//
// NOTE: be sure to place these in your def file
//    AbortProc
//    PrintDlgProc
//
//
//****************************************************************************


#include <windows.h>
#include "print.h"

FARPROC lpfnAbortProc = NULL;
FARPROC lpfnPrintDlgProc = NULL;
HWND hDlgPrint = NULL;
BOOL bError;
BOOL bUserAbort;
char szNull[] = "";


#pragma alloc_text(_PRINT, AbortProc, PrintDlgProc)


//*************************************************************
//
//  GetPrinterDC()
//
//  Purpose:
//	 Get a handle to a printer device context.	
//
//
//  Parameters:
//
//      
//      
//
//  Return: (HDC)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//
//
//*************************************************************

HDC PASCAL GetPrinterDC()
{
   char	msgbuf[128];
   LPSTR	 pch;
   LPSTR	 pchFile;
   LPSTR	 pchPort;
   
   // Get current printer from win.ini
   if (!GetProfileString("windows", "device", szNull, msgbuf, sizeof(msgbuf)))
      return NULL;
   
   // parse string to get driver name, device name, and port
   for (pch = msgbuf; *pch && *pch != ','; pch = AnsiNext(pch)) ;
   
   if (*pch)
      *pch++ = 0;
   
   /* Skip tabs, control chars */
   while (*pch && *pch <= ' ')
      pch=AnsiNext(pch);
   
   pchFile = pch;
   while (*pch && *pch != ',' && *pch > ' ')
      pch = AnsiNext(pch);
   
   if (*pch)
      *pch++ = 0;
   
   while (*pch && (*pch <= ' ' || *pch == ','))
      pch = AnsiNext(pch);
   
   pchPort = pch;
   while (*pch && *pch > ' ')
      pch = AnsiNext(pch);
   
   *pch = 0;
   
   return CreateDC(pchFile, msgbuf, pchPort, NULL);
}



//*************************************************************
//
//  InitPrinting()
//
//  Purpose:
//	 Get ready to print. Set up the printing dialog procedure and the 
//  abort procedure and then start the print job.
//
//
//  Parameters:
//
//      HDC hdc
//      HWND hWnd
//      LPSTR msg
//      
//
//  Return: (BOOL PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//
//
//*************************************************************

BOOL PASCAL InitPrinting(HDC hdc, HWND hWnd, LPSTR msg)
{
   HANDLE hInst = GetWindowWord(hWnd, GWW_HINSTANCE);
         
   bError = FALSE;		/* no errors yet */
   bUserAbort = FALSE;	/* user hasn't aborted */
   
   hWnd = hWnd;	/* save for Enable at Term time */
   
   lpfnPrintDlgProc = MakeProcInstance(PrintDlgProc, hInst);
   lpfnAbortProc    = MakeProcInstance(AbortProc, hInst);
   
   hDlgPrint = CreateDialogParam(hInst, "PRTDLG", hWnd, lpfnPrintDlgProc, NULL);
   
   if (!hDlgPrint)
      return FALSE;
   
   SetWindowText(hDlgPrint, msg);
   
   EnableWindow(hWnd, FALSE);	/* disable parent */
   
   if ((Escape(hdc, SETABORTPROC, 0, (LPSTR)lpfnAbortProc, NULL) > 0) && 
      (Escape(hdc, STARTDOC, lstrlen(msg), msg, NULL) > 0))
      bError = FALSE;
   else
      bError = TRUE;
   
   return !bError;
}


//*************************************************************
//
//  TermPrinting()
//
//  Purpose:
//	 Clean-up after printing.	
//
//
//  Parameters:
//
//      HDC hdc
//      HWND hWnd
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//
//
//*************************************************************

void PASCAL TermPrinting(HDC hdc, HWND hWnd)
{
   if (!bError)
   {
   	Escape(hdc, ENDDOC, 0, NULL, NULL);
   }
   
   if (bUserAbort)
   {		/* already been done? */
   	Escape (hdc, ABORTDOC, 0, NULL, NULL) ;
   }
   else
   {
   	EnableWindow(hWnd, TRUE);/* turn parent back on */
   	DestroyWindow(hDlgPrint);/* toast out dialog */
   }

   FreeProcInstance(lpfnAbortProc);
   FreeProcInstance(lpfnPrintDlgProc);
}





//*************************************************************
//
//  PrintDlgProc()
//
//  Purpose:
//	 Printing dialog procedure. Dialog Box has a cancel button on it
//  that allows you to abort printing.
//
//
//  Parameters:
//
//      HWND hDlg
//      unsigned iMessage
//      WORD wParam
//      DWORD lParam
//      
//  Globals used:
//  szAppName	   name of the app for dialog caption
//  bUserAbort	   sets this when the user hits cancel
//  hDlgPrint	
//
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//
//
//*************************************************************

BOOL FAR PASCAL _export PrintDlgProc(HWND hDlg, unsigned iMessage, WORD wParam, DWORD lParam)
{
   switch (iMessage)
   {
      case WM_INITDIALOG:

      EnableMenuItem(GetSystemMenu(hDlg, FALSE), SC_CLOSE, MF_GRAYED);
      break;

      case WM_COMMAND:
         bUserAbort = TRUE;
         EnableWindow(GetParent(hDlg), TRUE);
         DestroyWindow(hDlg);
         hDlgPrint = 0;
         break;

      default:
         return FALSE;
   }
   return TRUE;
}






//*************************************************************
//
//  AbortProc()
//
//  Purpose:
//	 Printing abort procedure.	
//
//
//  Parameters:
//
//      HDC hPrnDC
//      short nCode
// 
// Globals used:
// bUserAbort 	   Indicates the user hit CANCEL on the print dialog
// hDlgPrint	   Handle of the print dialog
//
//     
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//
//
//*************************************************************

BOOL FAR PASCAL _export AbortProc(HDC hPrnDC, short nCode)
{
   MSG   msg;

   while (!bUserAbort && PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
   {
      if (!hDlgPrint || !IsDialogMessage(hDlgPrint, &msg))
      {
         TranslateMessage (&msg);
         DispatchMessage (&msg);
      }
   }
   return !bUserAbort;
}
