// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
/*
    appexec.c

    A sample application to demonstrate the launching 
    of other applications.

    Another fine Herman Rodent production

*/

#include "appexec.h"

//
// global data
//

char *szAppName = "AppExec";        // App name
HINSTANCE ghInstance;               // app instance
HWND ghwndMain;                     // main window handle
HWND hwndEdit;                      // edit control window handle
HWND hwndOK;                        // OK button window handle
WNDPROC lpOldEditProc;
BOOL bWaitForExit = FALSE;          // wait for spawned app to exit
BOOL bWaitingForAppExit;            // we are waiting if set.
UINT uiWaitTimer;                   // wait timer id
EXECAPPINFO ExecAppInfo;            // info from running an app

//
// local functions
//

static void Command(HWND hWnd, WPARAM wParam, LPARAM lParam);
BOOL InitFirstInstance(HANDLE hInstance);
BOOL InitCurrentInstance(HANDLE hInstance, int cmdShow);
LRESULT CALLBACK EditSubProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
void cdecl Error(LPSTR lpFormat, ...);

//
// Entry point
//

int PASCAL WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPSTR lpszCmdLine,
                   int cmdShow)
{
    MSG msg;

    ghInstance = hInstance;

    //
    // We allow multiple instances
    //

    if (!hPrevInstance) {

        if (!InitFirstInstance(hInstance)) {
            return 1;
        }
    }

    //
    // Do the per instance initialization
    //

    if (!InitCurrentInstance(hInstance, cmdShow)) {
        return 1;
    }

    //
    // Check for messages from Windows and process them.
    // 

    while (GetMessage(&msg, NULL, 0, 0)) {

        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    return (msg.wParam);
}
    
//
// Main window message handler
// If the bWaitForAppExit flag is set, we are waiting for a 
// spawned child to exit and only process WM_TIMER msgs.
//

LRESULT CALLBACK MainWndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    UINT uiResult;

    if (bWaitingForAppExit) {
        switch (msg) {
        case WM_TIMER:

            //
            // Test if the app is still running
            //

            if (!IsAppRunning(&ExecAppInfo)) {

                //
                // Reset the state and stop the timer
                //

                KillTimer(hWnd, uiWaitTimer);
                bWaitingForAppExit = FALSE;

                //
                // Reshow the main window
                //

                ShowWindow(hWnd, SW_SHOW);
            }

            break;

        default:
            return DefWindowProc(hWnd, msg, wParam, lParam);
            break;
        }
        return NULL;
    }

    switch(msg) {
    case WM_CREATE:
        break;

    case WM_SIZE:
        {
            int w;

            w = LOWORD(lParam) - GetSystemMetrics(SM_CXVSCROLL) * 3;

            SetWindowPos(hwndEdit, 
                         NULL,
                         0, 0,
                         w, HIWORD(lParam),
                         SWP_NOZORDER | SWP_NOACTIVATE);
            SetWindowPos(hwndOK, 
                         NULL,
                         w, 0,
                         LOWORD(lParam) - w, HIWORD(lParam),
                         SWP_NOZORDER | SWP_NOACTIVATE);
        }
        break;

    case WM_PAINT:
        BeginPaint(hWnd, &ps);
        EndPaint(hWnd, &ps);
        break;

    case WM_SETFOCUS:
        SetFocus(hwndEdit);
        break;


    case WM_INITMENUPOPUP:

        //
        // Show what options are currently selected
        //

        CheckMenuItem((HMENU)wParam,
                      IDM_WAIT,
                      bWaitForExit ? MF_BYCOMMAND | MF_CHECKED
                                   : MF_BYCOMMAND | MF_UNCHECKED);
        break;

    case WM_COMMAND:
        switch(wParam) {
        case IDC_OK:

            //
            // Get the text in the edit field and try to
            // run the app by that name
            //

            {
                char buf[_MAX_PATH];

                buf[0] = '\0'; // start with an empty string
                SendMessage(hwndEdit,
                            WM_GETTEXT,
                            (WPARAM)sizeof(buf),
                            (LPARAM)(LPSTR)buf);

                if (lstrlen(buf)) {

                    //
                    // Exec the new app
                    //

                    uiResult = ExecApp(hWnd, buf, "", &ExecAppInfo);

                    switch (uiResult) {
                    case RAN_OK:
                        if (bWaitForExit) {

                            //
                            // Set a state flag to say we are
                            // in limbo until the spawned app exits.
                            //

                            bWaitingForAppExit = TRUE;

                            //
                            // Hide the main window (just for fun)
                            // You remember fun don't you?
                            //

                            ShowWindow(hWnd, SW_HIDE);

                            //
                            // Start a timer so we can test if the app
                            // is still running occasionally
                            //

                            uiWaitTimer = SetTimer(hWnd, 1, 200, NULL);
                        }
                        break;

                    case NOT_FOUND:

                        Error("'%s' could not be found.", (LPSTR)buf);
                        break;

                    case INVALID_NAME:

                        Error("'%s' is not a valid application name",
                             (LPSTR)buf);
                        break;

                    default:

                        Error("'%s' could not be run because an error occured",
                              (LPSTR)buf);
                        break;
                    }

                }
            }
            break;

        case IDM_EXIT:
            PostMessage(hWnd, WM_CLOSE, 0, 0l);
            break;

        case IDM_WAIT:
            bWaitForExit = !bWaitForExit;
            break;

        default:
            break;
        }
        break;

    case WM_DESTROY:

        //
        // Unsubclass the edit control
        //

        if (lpOldEditProc) {
            SetWindowLong(hwndEdit, GWL_WNDPROC, (LONG)lpOldEditProc);
            lpOldEditProc = NULL;
        }

        //
        // ... in the castle Aaaaaaaaargh..  (Name that movie)
        //

        PostQuitMessage(0);
        break;

    default:
        return DefWindowProc(hWnd, msg, wParam, lParam);
        break;
    }
    return NULL;
}

//
// Show an error message box
//

void cdecl Error(LPSTR lpFormat, ...) 
{
    char buf[256];

    wvsprintf(buf, lpFormat, (LPSTR)(&lpFormat+1));
    MessageBeep(MB_ICONEXCLAMATION);
    MessageBox(ghwndMain, buf, szAppName, MB_OK|MB_ICONEXCLAMATION);
}

//
// Initialise the first instance
//

BOOL InitFirstInstance(HANDLE hInstance)
{
    WNDCLASS wc;

    //
    // define the class of window we want to register
    //

    wc.lpszClassName    = szAppName;
    wc.style            = CS_HREDRAW | CS_VREDRAW;
    wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
    wc.hIcon            = LoadIcon(hInstance,"Icon");
    wc.lpszMenuName     = "Menu";
    wc.hbrBackground    = GetStockObject(WHITE_BRUSH);
    wc.hInstance        = hInstance;
    wc.lpfnWndProc      = MainWndProc;
    wc.cbClsExtra       = 0;
    wc.cbWndExtra       = 0;
    
    return RegisterClass(&wc);
}

//
// Init the current instance
//

BOOL InitCurrentInstance(HANDLE hInstance, int cmdShow)
{
    //
    // Create the main window
    //

    ghwndMain = CreateWindow(szAppName,
                            szAppName,
                            WS_OVERLAPPED | WS_SYSMENU | WS_BORDER
                                | WS_CAPTION | WS_VISIBLE,
                            CW_USEDEFAULT,
                            0,
                            GetSystemMetrics(SM_CXSCREEN) / 2,
                            GetSystemMetrics(SM_CYBORDER) * 2 
                             + GetSystemMetrics(SM_CYCAPTION)
                             + GetSystemMetrics(SM_CYMENU) * 5 / 2,
                            (HWND)NULL,
                            (HMENU)NULL,
                            ghInstance,
                            (LPSTR)NULL
                            );
    
    if (!ghwndMain) {
        return FALSE;
    }

    //
    // Create the edit control window
    //

    hwndEdit = CreateWindow("Edit",
                            "",
                            WS_CHILD | WS_VISIBLE | WS_BORDER
                             | ES_AUTOHSCROLL,
                            0,
                            0,
                            0,
                            0,
                            ghwndMain,
                            (HMENU)IDC_EDIT,
                            ghInstance,
                            (LPSTR)NULL
                            );

    if (hwndEdit) {

        //
        // Subclass it so we can get the enter key
        //

        lpOldEditProc = (WNDPROC) GetWindowLong(hwndEdit, GWL_WNDPROC);
        SetWindowLong(hwndEdit, GWL_WNDPROC, (LONG)EditSubProc);
    }

    //
    // Create the OK button
    //

    hwndOK = CreateWindow("Button",
                            "OK",
                            WS_CHILD | WS_VISIBLE
                             | BS_DEFPUSHBUTTON,
                            0,
                            0,
                            0,
                            0,
                            ghwndMain,
                            (HMENU)IDC_OK,
                            ghInstance,
                            (LPSTR)NULL
                            );


    ShowWindow(ghwndMain, cmdShow);
    UpdateWindow(ghwndMain);

    return TRUE;
}

//
// Subclass proc to get the return key from the edit control
//

LRESULT CALLBACK EditSubProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
    case WM_CHAR:
        if (wParam == 0x0D) {
            PostMessage(GetParent(hWnd), WM_COMMAND, IDC_OK, 0);
            return 0;
        }
        break;

    default:
        break;
    }

    if (lpOldEditProc) {
        return CallWindowProc(lpOldEditProc, hWnd, msg, wParam, lParam);
    }
    return 0;
}

