// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
/*
    exec.c

    This module contains the ExecApp and IsAppRunning functions

    It relies on the global ghInstance;

*/

#include "appexec.h"

//
// define a procedure to hook the common dialogs.
//

typedef UINT (FAR PASCAL *DLGHOOKPROC)(HWND, UINT , WPARAM, LPARAM);

//
// local structure for enum windows call
//

typedef struct _ENUMINFO {
    HINSTANCE hInstance;        // supplied on input
    HWND hWnd;                  // filled in by enum func
} ENUMINFO, FAR *LPENUMINFO;

//
// local data
//

static char *szProgramsSection = "programs"; // WIN.INI section
static char *szFound = "(found)";
static char *szNotFound = "(not found)";

//
// local functions
//

static BOOL BrowseForApp(HWND hwndParent, LPSTR pszPath, LPSTR pszNewPath);
static int TryWinExec(LPSTR pszPath, LPSTR pszParams, LPEXECAPPINFO pInfo);
BOOL CALLBACK EnumWndProc(HWND hWnd, LPARAM lParam);

//
// Run a named application
//

UINT ExecApp(HWND hwndParent, 
             LPSTR pszPath, 
             LPSTR pszParams,
             LPEXECAPPINFO pInfo)
{
    char szPath[_MAX_PATH];
    char szNewPath[_MAX_PATH];
    char szDrive[_MAX_DRIVE];
    char szDir[_MAX_DIR];
    char szFname[_MAX_FNAME];
    char szExt[_MAX_EXT];
    int i;

    //
    // Copy the path and convert to upper case
    //

    _fstrcpy(szPath, pszPath);
    _fstrupr(szPath);

    //
    // Split up the components of the path
    //

    _splitpath(szPath, szDrive, szDir, szFname, szExt);

    //
    // See if it already has an extension
    // and if not, provide .EXE
    //

    if (!_fstrlen(szExt)) {

        _fstrcpy(szExt, ".EXE");

    } else {

        //
        // Make sure it's EXE and not something bogus
        //

        if (!_fstricmp(szExt, ".EXE")) {

            return INVALID_NAME;
        }
    }

    //
    // Try and run it.
    //

    _makepath(szNewPath, szDrive, szDir, szFname, szExt);

    i = TryWinExec(szNewPath, pszParams, pInfo);

    if ((i == RAN_OK) || (i == SOME_ERROR)) {
        return i;
    }

    //
    // It couldn't be found because it either wasn't on the search
    // path or because the path supplied was invalid.
    // See if we have a WIN.INI entry for it.
    //

    _makepath(szPath, "", "", szFname, szExt);
    i = GetProfileString(szProgramsSection,     // [programs]
                         szPath,                // <app>.EXE
                         "",                    // default
                         szNewPath,             // result
                         sizeof(szNewPath));

    //
    // If we got a result, try executing that
    //

    if (i != 0) {

        i = TryWinExec(szNewPath, pszParams, pInfo);

        if ((i == RAN_OK) || (i == SOME_ERROR)) {
            return i;
        }
    }

    //
    // If we get here, either it couldn't be found or the
    // existing WIN.INI entry is wrong or is not there.
    // Bring up the browse dialog to see if the user can
    // find it for us.
    //

    _makepath(szPath, "", "", szFname, szExt);
    if (!BrowseForApp(hwndParent, szPath, szNewPath)) {
        return NOT_FOUND;
    }

    //
    // We found it so make a new entry in WIN.INI
    // and then execute it.
    //

    WriteProfileString(szProgramsSection,
                       szPath,
                       szNewPath);

    return TryWinExec(szNewPath, pszParams, pInfo);
}

//
// Common dialog hook function for the browse dialog.  
// All this does is test to see if the file exists in the current
// file list (which is not visible in this dialog) and changes the
// found/not found text accordingly.  It also enables the OK button
// when the file has been found.  This is more or less the same
// dialog that the File Manager uses.
//

UINT FAR PASCAL CommDlgHookProc(HWND hDlg, 
                                WORD iMessage, 
                                WORD wParam, 
                                LONG lParam)
{
    char szTemp[40];
    HICON hIcon;

    switch (iMessage) {
    case WM_INITDIALOG:

        PostMessage(hDlg, WM_COMMAND, ctlLast+1, 0L);
        break;

    case WM_COMMAND:

        switch (wParam) {
        case ctlLast+1:

            GetDlgItemText(hDlg, edt1, szTemp, sizeof(szTemp));
            if (SendDlgItemMessage(hDlg, 
                                   lst1, 
                                   LB_FINDSTRING, 
                                   (WPARAM)-1,
                                   (LONG)(LPSTR)szTemp) >= 0) {
                SetDlgItemText(hDlg, ctlLast+2, szFound);
                EnableWindow(GetDlgItem(hDlg, IDOK), TRUE);
                hIcon = LoadIcon(ghInstance, "Happy");
            } else {
                SetDlgItemText(hDlg, ctlLast+2, szNotFound);
                EnableWindow(GetDlgItem(hDlg, IDOK), FALSE);
                hIcon = LoadIcon(ghInstance, "Sad");
            }

            //
            // Set the icon to be happy or sad.
            // Setting the icon control text to the name of
            // the resource won't work here because it will 
            // try to load it from SHELL.DLL
            //

            SendDlgItemMessage(hDlg,
                               ctlLast+3,
                               STM_SETICON,
                               (WPARAM)hIcon,
                               0L);
            break;

        case lst2:

            if (HIWORD(lParam) == LBN_DBLCLK) {
                PostMessage(hDlg, WM_COMMAND, ctlLast+1, 0L);   
            }
            break;

        case cmb2:

            switch (HIWORD(lParam)) {
            case CBN_SELCHANGE:

                PostMessage(hDlg, WM_COMMAND, ctlLast+1, 1L);
                break;

            case CBN_CLOSEUP:
                PostMessage(hDlg, 
                            WM_COMMAND, 
                            cmb2,
                            MAKELONG(GetDlgItem(hDlg, cmb2), CBN_SELCHANGE));
                break;
            }
            break;

        case IDOK:
        case IDCANCEL:
        case IDABORT:

            PostMessage(hDlg, WM_COMMAND, ctlLast+1, 0L);
            break;
        }
        break;
    }

  return FALSE;  // commdlg, do your thing
}

//
// See if the user wants to try browsing for the app.  If they
// do bring up the browse dialog.
//

static BOOL BrowseForApp(HWND hwndParent, LPSTR pszPath, LPSTR pszNewPath)
{
    int iResult;
    BOOL bResult;
    OPENFILENAME of;

    //
    // Put up a message box asking if we want to go browse for it. 
    // Note that you could do a better dialog box with the option
    // to type the file path in but this keeps the example simpler.
    //

    iResult  = MessageBox(hwndParent,
                          "The application could not be found. Do you "
                            "want to browse for it?",
                          pszPath,
                          MB_YESNO | MB_ICONQUESTION);

    if (iResult != IDYES) {
        return FALSE;
    }

    //
    // Put up a common dialog to browse for the file.
    //

    _fstrcpy(pszNewPath, pszPath);

    of.lStructSize      = sizeof(of);
    of.hwndOwner        = hwndParent;
    of.hInstance        = ghInstance;
    of.lpstrFilter      = "Applications (*.EXE)\0*.EXE\0";
    of.nFilterIndex     = 1;
    of.lpstrCustomFilter= NULL;
    of.nMaxCustFilter   = 0;
    of.lpstrFile        = pszNewPath;
    of.nMaxFile         = _MAX_PATH;
    of.lpstrFileTitle   = NULL;
    of.nMaxFileTitle    = 0;
    of.lpstrInitialDir  = NULL; // use current dir
    of.lpstrTitle       = "Browse for Application";
    of.Flags            = OFN_HIDEREADONLY | OFN_FILEMUSTEXIST
                        | OFN_ENABLETEMPLATE | OFN_ENABLEHOOK;
    of.nFileOffset      = 0;
    of.nFileExtension   = 0;
    of.lpstrDefExt      = NULL;
    of.lCustData        = 0;  
    of.lpfnHook         = 
        (DLGHOOKPROC)MakeProcInstance((FARPROC)CommDlgHookProc, ghInstance);
    of.lpTemplateName   = "DLG_BROWSE";
    
    bResult = GetOpenFileName(&of);
    FreeProcInstance((FARPROC)of.lpfnHook);

    return bResult;
}

//
// Try running an app by calling WinExec.  If it works, return some
// info about the app in the struct pointed to by pInfo.
// It returns RAN_OK, NOT_FOUND or SOME_ERROR.
//

static int TryWinExec(LPSTR pszPath, LPSTR pszParams, LPEXECAPPINFO pInfo)
{
    char szCmdLine[_MAX_PATH + 256];
    UINT uiResult;
    ENUMINFO EnumInfo;
    WNDENUMPROC lpEnumProc;

    _fstrcpy(szCmdLine, pszPath);
    _fstrcat(szCmdLine, " ");
    _fstrcat(szCmdLine, pszParams);
    uiResult = WinExec(szCmdLine, SW_SHOWNORMAL);

    if ((uiResult == 2) || (uiResult == 3)) {
        return NOT_FOUND;
    }

    if (uiResult < 32) {
        return SOME_ERROR;
    }

    //
    // The app is running (or at least it was) so return some info about it
    //

    pInfo->hInstance = (HINSTANCE) uiResult;

    //
    // Find the window handle of the instance by enumerating all
    // the main windows until we find one with the correct
    // instance value
    //

    EnumInfo.hInstance = pInfo->hInstance;
    EnumInfo.hWnd = NULL;

    lpEnumProc = (WNDENUMPROC) MakeProcInstance((FARPROC)EnumWndProc, 
                                                ghInstance);
    EnumWindows(lpEnumProc, (LPARAM)(LPENUMINFO)&EnumInfo);
    FreeProcInstance((FARPROC)lpEnumProc);

    if (!EnumInfo.hWnd) {

        //
        // Didn't find a matching window so the task probably
        // isn't running any more.  Maybe it did what it had to
        // and went away already.
        //

        pInfo->hWnd = NULL;
        pInfo->hTask = NULL;

    } else {

        //
        // We have the window handle. Now get the task.
        //

        pInfo->hWnd = EnumInfo.hWnd;
        pInfo->hTask = GetWindowTask(pInfo->hWnd);
    }

    return RAN_OK;
}

//
// Procedure called by EnumWindows
// Test to see if this window has the instance value we are looking for.
//

BOOL CALLBACK EnumWndProc(HWND hWnd, LPARAM lParam)
{
    HINSTANCE hInstance;
    LPENUMINFO lpInfo;

    //
    // Get a pointer to our info structure
    //

    lpInfo = (LPENUMINFO) lParam;

    //
    // Get the window instance value
    //

    hInstance = (HINSTANCE) GetWindowWord(hWnd, GWW_HINSTANCE);

    //
    // See if it's the one we are looking for
    //

    if (hInstance == lpInfo->hInstance) {

        //
        // Save the window handle
        //

        lpInfo->hWnd = hWnd;

        //
        // Stop enumerating
        //

        return FALSE;
    }

    //
    // Continue enumerating
    //

    return TRUE;
}

//
// Test if an app is still running.  Since window handles,
// instance values and task handles can all be reused, we need to
// test that at least two are the same.  This example tests all three.
//

BOOL IsAppRunning(LPEXECAPPINFO pInfo)
{
    //
    // See if the window is still valid and the task is still active
    //

    if (!IsWindow(pInfo->hWnd) || !IsTask(pInfo->hTask)) {
        return FALSE;
    }

    //
    // See if it's the same instance
    //

    if ((HINSTANCE) GetWindowWord(pInfo->hWnd, 
                                  GWW_HINSTANCE) != pInfo->hInstance) {
        return FALSE;
    }

    //
    // And the same task (just in case)
    //

    if (GetWindowTask(pInfo->hWnd) != pInfo->hTask) {
        return FALSE;
    }

    return TRUE;

}
