/***    DRVINFO.C - IsDoubleSpaceDrive function
 *
 *      Version 1.00.58  12-Mar-1993
 */

#include "drvinfo.h"

#pragma pack(1)

/** DSPACKET - IOCTL packet definition for DoubleSpace API IOCtls
 *
 */
typedef struct DSPACKET_t { /* dsp */
    WORD    stamp;                      // Identifying stamp ('DM')
    BYTE    command;                    // Command ('F' or 'I')
    WORD    result;                     // Result code ('OK', else unmodified)
    BYTE    abPadding[5];               // Padding
} DSPACKET;
#define cbDSPACKET  10                  // sizeof(DSPACKET), for ASM code

#define MD_STAMP    (('D'<<8) + 'M')    // dsp.stamp value

#define DSCMD_FLUSH                 'F' // dsp.command for Flush
#define DSCMD_FLUSH_AND_INVALIDATE  'I' // dsp.command for Flush & Invalidate

#define DSP_NO_ERROR    ('O'<<8 + 'K')  // Success value for dsp.result
#define DSP_ERROR       ('?'<<8 + '?')  // Initial value for dsp.result



/***    IsDoubleSpaceDrive - Get information on a DoubleSpace drive
 *
 *      Entry:
 *          drive     - Drive to test (0=A, 1=B, etc.)
 *                      NOTE: No parameter checking is done on drive.
 *          pdrHost   - Receives drive number of host drive
 *          pfSwapped - Receives TRUE/FALSE indicating if drive is swapped.
 *          pseq      - Receives CVFs sequence number if DoubleSpace drive
 *
 *      Exit:
 *          returns TRUE, if DoubleSpace drive:
 *              *pdrHost   = current drive number of host drive (0=A,...)
 *              *pfSwapped = TRUE, if drive is swapped with host,
 *                           FALSE, if drive is not swapped with host
 *              *pseq      = CVF sequence number (always zero if swapped
 *                             with host drive)
 *
 *                           NOTE: The full file name of the CVF is:
 *                                   *pdrHost:\DBLSPACE.*pseq
 *
 *                               pdrHost  pseq  Full Path
 *                               -------  ----  -----------
 *                                  0       1   a:\dblspace.001
 *                                  3       0   d:\dblspace.000
 *
 *          returns FALSE, if *not* DoubleSpace drive:
 *              *pdrHost   = drive number of host drive at boot time
 *              *pfSwapped = TRUE, if swapped with a DoubleSpace drive
 *                           FALSE, if not swapped with a DoubleSpace drive
 */
BOOL IsDoubleSpaceDrive(BYTE drive, BOOL *pfSwapped, BYTE *pdrHost, int *pseq)
{
    BYTE        seq;
    BYTE        drHost;
    BOOL        fSwapped;
    BOOL        fDoubleSpace;

    // Assume drive is a normal, non-host drive
    drHost = drive;
    fSwapped = FALSE;
    fDoubleSpace = FALSE;
    seq = 0;

    _asm {

        mov     ax,4A11h        ; DBLSPACE.BIN INT 2F number
        mov     bx,1            ; bx = GetDriveMap function
        mov     dl,drive        ;
        int     2Fh             ; (bl AND 80h) == DS drive flag
                                ; (bl AND 7Fh) == host drive

        or      ax,ax           ; Success?
        jnz     gdiExit         ;    NO, DoubleSpace not installed

        test    bl,80h          ; Is the drive compressed?
        jz      gdiHost         ;    NO, could be host drive

        ; We have a DoubleSpace Drive, need to figure out host drive.
        ;
        ; This is tricky because of the manner in which DBLSPACE.BIN
        ; keeps track of drives.
        ;
        ; For a swapped CVF, the current drive number of the host
        ; drive is returned by the first GetDriveMap call.  But for
        ; an unswapped CVF, we must make a second GetDriveMap call
        ; on the "host" drive returned by the first call.  But, to
        ; distinguish between swapped and unswapped CVFs, we must
        ; make both of these calls.  So, we make them, and then check
        ; the results.

        mov     fDoubleSpace,TRUE ; Drive is DS drive
        mov     seq,bh          ; Save sequence number

        and     bl,7Fh          ; bl = "host" drive number
        mov     drHost,bl       ; Save 1st host drive
        mov     dl,bl           ; Set up for query of "host" drive

        mov     ax,4A11h        ; DBLSPACE.BIN INT 2F number
        mov     bx,1            ; bx = GetDriveMap function
        int     2Fh             ; (bl AND 7Fh) == 2nd host drive

        and     bl,7Fh          ; bl = 2nd host drive
        cmp     bl,drive        ; Is host of host of drive itself?
        mov     fSwapped,TRUE   ; Assume CVF is swapped
        je      gdiExit         ;   YES, CVF is swapped

        mov     fSwapped,FALSE  ;   NO, CVF is not swapped
        mov     drHost,bl       ; True host is 2nd host drive
        jmp     short gdiExit

    gdiHost:
        and     bl,7Fh          ; bl = host drive number
        cmp     bl,dl           ; Is drive swapped?
        je      gdiExit         ;    NO

        mov     fSwapped,TRUE   ;    YES
        mov     drHost,bl       ; Set boot drive number

    gdiExit:
    }

    *pdrHost   = drHost;
    *pfSwapped = fSwapped;
    *pseq      = seq;
    return fDoubleSpace;
}


/***    FlushDrive - Flush buffers for DoubleSpace drive
 *
 *      Entry:
 *          drive - Drive to flush (0=A, 1=B, etc.)
 *          fdop  - Additional operations (invalidate and/or disk reset)
 *
 *      Exit-Success:
 *          Returns TRUE.
 *
 *      Exit-Failure:
 *          Returns FALSE.
 *              Drive was not a DoubleSpace drive.
 */
BOOL FlushDrive(int drive, FDOP fdop)
{
    static DSPACKET dsp;
    BYTE            bDrive = (BYTE)drive;

    dsp.stamp   = MD_STAMP;         // Set stamp
    dsp.result  = DSP_ERROR;        // Init result code to assume error

    if (fdop & FDOP_INVALIDATE)
        dsp.command = DSCMD_FLUSH_AND_INVALIDATE; // Flush command
    else
        dsp.command = DSCMD_FLUSH;      // Flush command

    if (fdop & FDOP_DISK_RESET)
        _asm {
        MOV     ax,0Dh              ; Disk reset
        INT     21
        }

    _asm {
        MOV     AX,4404h            ; IOCtl read command
        MOV     BL,bDrive           ; drive letter (0-based)
        INC     BL                  ; drive letter (1-based)
        MOV     CX,cbDSPACKET       ; packet length
        MOV     DX,OFFSET dsp       ; ds:dx = IOCTL packet
        INT     21h                 ; Flush DoubleSpace drive
    }

    return (dsp.result == DSP_NO_ERROR);
}
