/* MergeDIB.c
 *
 * MergeDIB main loop etc.
 */
/* 
     (C) Copyright Microsoft Corp. 1991.  All rights reserved.

     You have a royalty-free right to use, modify, reproduce and 
     distribute the Sample Files (and/or any modified version) in 
     any way you find useful, provided that you agree that 
     Microsoft has no warranty obligations or liability for any 
     Sample Application Files which are modified. 
	 
	 
 Merges a primary DIB (with any palette) and a secondary DIB (with a different
 palette) into a single, merged DIB (with special palette).

 The special dib and palette are a combination of the two images and
 palettes so that when the palette is gradually crossfaded (animated),
 the first and second DIB are partially displayed.  At complete fade,
 only one of the bitmaps is 'visible', while at a 50-50 mix, both
 are equally visible (merged).  Pixels are not dithered betweent he
 images, but are mixed in the palettes.

 This technique is limited in the number of pixels that have different
 targets between bitmaps, but it can create very nice effects when just
 text is 'faded' in for the target bitmap.

 This code is limited so that the two DIBs must be the same size, but this
 limitation could easily be eliminated by creating an artificial bitmap
 that is the desired size with a 'blank' (where blank is a chosen color)
 background.  The smaller image could be centered or otherwise placed
 in the background (easy to do using the DIB Driver).

	 
 */

#define WinAssert(x)	(x)

#include "nocrap.h"
#include <windows.h>
#include "MergeDIB.h"
#include "dib.h"
#include "gmem.h"


/* globals */
HWND		ghwndApp;	// main application window
HMENU		ghmenuApp;	// main application menu
HANDLE		ghInst;		// program instance handle
GLOBALHANDLE	gahdib[3];	// primary/secondary/merged DIB
HPALETTE	gahpal[3];	// primary/secondary/merged palette
BOOL		giDIBViewed = PRIMARY_DIB; // which DIB is being viewed?
NPLOGPALETTE	gaplogpal[3];	// two sets of colors for merged DIB
WORD		gwFadeDirection = SB_LINEDOWN; // SB_LINEUP or SB_LINEDOWN
BOOL		gfFading = FALSE; // currently doing fade?


/* fOK = CreateMergedDIB()
 *
 * Merge <gahdib[PRIMARY_DIB]> (with palette <gahpal[PRIMARY_DIB]>)
 * and <gahdib[SECONDARY_DIB]> (with palette <gahpal[SECONDARY_DIB]>)
 * into a single DIB <gahdib[MERGED_DIB]> (with palette <gahpal[MERGED_DIB]>).
 * Also, create <gaplogpal> so that if the merged DIB's palette is set to
 * <gaplogpal[PRIMARY_DIB]> then the merged DIB will look like the primary
 * DIB, and if the merged DIB's palette is set to <gaplogpal[SECONDARY_DIB]>
 * then the merged DIB will look like the secondary DIB.
 *
 * On success, return TRUE.  On error, display an error message and
 * return FALSE.
 */
BOOL FAR PASCAL
CreateMergedDIB(void)
{
	BOOL		fOK = TRUE;	// function return status
	LPBITMAPINFOHEADER lpbiP;	// header of primary DIB
	LPBITMAPINFOHEADER lpbiS;	// header of secondary DIB
	LPBITMAPINFOHEADER lpbiM;	// header of merged DIB
	BYTE huge *	pBitsP;		// ptr. into bits of primary DIB
	BYTE huge *	pBitsS;		// ptr. into bits of secondary DIB
	BYTE huge *	pBitsM;		// ptr. into bits of merged DIB
	LPBYTE		pbMergeMap = NULL; // map (P,S) DIB to merged DIB
	LPBYTE		pbCell;		// pointer into merge map
	int		r, c;		// row/column of <aabMergMap>
	int		x, y;		// coordinate in a DIB
	DWORD		dwMergedColors = 0L; // no. colors in merged map
	BYTE		bPal;		// palette index
	HCURSOR		hcurPrev = NULL; // cursor before hourglass
	int		i;
	LPWORD		pw;

	/* show hourglass cursor */
	hcurPrev = SetCursor(LoadCursor(NULL, IDC_WAIT));

	/* get pointers to the DIBs */
	lpbiP = GLock(gahdib[PRIMARY_DIB]);
	lpbiS = GLock(gahdib[SECONDARY_DIB]);

	/* create the merged DIB */
	if ((gahdib[MERGED_DIB] = CreateDib(8,
			(int) lpbiP->biWidth, (int) lpbiP->biHeight)) == NULL)
		goto ERROR_OUTOFMEM;
	lpbiM = GLock(gahdib[MERGED_DIB]);
	for (pw = (LPWORD) ((LPSTR) lpbiM + lpbiM->biSize), i = 0; i < 256; i++)
		*pw++ = i;
	lpbiM->biClrUsed = 256;		// for now, will contain unused entries

	/* <pbMergeMap> is a 256x256 2-dimensional array of bytes;
	 * if a pixel in <gahdib[0]> contains palette entry number <i>
	 * and the pixel in the same location in <gahdib[0]> contains
	 * palette entry number <j>, then <pbMergeMap[i][j]> will contain
	 * <k> where <k> is the palette entry number in the merged palette
	 */
	#define MERGEMAP(r, c)	(pbMergeMap + ((((WORD) (r)) << 8) | (c)))
	#define UNUSEDENTRY	0xFF		// color combination not used
	if ((pbMergeMap = (LPBYTE) GAllocPtr(0x10000L)) == NULL)
		goto ERROR_OUTOFMEM;
	for (pbCell = pbMergeMap, r = 0; r < 256; r++)
		for (c = 0; c < 256; c++)
			*pbCell++ = UNUSEDENTRY;
	
	/* palette entry <i> of the merged DIB will contain either the i-th
	 * entry of <gaplogpal[0]> (if the merged DIB is being displayed
	 * to look like the primary DIB), or the i-th entry of <gaplogpal[1]>
	 * (if the merged DIB is being displayed to look like the secondary
	 * DIB), or some value in between
	 */
	for (i = PRIMARY_DIB; i <= MERGED_DIB; i++)
	{
		if ((gaplogpal[i] = (NPLOGPALETTE) LocalAlloc(LMEM_FIXED,
				sizeof(LOGPALETTE) +
				MAXPALSIZE * sizeof(PALETTEENTRY))) == NULL)
			goto ERROR_OUTOFMEM;
		gaplogpal[i]->palVersion = 0x300;
		gaplogpal[i]->palNumEntries = 0;
	}

	/* traverse the primary, secondary and merged DIBs simultaneously */
	for (y = 0; y < (int) lpbiM->biHeight; y++)
	{
		HDC		hdc;
		char		ach[100];
		RECT		rc;

		hdc = GetDC(ghwndApp);
		GetClientRect(ghwndApp, &rc);
		wsprintf(ach, "Merging DIBs: %ld%% done; %lu colors",
			((long) y * 100L) / lpbiM->biHeight, dwMergedColors);
		DrawText(hdc, ach, lstrlen(ach), &rc,
			DT_LEFT | DT_TOP | DT_NOPREFIX);
		ReleaseDC(ghwndApp, hdc);

		for (x = 0; x < (int) lpbiM->biWidth; x++)
		{
			pBitsP = DibXY(lpbiP, x, y);
			pBitsS = DibXY(lpbiS, x, y);
			pBitsM = DibXY(lpbiM, x, y);
			pbCell = MERGEMAP(*pBitsP, *pBitsS);

			if (*pbCell == UNUSEDENTRY)
			{
				/* allocate a new entry in merged palette;
				 * if there are too many entries, keep track
				 * of the total no. entries for the
				 * error box
				 */
				bPal = (BYTE) dwMergedColors;
				if (++dwMergedColors > (DWORD) MAXPALSIZE)
					continue;	// too many colors

				/* add an entry to gaplogpal[PRIMARY_DIB] */
				GetPaletteEntries(gahpal[PRIMARY_DIB],
					*pBitsP, 1, gaplogpal[PRIMARY_DIB]
							->palPalEntry + bPal);
				gaplogpal[PRIMARY_DIB]->palNumEntries++;

				/* add an entry to gaplogpal[SECONDARY_DIB] */
				GetPaletteEntries(gahpal[SECONDARY_DIB],
					*pBitsS, 1, gaplogpal[SECONDARY_DIB]
							->palPalEntry + bPal);
				gaplogpal[SECONDARY_DIB]->palNumEntries++;

				/* add an entry to gaplogpal[MERGED_DIB] */
				gaplogpal[MERGED_DIB]->palPalEntry[bPal] =
				    gaplogpal[PRIMARY_DIB]->palPalEntry[bPal];
				gaplogpal[MERGED_DIB]
				    ->palPalEntry[bPal].peFlags = PC_RESERVED;
				gaplogpal[MERGED_DIB]->palNumEntries++;

				/* update <pbMergeMap> */
				*pbCell = bPal;
			}

			/* fill in a pixel in the merged DIB */
			*pBitsM = *pbCell;
		}
	}

	if (dwMergedColors > (DWORD) MAXPALSIZE)
	{
#ifdef DOERRORS	    
		ErrorResBox(ghwndApp, ghInst, MB_ICONEXCLAMATION | MB_OK, 
			IDS_APPNAME, IDS_MERGEPALEXCEEDED,
			dwMergedColors, (long) MAXPALSIZE);
#endif		    
		goto RETURN_ERROR;
	}

	/* create the merged palette */
	if ((gahpal[MERGED_DIB] = CreatePalette(gaplogpal[MERGED_DIB])) == NULL)
		goto ERROR_OUTOFMEM;

	/* make unused entries in the palette-index table in the DIB
	 * point to palette index zero
	 */
	for (i = gaplogpal[MERGED_DIB]->palNumEntries,
	     pw = ((LPWORD) ((LPSTR) lpbiM + lpbiM->biSize)) + i;
	     i < 256; i++)
		*pw++ = 0;

	goto RETURN_SUCCESS;

ERROR_OUTOFMEM:				// out of memory
#ifdef DOERRORS
	ErrorResBox(ghwndApp, ghInst, MB_ICONEXCLAMATION | MB_OK, 
		IDS_APPNAME, IDS_OUTOFMEM);
#endif	    
	goto RETURN_ERROR;

RETURN_ERROR:

	fOK = FALSE;
	FreeDIB(MERGED_DIB);
	/* fall through */

RETURN_SUCCESS:

	if (pbMergeMap != NULL)
		GFreePtr(pbMergeMap);

	if (hcurPrev != NULL)
		SetCursor(hcurPrev);

	return fOK;
}


/* AppPaint(hwnd, hdc)
 *
 * Repaint window <hwnd>.
 */
void FAR PASCAL
AppPaint(hwnd, hdc)
HWND		hwnd;		// window to paint into
HDC		hdc;		// DC to paint into
{
	RECT		rcClient;
	BITMAPINFOHEADER bih;		// information about this DIB

	GetClientRect(hwnd, &rcClient);

	/* save clipping region */
	SaveDC(hdc);

	if (gahdib[giDIBViewed] != NULL)
	{
		/* get the size etc. of the bitmap */
		DibInfo(gahdib[giDIBViewed], &bih);

		/* draw the DIB and then exclude its rectangle from the
		 * clipping region of <hdc>
		 */
		DrawDib(hdc, 0, 0, gahdib[giDIBViewed], gahpal[giDIBViewed],
			DIB_PAL_COLORS);
		ExcludeClipRect(hdc, 0, 0,
			(int) bih.biWidth, (int) bih.biHeight);

		/* draw the black border between the DIB and the gray
		 * background
		 */
		PatBlt(hdc, 0, 0, (int) bih.biWidth + 1, (int) bih.biHeight + 1,
			BLACKNESS);
		ExcludeClipRect(hdc, 0, 0,
			(int) bih.biWidth + 1, (int) bih.biHeight + 1);
	}
	
	/* draw the gray background */
	SelectObject(hdc, GetStockObject(GRAY_BRUSH));
	PatBlt(hdc, rcClient.left, rcClient.top, rcClient.right - rcClient.left,
		rcClient.bottom - rcClient.top, PATCOPY);

	/* restore clipping region */
	RestoreDC(hdc, -1);
}


/* InitMenus()
 *
 * Check, uncheck, gray, and enable menus items in response to a
 * WM_INITMENU message.
 */
void FAR PASCAL
InitMenus(void)
{
	/* update the Edit menu */
	EnableMenuItem(ghmenuApp, IDM_COPYMERGEDPALETTE, MF_BYCOMMAND |
		(gahdib[MERGED_DIB] != NULL) ? MF_ENABLED : MF_GRAYED);

	/* update the View menu */
	EnableMenuItem(ghmenuApp, IDM_VIEWPRIMARYDIB, MF_BYCOMMAND |
		(gahdib[PRIMARY_DIB] != NULL) ? MF_ENABLED : MF_GRAYED);
	EnableMenuItem(ghmenuApp, IDM_VIEWSECONDARYDIB, MF_BYCOMMAND |
		(gahdib[SECONDARY_DIB] != NULL) ? MF_ENABLED : MF_GRAYED);
	EnableMenuItem(ghmenuApp, IDM_VIEWMERGEDDIB, MF_BYCOMMAND |
		(((gahdib[PRIMARY_DIB] != NULL) &&
		  (gahdib[SECONDARY_DIB] != NULL)) ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(ghmenuApp, IDM_FADE, MF_BYCOMMAND |
		((gahdib[MERGED_DIB] != NULL) ? MF_ENABLED : MF_GRAYED));
	CheckMenuItem(ghmenuApp, IDM_VIEWPRIMARYDIB, MF_BYCOMMAND |
		((giDIBViewed == 0) ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem(ghmenuApp, IDM_VIEWSECONDARYDIB, MF_BYCOMMAND |
		((giDIBViewed == 1) ? MF_CHECKED : MF_UNCHECKED));
	CheckMenuItem(ghmenuApp, IDM_VIEWMERGEDDIB, MF_BYCOMMAND |
		((giDIBViewed == 2) ? MF_CHECKED : MF_UNCHECKED));
}


/* WinMain(hInst, hPrev, lpszCmdLine, cmdShow)
 * 
 * The main procedure for the App.  After initializing, it just goes
 * into a message-processing loop until it gets a WM_QUIT message
 * (meaning the app was closed).
 */
int PASCAL			// returns exit code specified in WM_QUIT
WinMain(hInst, hPrev, lpszCmdLine, iCmdShow)
HANDLE		hInst;		// instance handle of current instance
HANDLE		hPrev;		// instance handle of previous instance
LPSTR		lpszCmdLine;	// null-terminated command line
int		iCmdShow;	// how window should be initially displayed
{
	HANDLE		hAccel;		// accelerator table
	MSG		msg;		// message from queue

	/* save instance handle for dialog boxes */
	ghInst = hInst;

	/* call initialization procedure */
	if (!AppInit(hInst, hPrev))
		return FALSE;

	/* create the application's window */
	#define APPWNDSTYLE (WS_OVERLAPPEDWINDOW | WS_VSCROLL)

	ghwndApp = CreateWindow
	(
		gachAppName,		// window class
		gachAppName,		// window caption
		APPWNDSTYLE,		// window style
		CW_USEDEFAULT, 0,	// initial position
		CW_USEDEFAULT, 0,	// initial size
		NULL, 			// parent window handle
		NULL,			// window menu handle
		hInst,			// program instance handle
		NULL			// create parameters
	);
	WinAssert(ghwndApp != NULL);
	SetScrollRange(ghwndApp, SB_VERT, 0, 0, FALSE);
	ShowWindow(ghwndApp, iCmdShow);

	/* load accelerator table */
	hAccel = LoadAccelerators(hInst, "AppAccel");
	WinAssert(hAccel != NULL);

	/* poll for messages from the event queue */
	while (GetMessage(&msg, NULL, 0, 0))
	{
		if (!TranslateAccelerator(ghwndApp, hAccel, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	return msg.wParam;
}


/* AppWndProc(hwnd, wMsg, wParam, lParam)
 * 
 * The window proc for the app's main window.
 */
LONG FAR PASCAL _export		// returns 0 iff processed message
AppWndProc(hwnd, wMsg, wParam, lParam)
HWND		hwnd;		// window's handle
WORD		wMsg;		// message number
WORD		wParam;		// message-dependent parameter
LONG		lParam;		// message-dependent parameter
{
	BYTE		bPal;		// palette index
	PALETTEENTRY NEAR *ppeP;	// palette entry in primary palette
	PALETTEENTRY NEAR *ppeS;	// palette entry in secondary palette
	PALETTEENTRY NEAR *ppeM;	// palette entry in merged palette
	FARPROC		fpfn;
	PAINTSTRUCT	ps;
	HDC		hdc;
	HPALETTE	hpalPrev;
	BOOL		f;
	int		iPos;
	HPALETTE	hpal;

	switch (wMsg)
	{

	case WM_CREATE: 

		ghmenuApp = GetMenu(hwnd);
		break;
	
	case WM_INITMENU:

		InitMenus();
		break;

	case WM_QUERYENDSESSION:
	case WM_CLOSE:

		/* clean up */
		AppExit(hwnd);
		break;

	case WM_DESTROY:

		/* exit this program */
		PostQuitMessage(0);
		break;


	case WM_PALETTECHANGED:

		/* if my window caused the palette change, do nothing */
		if (wParam == hwnd)
			break;
		
		/* fall through */

	case WM_QUERYNEWPALETTE:

		/* process WM_PALETTECHANGED or WM_QUERYNEWPALETTE:
		 * temporarily select and realize <gahpal[giDIBViewed]> to see
		 * if a full redraw is required (due to some of the physical
		 * palette entries <hwnd> uses being changed)
		 */
		if (gahpal[giDIBViewed] == NULL)
			return (long) FALSE;	// no palette to realize

		hdc = GetDC(hwnd);
		hpalPrev = SelectPalette(hdc, gahpal[giDIBViewed], FALSE);
		f = RealizePalette(hdc);
		SelectPalette(hdc, hpalPrev, FALSE);
		ReleaseDC(hwnd, hdc);

		if (f)
			InvalidateRect(hwnd, NULL, TRUE);

		return (LONG) f;

	case WM_ERASEBKGND:

		return 0L;			// WM_PAINT does erase

	case WM_PAINT:

		BeginPaint(hwnd, &ps);
		AppPaint(hwnd, ps.hdc);
		EndPaint(hwnd, &ps);
		break;

	case WM_COMMAND:

		switch (wParam)
		{

		case IDM_OPENPRIMARYDIB:

			FileOpen(FALSE, NULL);
			break;

		case IDM_OPENSECONDARYDIB:

			FileOpen(TRUE, NULL);
			break;

		case IDM_ABOUT:

			/* request to display "About" dialog box */
			fpfn = MakeProcInstance((FARPROC) AboutDlgProc, ghInst);
			DialogBox(ghInst, "AboutBox", hwnd, fpfn);
			FreeProcInstance(fpfn);
			break;

		case IDM_EXIT:

			/* request to exit this program */
			PostMessage(hwnd, WM_SYSCOMMAND, SC_CLOSE, 0L);
			break;

		case IDM_COPYMERGEDPALETTE:

			/* copy the current merged palette */
			hpal = CreatePalette(gaplogpal[MERGED_DIB]);
			if (hpal == NULL)
			{
#ifdef DOERRORS			    
				ErrorResBox(ghwndApp, ghInst,
					MB_ICONEXCLAMATION | MB_OK,
					IDS_APPNAME, IDS_OUTOFMEM);
#endif				    
				return 0L;
			}

			/* copy the palette to the clipboard */
			if (OpenClipboard(hwnd))
			{
				EmptyClipboard();
				SetClipboardData(CF_PALETTE, hpal);
				CloseClipboard();
			}
			break;

		case IDM_VIEWPRIMARYDIB:

			if (gahdib[PRIMARY_DIB] == NULL)
				return 0L;
			giDIBViewed = PRIMARY_DIB;
			SetScrollRange(hwnd, SB_VERT, 0, 0, TRUE);
			InvalidateRect(hwnd, NULL, TRUE);
			break;

		case IDM_VIEWSECONDARYDIB:

			if (gahdib[SECONDARY_DIB] == NULL)
				return 0L;
			giDIBViewed = SECONDARY_DIB;
			SetScrollRange(hwnd, SB_VERT, 0, 0, TRUE);
			InvalidateRect(hwnd, NULL, TRUE);
			break;

		case IDM_VIEWMERGEDDIB:

			if ((gahdib[PRIMARY_DIB] == NULL) ||
			    (gahdib[SECONDARY_DIB] == NULL))
				return 0L;

			if (gahdib[MERGED_DIB] == NULL)
			{
				/* create the merged DIB */
				if (!CreateMergedDIB())
					break;
			}

			giDIBViewed = MERGED_DIB;
			SetScrollRange(hwnd, SB_VERT, 0, SCROLL_RANGE, FALSE);
			SetScrollPos(hwnd, SB_VERT, 0, TRUE);
			InvalidateRect(hwnd, NULL, TRUE);
			break;

		case IDM_FADE:

			if (gahdib[MERGED_DIB] == NULL)
				return 0L;
			
			/* change fade direction at either end of scroll bar */
			iPos = GetScrollPos(hwnd, SB_VERT);
			if (iPos == 0)
				gwFadeDirection = SB_LINEDOWN;
			else
			if (iPos == SCROLL_RANGE)
				gwFadeDirection = SB_LINEUP;

			/* toggle fade on/off */
			gfFading = !gfFading;
			if (gfFading)
				SetTimer(hwnd, 1, 50, NULL);
			break;

		}
		return 0L;

	case WM_VSCROLL:

		if (gahdib[MERGED_DIB] == NULL)
			break;
		iPos = GetScrollPos(hwnd, SB_VERT);
		switch(wParam)
		{
		case SB_LINEUP:
			iPos -= SCROLL_LINE;
			break;
		case SB_LINEDOWN:
			iPos += SCROLL_LINE;
			break;
		case SB_PAGEUP:
			iPos -= SCROLL_PAGE;
			break;
		case SB_PAGEDOWN:
			iPos += SCROLL_PAGE;
			break;
		case SB_THUMBTRACK:
		case SB_THUMBPOSITION:
			iPos = LOWORD(lParam);
			break;
		case SB_TOP:
			iPos = 0;
			break;
		case SB_BOTTOM:
			iPos = SCROLL_RANGE;
			break;
		default:
			return 0L;
		}

		if (iPos <= 0)
			iPos = 0, gfFading = FALSE;
		if (iPos >= SCROLL_RANGE)
			iPos = SCROLL_RANGE, gfFading = FALSE;

		/* update <gaplogpal[MERGED_DIB]> */
		ppeP = gaplogpal[PRIMARY_DIB]->palPalEntry;
		ppeS = gaplogpal[SECONDARY_DIB]->palPalEntry;
		ppeM = gaplogpal[MERGED_DIB]->palPalEntry;
		for (bPal = 0;
		     bPal < (BYTE) gaplogpal[MERGED_DIB]->palNumEntries;
		     bPal++, ppeP++, ppeS++, ppeM++)
		{
			ppeM->peRed = (BYTE) (((long) ppeS->peRed * iPos +
				(long) ppeP->peRed * (SCROLL_RANGE - iPos))
					/ SCROLL_RANGE);

			ppeM->peGreen = (BYTE) (((long) ppeS->peGreen * iPos +
				(long) ppeP->peGreen * (SCROLL_RANGE - iPos))
					/ SCROLL_RANGE);

			ppeM->peBlue = (BYTE) (((long) ppeS->peBlue * iPos +
				(long) ppeP->peBlue * (SCROLL_RANGE - iPos))
					/ SCROLL_RANGE);
		}

		/* animate the palette */
		hdc = GetDC(hwnd);
		SelectPalette(hdc, gahpal[MERGED_DIB], FALSE);
		RealizePalette(hdc);
		AnimatePalette(gahpal[MERGED_DIB], 0,
			gaplogpal[MERGED_DIB]->palNumEntries,
			gaplogpal[MERGED_DIB]->palPalEntry);
		ReleaseDC(hwnd, hdc);

		/* update the scroll bar position */
		SetScrollPos(hwnd, SB_VERT, iPos, TRUE);
		return 0L;

	case WM_KEYDOWN:

		switch (wParam)
		{
		case VK_LEFT:
		case VK_UP:
			SendMessage(ghwndApp, WM_VSCROLL, SB_LINEUP, 0L);
			return 0L;
		case VK_RIGHT:
		case VK_DOWN:
			SendMessage(ghwndApp, WM_VSCROLL, SB_LINEDOWN, 0L);
			return 0L;
		case VK_PRIOR:
			SendMessage(ghwndApp, WM_VSCROLL, SB_PAGEUP, 0L);
			return 0L;
		case VK_NEXT:
			SendMessage(ghwndApp, WM_VSCROLL, SB_PAGEDOWN, 0L);
			return 0L;
		case VK_HOME:
			SendMessage(ghwndApp, WM_VSCROLL, SB_TOP, 0L);
			return 0L;
		case VK_END:
			SendMessage(ghwndApp, WM_VSCROLL, SB_BOTTOM, 0L);
			return 0L;
		}
		break;
	
	case WM_TIMER:

		if (!gfFading)
			KillTimer(hwnd, 1);
		SendMessage(hwnd, WM_VSCROLL, gwFadeDirection, 0L);
		break;

	}

	return DefWindowProc(hwnd, wMsg, wParam, lParam);
}
