/* framehk.c - Frame callback procedure and associated support
 *  code for MMPLAY sample application.
 */

#include <windows.h>
#include <mmp.h>
#include "mmplay.h"
#include "framehk.h"


/* External global variables defined in MMPLAY.C.
 */
extern HANDLE  hInst;
extern HWND    hMainWnd;
extern HWND    hFullWnd;
extern char    szAppName[];
extern MMPID   idMovie;
extern BOOL    bFullScreen;


/* Global variables.
 */
HANDLE hScriptText;
LPMMPFRAMEHOOK lpfnFrameCallback;

#define LOOP_CLEAR    0
#define LOOP_WAIT     1
#define LOOP_CONTINUE 2

WORD    wLoopState = LOOP_CLEAR;

char szKeytestName[] = "Looping...";

#define MAX_SCRIPT 1023
char szScriptText[MAX_SCRIPT+1];

#define BREAK_CHAR 'b'
#define GOTO_CHAR  'g'
#define CLOSE_CHAR 'c'
#define LOOP_CHAR  'l'


/* MMPlayFrameHook - Frame callback procedure for MMPLAY.
 *
 *  Params:     idMovie - Movie Player instance ID.
 *              wMsg - The message.
 *              wParam - Message dependent parameter.
 *              lParam - Message dependent parameter.
 *
 *  Returns:    Depends on message.
 */
BOOL FAR PASCAL
MMPlayFrameHook(MMPID idMovie, WORD wMsg, WORD wParam, LONG lParam)
{
    WORD wMsgOut, wParamOut;
    LONG lParamOut;

    switch(wMsg)
    {
    case MMP_HOOK_FRAME :
        return FALSE;

    case MMP_HOOK_SCRIPT :
        /* Check the script-channel text. If there's no text, or it's not
         * an MMPLAY command, let the Movie Player handle it.
         */
        if(lParam == NULL || *((LPSTR)lParam) != '!')
            return FALSE;

        if(lstrlen((LPSTR)lParam) > MAX_SCRIPT)
            return FALSE;

        /* Copy the script-channel text.
         */
        lstrcpy(szScriptText, (LPSTR)lParam);

        /* Convert the script-channel text to window-message format.
         * If the parsing succeeds, process the message.
         */
        AnsiLower(szScriptText);
        if(Parse(szScriptText, &wMsgOut, &wParamOut, &lParamOut))
        {
            if(bFullScreen && hFullWnd)
                DoCommand(hFullWnd, wMsgOut, wParamOut, lParamOut);
            else
                DoCommand(hMainWnd, wMsgOut, wParamOut, lParamOut);
            return TRUE;               // MMPLAY handled it
        }
        else
            return FALSE;              // Let Movie Player handle it

        break;

    case MMP_HOOK_LOAD:
        /* Monitor ESC key state while load proceeds. If key is pressed
         * any time during load process, return FALSE to cancel load.
         */         
        if(wParam == 0)     
        {
            GetAsyncKeyState(VK_ESCAPE);
            return TRUE;
        }

        if(GetAsyncKeyState(VK_ESCAPE) & 0x0001)
            return FALSE;
        else
            return TRUE;

    default:
        return FALSE;
    }
}

/* DoCommand - Processes script channel commands. 
 *
 *  Params: Standard window-procedure parameters.
 *
 *  Return: void
 */
void DoCommand(HWND hWnd, int iMessage, WORD wParam, LONG lParam)
{
    switch(iMessage)
    {
    case WM_MMPLAY_GOTO:
        if(wParam)
            mmpGoToFrame(idMovie, wParam, MMP_DRAW_FRAME);
        break;
        
    case WM_MMPLAY_BREAK:
        MessageBox(hWnd, (LPSTR)lParam, "Movie Player Frame Hook", MB_OK);
        break;

    case WM_MMPLAY_LOOP:

        /* The wLoopState flag indicates the current state of the loop.
         */
        switch(wLoopState)
        {
        /* If the flag is LOOP_CLEAR, the movie is just entering
         * the loop. Display the "Continue" button and begin the
         * loop.
         */
        case LOOP_CLEAR:
            CreateWindow(szKeytestName, szKeytestName,
                WS_CHILD | WS_VISIBLE | WS_DLGFRAME | WS_SYSMENU | WS_CAPTION,
                10, 10, 100, 60,
                hWnd, NULL, hInst, 0L);
            wLoopState = LOOP_WAIT;
            mmpGoToFrame(idMovie, wParam, MMP_DRAW_FRAME);
            break;

        /* If the flag is LOOP_WAIT, the movie is in the loop.
         * The flag remains LOOP_WAIT until the user presses the
         * Continue button.
         */
        case LOOP_WAIT:
            mmpGoToFrame(idMovie, wParam, MMP_DRAW_FRAME);
            break;

        /* If the flag is LOOP_CONTINUE, exit the loop.
         */
        case LOOP_CONTINUE:
            wLoopState = LOOP_CLEAR;
            break;
        }
        break;

    case WM_MMPLAY_CLOSE:
        MessageBox(hWnd, (LPSTR)lParam, "Movie Player Frame Hook", MB_OK);
        PostQuitMessage(1);
        break;
    }
}


WORD Tokenize(PSTR sz, PSTR aszTokens[])
{
    WORD i;

    sz++;                                               // Past '!'
    for(i = 0; i < MAX_TOKENS; i++)
    {
        while(*sz && *sz == ' ')                        // Past spaces
            sz++;

        if(*sz == 0)
            return i;

        if(*sz == '\"')                                 // Quoted token
        {
            aszTokens[i] = ++sz;
            while(*sz && *sz != '\"')
                sz++;
        }
        else
        {
            aszTokens[i] = sz;                         // One-word token
            while(*sz && *sz != ' ')
                sz++;
        }
        if(*sz)
            *sz++ = '\0';
        else
            return i+1;
    }
    return i;
}

                         
BOOL Parse(PSTR szText, WORD *pwMsg, WORD *pwParam, LONG *plParam)
{
    PSTR aszTokens[MAX_TOKENS];
    WORD  cTokens;
    LPSTR lpsz;

    if((cTokens = Tokenize(szText, aszTokens)) == 0)
        return FALSE;

    *pwParam = NULL;
    *plParam = NULL;

    switch(*aszTokens[0])
    {
    case BREAK_CHAR:                                  // BREAK
        if(cTokens < 2)
            return FALSE;

        *pwMsg = WM_MMPLAY_BREAK;
        *plParam = (LONG)(LPSTR)aszTokens[1];

        return TRUE;

    case GOTO_CHAR:                                   // GOTO
        if(aszTokens[1] == NULL)
            return FALSE;

        *pwMsg = WM_MMPLAY_GOTO;
        for(lpsz = aszTokens[1]; *lpsz >= '0' && *lpsz <= '9'; lpsz++)
            *pwParam = (*pwParam * 10) + *lpsz - '0';

        return TRUE;

    case CLOSE_CHAR:                                   // CLOSE
        *pwMsg = WM_MMPLAY_CLOSE;
        if(cTokens > 1)
            *plParam = (LONG)(LPSTR)aszTokens[1];
        return TRUE;

    case LOOP_CHAR:                                    // KEYTEST
        *pwMsg = WM_MMPLAY_LOOP;
        if(aszTokens[1] == NULL)
            return FALSE;

        for(lpsz = aszTokens[1]; *lpsz >= '0' && *lpsz <= '9'; lpsz++)
            *pwParam = (*pwParam * 10) + *lpsz - '0';

        if(*pwParam == 0)
            *pwParam = 1;
        return TRUE;

    default:
        return FALSE;
    }
}


LONG FAR PASCAL KeytestWndProc (
    HWND hWnd,
    unsigned iMessage,
    WORD wParam,
    LONG lParam)
{
    static HWND hOK;
    switch(iMessage)
    {
    case WM_CREATE:
        wLoopState = LOOP_WAIT;
        hOK = CreateWindow("BUTTON","&Continue",
                           BS_DEFPUSHBUTTON | WS_CHILD | WS_VISIBLE,
                           5, 5, 80, 30, hWnd, IDOK, hInst, 0L);
        break;

    case WM_COMMAND:
        if(wParam == IDOK)
            DestroyWindow(hWnd);
        return 0L;

    case WM_DESTROY:
        wLoopState = LOOP_CONTINUE;
        break;
    }
    return DefWindowProc(hWnd, iMessage, wParam, lParam);
}

