/* mmplay.c - WinMain() and MainWndProc() for MMPLAY, along with
 *      initialization and support code.
 *
 * MMPLAY is a Windows with Multimedia sample application that 
 *   illustrates how to use the Multimedia Movie Player (MMP).
 *   You can use MMPLAY to play any movie file. MMPLAY shows how 
 *   to load and play a movie, manage the stage window, and single-step 
 *   animation. 
 *
 *   MMPLAY also shows how to use a frame-callback function to 
 *   monitor script-channel text to recognize break, goto, loop, 
 *   and close commands. The frame-callback source code is in the 
 *   FRAMEHK.C module. SAMPLE.MMM, the movie file included with this
 *   project, contains custom script commands that MMPLAY recognizes.   
 *
 *   (C) Copyright Microsoft Corp. 1991.  All rights reserved.
 *
 *   You have a royalty-free right to use, modify, reproduce and 
 *   distribute the Sample Files (and/or any modified version) in 
 *   any way you find useful, provided that you agree that 
 *   Microsoft has no warranty obligations or liability for any 
 *   Sample Application Files which are modified. 
 */

#include <windows.h>
#include <mmp.h>
#include <direct.h>
#include "dlgopen.h"
#include "mmplay.h"
#include "framehk.h"


/* External global variables defined in FRAMEHK.C.
 */
extern LPMMPFRAMEHOOK lpfnFrameCallback;
extern char szKeytestName[];


/* Global variables.
 */
HANDLE  hInst;
HWND    hMainWnd;
HWND    hFullWnd = NULL;
MMPID   idMovie;
  
BOOL bRepeat;
BOOL bMute;
BOOL bFullScreen;
BOOL bLoadNoStatic;
BOOL bLoadOnDemand;
BOOL bLoadExpandDibs;

/* This enumeration type keeps track of the animation state.
 *
 * NO_MOVIE      No movie is loaded.
 * STOP_REQUEST  A "stop" command was received from user. The app will
 *               stop the animation after the current frame is finished.
 * STOPPED       The animation is stopped.
 * PLAYING       The animation is running.
 */
enum movieState { NO_MOVIE, STOP_REQUEST, STOPPED, PLAYING };
enum movieState iState = NO_MOVIE;

char szAppName[] =              "Multimedia Movie Player";
char szFullScreenName[] =       "Full Screen Window";

short nTotalFrames;

short nX;
short nY;
short nWidth;
short nHeight;

#define NAMESIZE 144
#define BUFSIZE  (NAMESIZE*2)
char szTextBuffer[BUFSIZE+1];
char szFilename[NAMESIZE+1];


/* WinMain - Entry point for MMPLAY.
 */
int PASCAL  WinMain (
    HANDLE  hInstance,
    HANDLE  hPrevInstance,
    LPSTR   lpszCmdLine,
    int     nCmdShow)
{
    MSG msg;
    int iAnimateResult;                // Return value from mmpAnimate

    /* Initialize the application.
     */
    if((hMainWnd = AppInit(hInstance, hPrevInstance)) == NULL)
        return FALSE;

    /* This message loop calls mmpAnimate only when there are no other
     * messages waiting. It calls WaitMessage in the following situations:
     *
     * - No movie is loaded, or app has playback stopped.
     * - mmpAnimate returns a device or mouse wait.
     * 
     * If mmpAnimate returns "stopped", the movie has reached the end;
     * otherwise, the app would be aware of the stopped status.
     *
     * When mmpAnimate returns "frame done", we check a global variable
     * to determine whether the user has stopped the animation. We only
     * call mmpStopAnimating immediately after the "frame done" return.
     */    
    while(1)
    {
        if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
        {
            if(msg.message == WM_QUIT)
                break;
            TranslateMessage ( &msg );
            DispatchMessage ( &msg );
        }
        else
        {
            if(iState == STOPPED || iState == NO_MOVIE)
            {
                WaitMessage();
            }
            else
            {
                iAnimateResult = mmpAnimate(idMovie);
                switch(iAnimateResult)
                {
                    case MMP_ANIM_STOPPED:
                        if(bFullScreen)
                            SendMessage(hFullWnd, WM_MMPLAY_SWITCH, 0, 0L);

                        iState = STOPPED;
                        SetMenuStates(hMainWnd);
                        break;

                    case MMP_MOUSE_WAIT:  // Device wait, no need to continue
                    case MMP_DEVICE_WAIT:
                        WaitMessage();
                        break;

                    case MMP_FRAME_DONE:        // See if stop was requested
                        if(iState == STOP_REQUEST)
                        {
                            mmpStopAnimating(idMovie,0);
                            iState = STOPPED;
                            SetMenuStates(hMainWnd);
                        }
                        break;
                }
            }
        }
    }
    AppClose();
    return msg.wParam;
}

/* AppClose - Unregisters window classes before application closes.
 *
 * Params:  void
 *
 * Return:  void
 */
void near AppClose(void)
{
    /* Unregister main window. If main window still exists, 
     * don't unregister others.
     */
    if(!UnregisterClass(szAppName,hInst))
        return;
            
    UnregisterClass(szFullScreenName,hInst);
    UnregisterClass(szKeytestName,hInst);
}


/* AppInit - Initializes application. Registers window classes
 *  and creates the main window.
 *
 * Params:  hInstance - Application instance handle.
 *          hPrevInstance - Previous application instance handle.
 *
 * Returns: Handle to main application window
 */
HWND AppInit (
    HANDLE hInstance,
    HANDLE hPrevInstance)
{
    WNDCLASS wndclass;
    hInst = hInstance;

    /* Only register window classes if this is the
     * first instance of the application.
     */
    if(!hPrevInstance)
    {
        /* Register main window class. Note the use of the CS_OWNDC
         * style flag for this window and the full-screen stage window.
         * This gives these windows their own DC (display context)
         * for better movie playback performance. If you use CS_OWNDC,
         * you must use mmpSetDC() to give MMP your DC.
         */
        wndclass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
        wndclass.lpfnWndProc = MainWndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon(hInstance, "MPPlayIcon");
        wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
        wndclass.hbrBackground = GetStockObject(WHITE_BRUSH);
        wndclass.lpszMenuName = "MMLoadMenu";
        wndclass.lpszClassName = szAppName;
        if(!RegisterClass (&wndclass))
            return NULL;

        /* Register full-screen stage window class.
         */
        wndclass.style = CS_OWNDC;
        wndclass.lpfnWndProc = StageWndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = NULL;
        wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
        wndclass.hbrBackground = GetStockObject(WHITE_BRUSH);
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szFullScreenName;
        if(!RegisterClass (&wndclass))
            return NULL;

        /* Register key-test window class.
         */
        wndclass.style = CS_HREDRAW | CS_VREDRAW;
        wndclass.lpfnWndProc = KeytestWndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = NULL;
        wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
        wndclass.hbrBackground = GetStockObject(WHITE_BRUSH);
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szKeytestName;
        if(!RegisterClass (&wndclass))
            return NULL;
    }

    /* Get setup preferences from MMPLAY.INI.
     */
    ReadFlags();

    /* Create the main window.
     */
    return CreateWindow(szAppName, szAppName,
                        WS_OVERLAPPEDWINDOW | WS_VISIBLE | WS_CLIPCHILDREN,
                        nX, nY, nWidth, nHeight,
                        NULL, NULL, hInstance, NULL);
}


/* InitMainWnd - Performs initialization for the main stage
 *  window. Opens an instance of the Movie Player, sets some
 *  MMP options, installs a frame callback function, and
 *  initializes menu items to the proper state.
 *
 * Params:  hWnd - Handle to main stage window.
 *
 * Return:  TRUE if initialization successful, FALSE otherwise.
 */
BOOL InitMainWnd(HWND hWnd)
{
    /* Open a Movie Player instance and set option flags.
     */
    if((idMovie = mmpOpen(hWnd, NULL)) == NULL)
    {
        ShowMMPError();
        return FALSE;
    }
    mmpSetMute(idMovie, (BYTE)bMute);
    mmpSetRepeat(idMovie, (BYTE)bRepeat);

    /* Initialize the movie state variable.
     */
    iState = NO_MOVIE;

    /* Give MMP our display context. Normally, MMP will Get/Release the 
     * DC for the window each time we call an API that requires it.
     * Supplying the DC skips this step. This optimization may not increase
     * performance much, but it does ensure that you won't run short
     * of DCs while playing the movie--it's possible that someone else
     * might allocate all the common DCs.
     */
    mmpSetDC(idMovie, GetDC(hWnd));

    /* Install a frame callback function to process script-channel commands.
     */
    if(lpfnFrameCallback = MakeProcInstance(MMPlayFrameHook, hInst))
        mmpSetFrameHook(idMovie, lpfnFrameCallback);

    /* Set up menus to the proper state.
     */
    SetMenuStates(hWnd);

    return TRUE;
}


/* MainWndProc - Window procedure function for main stage window.
 */
LONG FAR PASCAL MainWndProc (
    HWND hWnd,
    unsigned iMessage,
    WORD wParam,
    LONG lParam)
{
    PAINTSTRUCT ps;
    static BOOL bControl;

    switch(iMessage)
    {
    case WM_CREATE:
        /* Initialize the main stage window.
         */
        if(!InitMainWnd(hWnd))
        {
            PostQuitMessage(1);
            return -1;
        }
        break;

    case WM_COMMAND :
        return HandleCommands(hWnd, iMessage, wParam, lParam);

    case WM_PAINT :
        BeginPaint(hWnd, &ps);
        mmpUpdate(idMovie, ps.hdc, &ps.rcPaint);
        EndPaint(hWnd, &ps);
        break;

    case WM_CLOSE:
        FreeProcInstance(lpfnFrameCallback);

        if(!mmpClose(idMovie, 0))
            ShowMMPError();

        WriteFlags();

        DestroyWindow(hWnd);
        break;

    case WM_DESTROY :
        PostQuitMessage(1);
        break;

    case WM_KEYDOWN:
        if(iState == NO_MOVIE)
            break;

        switch(wParam)
        {
        case VK_SPACE:
            if(iState == PLAYING)
                SendMessage(hWnd, WM_COMMAND, IDM_STOP, 0L);
            else
                SendMessage(hWnd, WM_COMMAND, IDM_START, 0L);
            break;

        case VK_INSERT:
            if(bControl)
                CopyFrame(hWnd);
            break;

        case VK_LEFT:
            SendMessage(hWnd, WM_COMMAND, IDM_STEPBACKWARD, 0L);
            break;

        case VK_RIGHT:
            SendMessage(hWnd, WM_COMMAND, IDM_STEPFORWARD, 0L);
            break;

        case VK_CONTROL:
            bControl = TRUE;
            break;
        }
        break;

    case WM_KEYUP:
        if(wParam == VK_CONTROL)
            bControl = FALSE;
        break;

    default:
        return DefWindowProc(hWnd, iMessage, wParam, lParam);
    }
    return 0L;
}



/* HandleCommands - This function handles command messages for the
 * main application window. This is where much of the animation calls
 * are made.
 *
 * Params: Standard parameters passed to window function.
 */
LONG HandleCommands (
    HWND hWnd,
    int iMessage,
    WORD wParam,
    LONG lParam)
{
    short nFrame;
    FARPROC fpfn;
    BOOL  bReturn;
    int fh;

    switch(wParam)
    {
    case IDM_EXIT :
        PostMessage(hWnd, WM_CLOSE, 0, 0L);
        break;

    case IDM_ABOUT:
        fpfn = MakeProcInstance(AppAbout, hInst);
        DialogBox(hInst, MAKEINTRESOURCE(ABOUTBOX), hWnd, fpfn);
        FreeProcInstance (fpfn);
        break;

    case IDM_LOADOPTIONS:
        fpfn = MakeProcInstance(LoadOptions, hInst);
        bReturn = DialogBox(hInst, MAKEINTRESOURCE(OPTIONS), hWnd, fpfn);
        FreeProcInstance (fpfn);

        if(bReturn && iState != NO_MOVIE)
        {
            OpenMovie(TRUE);
            SetMenuStates(hWnd);
        }
        break;

    case IDM_OPEN :
        fh=OpenFileDialog(hWnd, "Open Animation File",
                          "*.mmm", DLGOPEN_MUSTEXIST | OF_EXIST | OF_READ,
                          NULL, szTextBuffer, 127);

        if(fh==DLGOPEN_CANCEL)
            break;

        lstrcpy(szFilename, szTextBuffer);
        OpenMovie(FALSE);
        SetMenuStates(hWnd);
            
        mmpGoToFrame(idMovie, MMP_FRAME_FIRST, MMP_DRAW_FRAME);
            
        break;

    case IDM_CLOSE :
        if(mmpFreeFile(idMovie, MMP_ERASE_FRAME))
        {
            SetWindowText(hWnd, szAppName);
            iState = NO_MOVIE;
        }
        else
        {
            if(!mmpClose(idMovie, 0) || !InitMainWnd(hWnd))
            {
                ShowMMPError();
                PostQuitMessage(1);
                return 0L;
            }
        }
        SetMenuStates(hWnd);
        break;

    case IDM_MUTE:
        bMute = !bMute;
        mmpSetMute(idMovie, (BYTE)bMute);
        SetMenuStates(hWnd);
        break;

    case IDM_LOOPMOVIE:
        bRepeat = !bRepeat;
        mmpSetRepeat(idMovie, (BYTE)bRepeat);
        SetMenuStates(hWnd);
        break;
        
    case IDM_FULLSCREEN :
        bFullScreen = !bFullScreen;
        if(bFullScreen && iState == PLAYING && !CreateStage())
        {
            MessageBeep(MB_ICONEXCLAMATION);
            MessageBox(hWnd, "Failed to create full-screen stage.",
                       szAppName, MB_OK | MB_ICONEXCLAMATION);
            bFullScreen = FALSE;
        }
        SetMenuStates(hWnd);

        break;

    case IDM_STOP :
        /* We should only call mmpStopAnimating after mmpAnimate returns
         * an MMP_FRAME_DONE value. Here, we just set a global variable
         * to indicate that user has stopped animation. We call
         * mmpStopAnimating in the message loop after the Movie Player
         * finishes processing the current frame.
         */
        if(iState == PLAYING)
            iState = STOP_REQUEST;
        break;
            
    case IDM_START :
        iState = PLAYING;
        if(bFullScreen && hFullWnd == NULL)
        {
            if(!CreateStage())
            {
                MessageBeep(MB_ICONEXCLAMATION);
                MessageBox(hWnd, "Failed to create full-screen window.",
                           szAppName, MB_OK | MB_ICONEXCLAMATION);
                bFullScreen = FALSE;
            }
        }
        else
        {
            SetMenuStates(hWnd);
        }
        mmpStartAnimating(idMovie, 0);
        break;

    case IDM_COPYFRAME:
        CopyFrame(hWnd);
        break;

    case IDM_REWIND:
        mmpGoToFrame(idMovie, MMP_FRAME_FIRST, MMP_DRAW_FRAME);
        break;

    case IDM_STEPFORWARD:
        if(iState == NO_MOVIE)
            break;

        if((nFrame = mmpGetCurFrame(idMovie)) == nTotalFrames)
        {
            MessageBeep(MB_ICONEXCLAMATION);
            MessageBox(hFullWnd ? hFullWnd : hWnd,
                       "Can't step beyond end.", szAppName,
                       MB_OK | MB_ICONEXCLAMATION);
            break;
        }

        if(!mmpGoToFrame(idMovie, ++nFrame, MMP_DRAW_FRAME))
            ShowMMPError();

        break;
                                            
    case IDM_STEPBACKWARD:
        if(iState == NO_MOVIE)
            break;

        if((nFrame = mmpGetCurFrame(idMovie)) == MMP_FRAME_FIRST)
        {
            MessageBeep(MB_ICONEXCLAMATION);
            MessageBox(hFullWnd ? hFullWnd : hWnd,
                       "Can't step before beginning.",
                       szAppName, MB_OK | MB_ICONEXCLAMATION);
            break;
        }

        if(!mmpGoToFrame(idMovie, --nFrame, MMP_DRAW_FRAME))
        {
            ShowMMPError();
            break;
        }
        break;

    default :
        return DefWindowProc (hWnd, iMessage, wParam, lParam);
    }
    return 0L;
}


/* StageWndProc -- Window function for full-screen stage window.
 *
 * Params: Standard parameters passed to window function.
 */
LONG FAR PASCAL StageWndProc(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    PAINTSTRUCT ps;
    static BOOL bControl;

    switch(wMsg)
    {
    case WM_CREATE:
        bControl = FALSE;
        mmpSetDC(idMovie, GetDC(hWnd));
        break;

    case WM_MMPLAY_SWITCH:
        if(iState == PLAYING)
            iState = STOP_REQUEST;

        if(! mmpSetStage(idMovie, hMainWnd, NULL, NULL))
            ShowMMPError();

        if(! mmpSetDC(idMovie, GetDC(hMainWnd)))
            ShowMMPError();

        DestroyWindow(hWnd);
        break;

    case WM_DESTROY:
        hFullWnd = NULL;
        break;
        
    case WM_PAINT:
        BeginPaint(hWnd, &ps);
        mmpUpdate(idMovie, ps.hdc, &ps.rcPaint);
        EndPaint(hWnd, &ps);
        break;

    case WM_KEYDOWN:
        switch(wParam)
        {
        case VK_ESCAPE:
            PostMessage(hWnd, WM_MMPLAY_SWITCH, 0, 0L);
            break;

        case VK_SPACE:
            if(iState == PLAYING)
            {
                iState = STOP_REQUEST;
            }
            else
            {
                iState = PLAYING;
                mmpStartAnimating(idMovie, 0);
            }
            break;

        case VK_INSERT:
            if(bControl)
                CopyFrame(hWnd);
            break;

        case VK_CONTROL:
            bControl = TRUE;
            break;

        case VK_LEFT:
            SendMessage(hMainWnd, WM_COMMAND, IDM_STEPBACKWARD, 0L);
            break;

        case VK_RIGHT:
            SendMessage(hMainWnd, WM_COMMAND, IDM_STEPFORWARD, 0L);
            break;
        }
        break;

    case WM_KEYUP:
        if(wParam == VK_CONTROL)
            bControl = FALSE;
        break;

    default:
        return DefWindowProc(hWnd, wMsg, wParam, lParam);
    }
    return 0L;
}



/* AppAbout -- Handles "About" dialog box message handling.
 *
 * Params:  Standard parameters passed to dialog box function.
 */
BOOL FAR PASCAL AppAbout(HWND hDlg, unsigned msg, WORD wParam, LONG lParam)
{
    switch (msg)
    {
    case WM_COMMAND:
        if(wParam == IDOK)
        {
            EndDialog(hDlg, TRUE);
        }
        break;

    case WM_INITDIALOG:
        return TRUE;
    }
    return FALSE;
}



/* LoadOptions -- Dialog box function for mmpLoadFile options.
 *
 * Params:  Standard parameters passed to dialog box function.
 */
BOOL FAR PASCAL LoadOptions(HWND hWnd, unsigned msg, WORD wParam, LONG lParam)
{
    switch (msg)
    {
    case WM_COMMAND:
        switch(wParam)
        {
        case IDOK:
            bLoadNoStatic = IsDlgButtonChecked(hWnd, IDD_NOSTATIC);
            bLoadOnDemand = IsDlgButtonChecked(hWnd, IDD_ONDEMAND);
            bLoadExpandDibs = IsDlgButtonChecked(hWnd, IDD_EXPANDDIBS);
            EndDialog(hWnd,TRUE);
            break;

        case IDCANCEL:
            EndDialog(hWnd,FALSE);
            break;
        }
        break;

    case WM_INITDIALOG:
        CheckDlgButton(hWnd, IDD_NOSTATIC, bLoadNoStatic);
        CheckDlgButton(hWnd, IDD_ONDEMAND, bLoadOnDemand);
        CheckDlgButton(hWnd, IDD_EXPANDDIBS, bLoadExpandDibs);

        return TRUE;
    }
    return FALSE;
}

/* CreateStage - Create full-screen stage window and transfer
 * movie playback to it.
 *
 * Params:  None
 *
 * Return:  TRUE if window created successfully, FALSE otherwise.
 */
BOOL CreateStage(void)
{
    hFullWnd = CreateWindow(szFullScreenName,
                            NULL, WS_POPUP | WS_VISIBLE | WS_CLIPCHILDREN,
                            0, 0,
                            GetSystemMetrics(SM_CXSCREEN),
                            GetSystemMetrics(SM_CYSCREEN),
                            hMainWnd, NULL, hInst, NULL);

    if(hFullWnd == NULL)
        return FALSE;

    /* Switch the stage to the full-screen window.
     */
    if(!mmpSetStage(idMovie, hFullWnd, NULL,
                        MMP_STAGE_BORDER | MMP_STAGE_CENTER))
    {
        ShowMMPError();
    }

    return TRUE;
}


/* OpenMovie -- This function loads a movie file for playback.
 *
 * Params:  bReload = TRUE if same movie is being loaded again (options
 *                    changing)
 *
 * Return:  TRUE if movie loaded when function finished, FALSE otherwise.
 */
BOOL OpenMovie(BOOL bReload)        // bReload is TRUE if loading same file
{                                   // Global szFilename holds the filename
    BOOL bLoadResult;
    WORD wFrameNum, wOptions = 0;
    HCURSOR hSaveCursor;

    if(bReload)
    {
        wFrameNum = mmpGetCurFrame(idMovie);   // If reload, record position
    }
    else
    {
        SetWindowText(hMainWnd, szAppName);  // If new file, reset window text
        wOptions = MMP_ERASE_FRAME | MMP_DRAW_FRAME;
    }

    if(bLoadNoStatic)   wOptions |= MMP_LOAD_NOSTATIC;
    if(bLoadOnDemand)   wOptions |= MMP_LOAD_ONDEMAND;
    if(bLoadExpandDibs) wOptions |= MMP_LOAD_EXPANDDIBS;

    hSaveCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    bLoadResult = mmpLoadFile(idMovie, szFilename, wOptions);
    SetCursor(hSaveCursor);

    if(bLoadResult)
    {
        if(bReload)                
            // Jump to saved frame position
            mmpGoToFrame(idMovie, wFrameNum, MMP_DRAW_FRAME);
        else                        
            // Change window caption and size
            SetupStageWindow(hMainWnd, idMovie);

        iState = STOPPED;
    }
    else
    {
        if(mmpError(idMovie, NULL, NULL) != MMPERR_USER_ABORT)
        {
            ShowMMPError();
        }
        iState = NO_MOVIE;
    }
    return bLoadResult;
}



/* SetupStageWindow -- This function changes the title of the stage window
 * and resizes the stage window to fit the movie size.
 *
 * Params:  hWndStage -- Handle of stage window
 *          idMovie   -- Current movie ID
 *
 * Return:  None
 */
void SetupStageWindow(HWND hWndStage, MMPID idMovie)
{
    RECT rc;
    MMPINFO mmpInfo;

    mmpInfo.achFullMacName[0] = 0;
    mmpGetMovieInfo(idMovie, &mmpInfo);

    /* Get the number of frames in the movie.
     */
    nTotalFrames = (short) mmpInfo.dwTotalFrames;
    
    if(mmpInfo.achFullMacName[0])
        SetWindowText(hWndStage, mmpInfo.achFullMacName);
    else
        SetWindowText(hWndStage, szFilename);

    if(mmpInfo.dwMovieExtentX && mmpInfo.dwMovieExtentY && mmpInfo.dwMovieExtentX<32767L)
    {
        SetRect(&rc, 0, 0, (WORD)mmpInfo.dwMovieExtentX, (WORD)mmpInfo.dwMovieExtentY);
        AdjustWindowRect(&rc, GetWindowLong(hWndStage, GWL_STYLE), TRUE);
        SetWindowPos(hWndStage, NULL, 0, 0, rc.right-rc.left, rc.bottom-rc.top,
            SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);
    }
}


/* SetMenuStates -- Enable or disable pull down menus as needed depending
 * on whether the movie file is absent, stopped, or running.
 *
 * Params:  None
 */
void SetMenuStates(HWND hWnd)
{
    static int iOldState = NO_MOVIE;
    HMENU hMenu;
    HMENU hOldMenu = NULL;

    /* Kind of bogus, but eliminates jumpiness by only reloading the
     * menu when we've gone from "movie loaded" to "movie not loaded."
     */
    if(iOldState != iState)
    {
        iOldState = iState;

        if(iState == NO_MOVIE)
            hMenu = LoadMenu(hInst, "MMLoadMenu");
        else
        {
            hMenu = LoadMenu(hInst, "MMPlayMenu");
            if(iState == PLAYING)
                ModifyMenu(hMenu, IDM_START, MF_BYCOMMAND,
                                  IDM_STOP, "&Stop animating\tSpace");
        }
        hOldMenu = GetMenu(hWnd);
    }
    else
    {
        hMenu = GetMenu(hWnd);
    }

    CheckMenuItem(hMenu, IDM_LOOPMOVIE,
                    bRepeat ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hMenu, IDM_MUTE,
                    bMute ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hMenu, IDM_FULLSCREEN,
                    bFullScreen ? MF_CHECKED : MF_UNCHECKED);

    if(hOldMenu)
    {
        DestroyMenu(hOldMenu);
        SetMenu(hWnd, hMenu);
    }
    else
    {
        DrawMenuBar(hWnd);
    }
}


/* ShowMMPError - Calls the MMP error function (mmpError) to get a
 *   textual description of an MMP error after a MMP function has
 *   failed. Displays the error text using MessageBox().
 *
 * Params:  none
 *
 * Return:  void
 */
void ShowMMPError(void)
{
    char szErrorBuf[MMP_MAXERRORLENGTH];

    mmpError(idMovie, szErrorBuf, MMP_MAXERRORLENGTH);

    MessageBeep(MB_ICONEXCLAMATION);
    MessageBox(NULL, szErrorBuf, "MMP Error", MB_OK | MB_ICONEXCLAMATION);
}



/* CopyFrame -- This function images a frame into a bitmap and copies the
 *              bitmap and palette to the Clipboard.
 *
 * Params:      hWnd -- Handle to window.
 */
void CopyFrame(HWND hWnd)
{
    HPALETTE hPalette, hClipPalette;
    HDC hDC, hMemoryDC;
    HBITMAP hBitmap;

    MMPINFO mmpInfo;
    int nX, nY;

    hPalette = mmpGetPaletteHandle(idMovie);
    if((hClipPalette = CopyPalette(hPalette)) == NULL)
    {
        MessageBeep(MB_ICONEXCLAMATION);
        MessageBox(hWnd, "Could not copy palette.", szAppName,
                        MB_OK | MB_ICONEXCLAMATION);
        return;
    }

    hDC = GetDC(hWnd);
    hMemoryDC = CreateCompatibleDC(hDC);

    if(bFullScreen)
    {
        mmpInfo.dwMovieExtentX = 0;
        mmpInfo.dwMovieExtentY = 0;
    }
    else
        mmpGetMovieInfo(idMovie, &mmpInfo);

    /* Create bitmap, sized to match the movie frame area
     */
    nX = (int)(mmpInfo.dwMovieExtentX ? mmpInfo.dwMovieExtentX :
                                        GetSystemMetrics (SM_CXSCREEN));
    nY = (int)(mmpInfo.dwMovieExtentY ? mmpInfo.dwMovieExtentY :
                                        GetSystemMetrics (SM_CYSCREEN));
    hBitmap = CreateCompatibleBitmap(hDC, nX, nY);

    /* Select the bitmap into the memory DC and draw the movie frame
     * into the bitmap
     */
    SelectObject(hMemoryDC, hBitmap);
    mmpUpdate(idMovie, hMemoryDC, NULL);

    /* Place the bitmap and logical palette in the Clipboard
     */
    if(OpenClipboard(hWnd))
    {
        EmptyClipboard();
        SetClipboardData(CF_BITMAP, hBitmap);
        SetClipboardData(CF_PALETTE, hClipPalette);
        CloseClipboard();
    }
    else
    {
        MessageBeep(MB_ICONEXCLAMATION);
        MessageBox(hWnd, "Could not open Clipboard.", szAppName,
                    MB_OK | MB_ICONEXCLAMATION );
    }

    DeleteDC(hMemoryDC);
    ReleaseDC(hWnd, hDC);
}



/* CopyPalette -- This function makes a copy of a GDI logical palette.
 *
 * Params:        hPal -- Handle of the logical palette to copy
 *
 * Returns:       Handle to logical palette if successful.
 *                NULL if unsuccessful.
 */
HPALETTE CopyPalette(HPALETTE hpal)
{
    PLOGPALETTE ppal;
    int nNumEntries;

    if(!hpal)
        return NULL;

    // Get number of entries in palette
    GetObject(hpal, sizeof(int), (LPSTR)&nNumEntries);

    if(nNumEntries == 0)
        return NULL;

    // Allocate palette buffer
    ppal = (PLOGPALETTE)LocalAlloc(LPTR, sizeof(LOGPALETTE) +
                nNumEntries * sizeof(PALETTEENTRY));

    if(!ppal)
        return NULL;

    ppal->palVersion    = PALVERSION;
    ppal->palNumEntries = nNumEntries;

    GetPaletteEntries(hpal, 0, nNumEntries, ppal->palPalEntry);

    hpal = CreatePalette(ppal);

    LocalFree((HANDLE)ppal);
    return hpal;
}


/* Strings used by WriteFlags() and ReadFlags().
 */
char szIniName[] =              "MMPLAY.INI";
char szIniX[] =                 "Window left";
char szIniY[] =                 "Window top";
char szIniWidth[] =             "Window width";
char szIniHeight[] =            "Window height";
char szIniLoadNoStatic[] =      "Use static colors for movie palette";
char szIniLoadOnDemand[] =      "Load bitmaps on demand";
char szIniLoadExpandDibs[] =    "Expand bitmaps while loading";
char szIniMute[] =              "Mute sound";
char szIniRepeat[] =            "Repeat movie";
char szIniFullScreen[] =        "Full screen playback";
char szIniDirectory[] =         "Default directory";

/* WriteFlags - Writes the current window dimensions,
 *              option flags, and working directory to MMPLAY.INI.
 *
 * Params:  None
 *
 * Return:  void
 */
void WriteFlags(void)
{
    RECT rc;
    static char szYes[] = "yes";
    static char szNo[] = "no";

    GetWindowRect(hMainWnd, &rc);

    wsprintf(szTextBuffer, "%d", rc.left);
    WritePrivateProfileString(szAppName, szIniX, szTextBuffer, szIniName);
    wsprintf(szTextBuffer, "%d", rc.top);
    WritePrivateProfileString(szAppName, szIniY, szTextBuffer, szIniName);
    wsprintf(szTextBuffer, "%d", rc.right-rc.left);
    WritePrivateProfileString(szAppName, szIniWidth, szTextBuffer, szIniName);
    wsprintf(szTextBuffer, "%d", rc.bottom-rc.top);
    WritePrivateProfileString(szAppName, szIniHeight, szTextBuffer, szIniName);
    WritePrivateProfileString(szAppName, szIniLoadNoStatic,
                              bLoadNoStatic ? szYes : szNo, szIniName);
    WritePrivateProfileString(szAppName, szIniMute,
                              bMute ? szYes : szNo, szIniName);
    WritePrivateProfileString(szAppName, szIniRepeat,
                              bRepeat ? szYes: szNo, szIniName);
    WritePrivateProfileString(szAppName, szIniFullScreen,
                              bFullScreen ? szYes : szNo, szIniName);
    WritePrivateProfileString(szAppName, szIniLoadOnDemand,
                              bLoadOnDemand ? szYes : szNo, szIniName);
    WritePrivateProfileString(szAppName, szIniLoadExpandDibs,
                              bLoadExpandDibs ? szYes : szNo, szIniName);

    getcwd(szTextBuffer, BUFSIZE);
    WritePrivateProfileString(szAppName, szIniDirectory,
                              szTextBuffer, szIniName);
}


/* ReadFlags - Reads the window dimensions, option flags,
 *             and working directory from MMPLAY.INI.
 *
 * Params:  None
 *
 * Return:  void
 */
void ReadFlags(void)
{
    nX = GetPrivateProfileInt(szAppName, szIniX, CW_USEDEFAULT, szIniName);
    nY = GetPrivateProfileInt(szAppName, szIniY, CW_USEDEFAULT, szIniName);
    nWidth = GetPrivateProfileInt(szAppName, szIniWidth,
                                  CW_USEDEFAULT, szIniName);
    nHeight = GetPrivateProfileInt(szAppName, szIniHeight,
                                   CW_USEDEFAULT, szIniName);

    GetPrivateProfileString(szAppName, szIniMute, "n", szTextBuffer, 2, szIniName);
    bMute = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniRepeat, "n", szTextBuffer, 2, szIniName);
    bRepeat = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniFullScreen, "n", szTextBuffer, 2, szIniName);
    bFullScreen = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniLoadNoStatic, "n", szTextBuffer, 2, szIniName);
    bLoadNoStatic = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniLoadOnDemand, "n", szTextBuffer, 2, szIniName);
    bLoadOnDemand = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniLoadExpandDibs, "n", szTextBuffer, 2, szIniName);
    bLoadExpandDibs = (szTextBuffer[0] == 'y') || (szTextBuffer[0] == 'Y');

    GetPrivateProfileString(szAppName, szIniDirectory, ".", szTextBuffer, BUFSIZE, szIniName);
    chdir(szTextBuffer);
}
