/* SnapShot.c

     (C) Copyright Microsoft Corp. 1991.  All rights reserved.

     You have a royalty-free right to use, modify, reproduce and 
     distribute the Sample Files (and/or any modified version) in 
     any way you find useful, provided that you agree that 
     Microsoft has no warranty obligations or liability for any 
     Sample Application Files which are modified. 

 */

// the correct definition...
unsigned int    far pascal GetSystemPaletteUse(unsigned int);


// see the WDB sample code for debugging ideas.
#define dprintf	


#include <windows.h>
#include "SnapShot.h"


/* constants */
#define APPNAME "SnapShot"


/* globals */
char    gszAppName[] = APPNAME;        // for title bar etc.

HANDLE        ghInst;                // program instance handle

int    gcxScreen, gcyScreen;        // screen size


/* prototypes */
BOOL FAR PASCAL _export AppAbout ( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
BOOL PASCAL AppInit(HANDLE hInst, HANDLE hPrev);
int    PASCAL WinMain(HANDLE hInst, HANDLE hPrev, LPSTR lpszCmdLine, int iCmdShow);
LONG FAR PASCAL _export AppWndProc(HWND hwnd, UINT uiMessage, WPARAM wParam, LPARAM lParam);
HPALETTE NEAR PASCAL CreateSystemPalette(HDC hdcScreen);
void NEAR PASCAL SnapShot(HWND hwnd);



/* AppAbout(hDlg, uiMessage, wParam, lParam)
 *
 * This function handles messages belonging to the "About" dialog box.
 * The only message that it looks for is WM_COMMAND, indicating the use
 * has pressed the "OK" button.  When this happens, it takes down
 * the dialog box.
 */
BOOL FAR PASCAL _export           // TRUE iff message has been processed
AppAbout ( HWND hwnd, UINT uiMessage, WPARAM wParam, LPARAM lParam)
{
    char ach[128];
    
    switch (uiMessage)
    {
	case WM_INITDIALOG:
	    // nothing to initialize 
	    wsprintf(ach,"Compiled: %s %s",(LPSTR)__DATE__,(LPSTR)__TIME__);
	    SetDlgItemText(hwnd,IDD_TEXT,ach);
	    break;

	case WM_COMMAND:
	    switch(wParam)
	    {
		case IDOK:
		    EndDialog(hwnd, TRUE);
		    break;
		    
		default:
		    return FALSE;
	    }
	    break;
	    
	default:
	    return FALSE;
    }
    return TRUE;
}


/* AppInit(hInst, hPrev)
 *
 * This is called when the application is first loaded into memory.
 * It performs all initialization that doesn't need to be done once
 * per instance.
 */
BOOL PASCAL            // returns TRUE iff successful
AppInit(HANDLE hInst, HANDLE hPrev)
{
    WNDCLASS    wndclass;

#ifdef DEBUG
//    wpfGetDebugLevel(gszAppName);
#endif

    if (!hPrev)
    {
/* Register a class for the main application window */
        wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
        wndclass.hIcon = LoadIcon(hInst, "AppIcon");
        wndclass.lpszMenuName = "AppMenu";
        wndclass.lpszClassName = gszAppName;
        wndclass.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
        wndclass.hInstance = hInst;
        wndclass.style = CS_BYTEALIGNCLIENT | CS_VREDRAW | CS_HREDRAW;
        wndclass.lpfnWndProc = AppWndProc;
        wndclass.cbWndExtra = 0;
        wndclass.cbClsExtra = 0;

        if (!RegisterClass(&wndclass))
            return FALSE;
    }

    return TRUE;
}


/* WinMain(hInst, hPrev, lpszCmdLine, cmdShow)
 * 
 * The main procedure for the App.  After initializing, it just goes
 * into a message-processing loop until it gets a WM_QUIT message
 * (meaning the app was closed).
 */
int    PASCAL            // returns exit code specified in WM_QUIT
WinMain(HANDLE hInst, HANDLE hPrev, LPSTR lpszCmdLine, int iCmdShow)
{
    MSG        msg;        // message from queue

    HWND        hwnd;        // handle to app's window

    HMENU        hmenuSystem;    // system menu

    RECT        rc;

/* save instance handle for dialog boxes */
    ghInst = hInst;

/* call initialization procedure */
    if (!AppInit(hInst, hPrev))
        return FALSE;

/* define the app window style */
#define APPWINDOWSTYLE        \
        (WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU)

/* get the screen size */
    gcxScreen = GetSystemMetrics(SM_CXSCREEN);
    gcyScreen = GetSystemMetrics(SM_CYSCREEN);

/* size the app window to just fit around the edit window */
    rc.left = 0;
    rc.right = gcxScreen / 4;
    rc.top = 0;
    rc.bottom = 0;
    AdjustWindowRect(&rc, APPWINDOWSTYLE, TRUE);

/* create the application's window */
    hwnd = CreateWindow
        (
    gszAppName,         // window class
        gszAppName,         // window caption
        APPWINDOWSTYLE,         // window style
            gcxScreen - (rc.right - rc.left),     // initial x-coord
            gcyScreen - (rc.bottom - rc.top),     // initial y-coord
        rc.right - rc.left,             // initial width
        rc.bottom - rc.top,             // initial height
        NULL,             // parent window handle
        NULL,             // window menu handle
        hInst,             // program instance handle
        NULL            // create parameters
        );
    ShowWindow(hwnd, iCmdShow);

/* append "Show Clipboard" and "About" menu item to system menu */
    hmenuSystem = GetSystemMenu(hwnd, FALSE);
    AppendMenu(hmenuSystem, MF_SEPARATOR, 0, NULL);
    AppendMenu(hmenuSystem, MF_STRING, IDM_SHOWCLIP, "S&how Clipboard");
    AppendMenu(hmenuSystem, MF_STRING, IDM_ABOUT, "&About " APPNAME "...");

/* get messages from event queue and dispatch them */
    while (GetMessage(&msg, NULL, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    return msg.wParam;
}


/* AppWndProc(hwnd, uiMessage, wParam, lParam)
 * 
 * The window proc for the app's main window.  This processes all
 * of the parent window's messages.
 */
LONG FAR PASCAL _export        // returns 0 iff processed message
AppWndProc(HWND hwnd, UINT uiMessage, WPARAM wParam, LPARAM lParam)
{
    FARPROC        fpfn;
    PAINTSTRUCT    ps;

    switch (uiMessage)
    {

	case WM_COMMAND:

	    switch (wParam)
	    {

		case IDM_NOW:

		    ShowWindow(hwnd, SW_HIDE);
		    SnapShot(hwnd);
		    ShowWindow(hwnd, SW_NORMAL);
		    break;

		case IDM_5SEC:

		    SetTimer(hwnd, 1, 5000, NULL);
		    ShowWindow(hwnd, SW_HIDE);
		    break;

	    }
	    return 0L;

	case WM_TIMER:

	    KillTimer(hwnd, 1);
	    MessageBeep(0);
	    SnapShot(hwnd);
	    MessageBeep(0);
	    ShowWindow(hwnd, SW_NORMAL);
	    break;

	case WM_SYSCOMMAND:

	    switch (wParam)
	    {

		case IDM_SHOWCLIP:

		    WinExec("CLIPBRD.EXE", SW_NORMAL);
		    break;

		case IDM_ABOUT:

		/* request to display "About" dialog box */
		fpfn = MakeProcInstance((FARPROC) AppAbout, ghInst);
		DialogBox(ghInst, MAKEINTRESOURCE(ABOUTBOX),
			hwnd, fpfn);
		FreeProcInstance(fpfn);
		break;

	    }
	    break;

	case WM_DESTROY:

	    /* exit this program */
	    PostQuitMessage(0);
	    break;

    }

    return DefWindowProc(hwnd, uiMessage, wParam, lParam);
}


/* hpal = CreateSystemPalette(hdcScreen)
 *
 * Return a palette which represents the system palette.  By selecting this
 * palette into a screen DC and realizing the palette, you can BitBlt from
 * the screen DC to a memory bitmap to get a correct screen shot.
 *
 * <hdcScreen> is a DC onto the screen.
 *
 * On error (e.g. out of memory), NULL is returned.
 */
HPALETTE NEAR PASCAL
CreateSystemPalette(HDC hdcScreen)        // DC onto the screen

{
    int    iSizePalette;        // size of entire palette

    int    iFixedPalette;        // number of reserved colors

    NPLOGPALETTE    pLogPal = NULL;
    HPALETTE    hpal = NULL;
    int    i;

/* calculate <iSizePalette> and <iFixedPalette> */
    if (GetSystemPaletteUse(hdcScreen) == SYSPAL_STATIC)
        iFixedPalette = GetDeviceCaps(hdcScreen, NUMCOLORS);
    else
        iFixedPalette = 2;

    iSizePalette = GetDeviceCaps(hdcScreen, SIZEPALETTE);
    dprintf("CreateSystemPalette: iFixedPalette=%d, iSizePalette=%d\n",
        iFixedPalette, iSizePalette);

/* create a logical palette containing the system colors;
     * this palette has all entries except fixed (system) colors
     * flagged as PC_NOCOLLAPSE
     */
    pLogPal = (NPLOGPALETTE) LocalAlloc(LMEM_FIXED, sizeof(LOGPALETTE)
         + iSizePalette * sizeof(PALETTEENTRY));

    dprintf("CreateSystemPalette: pLogPal=0x%04x\n", pLogPal);

    if (pLogPal == NULL)
        goto CreateSystemPalette_ERROR;

    pLogPal->palVersion = 0x300;
    pLogPal->palNumEntries = iSizePalette;
    GetSystemPaletteEntries(hdcScreen, 0, iSizePalette,
        pLogPal->palPalEntry);

    for (i = 0; i < iSizePalette; i++)
        pLogPal->palPalEntry[i].peFlags = 
            ((i >= iFixedPalette / 2) && 
            (i < iSizePalette - iFixedPalette / 2))
         ? PC_NOCOLLAPSE : 0;

    hpal = CreatePalette((LPLOGPALETTE) pLogPal);
    dprintf("CreateSystemPalette: hpal=0x%04x\n", hpal);

CreateSystemPalette_ERROR:

    if (pLogPal != NULL)
        LocalFree((HANDLE) pLogPal);

    return hpal;
}


void NEAR PASCAL
SnapShot(HWND hwnd)            // owner of clipboard data
{
    HDC        hdcScreen = NULL;
    HBITMAP        hbitmap = NULL;
    HDC        hdcBitmap = NULL;
    HPALETTE    hpal = NULL;
    HPALETTE    hpalScreenOld;

    hdcScreen = GetDC(NULL);
    dprintf("SnapShot: hdcScreen=0x%04x\n", hdcScreen);

/* create a memory bitmap */
    hdcBitmap = CreateCompatibleDC(hdcScreen);
    hbitmap = CreateCompatibleBitmap(hdcScreen, gcxScreen, gcyScreen);
    dprintf("SnapShot: hdcBitmap=0x%04x, hbitmap=0x%04x\n",
        hdcBitmap, hbitmap);
    if (hbitmap == NULL)
    {
//        Error(IDS_OUTOFMEMORY);
        goto SnapShot_ERROR;
    }
    SelectObject(hdcBitmap, hbitmap);

/* create a palette corresponding to the current physical palette */
    hpal = CreateSystemPalette(hdcScreen);
    if (hpal == NULL)
    {
//        Error1(IDS_INTERNALERROR, __LINE__);
        goto SnapShot_ERROR;
    }

/* copy the screen to the bitmap */
    hpalScreenOld = SelectPalette(hdcScreen, hpal, FALSE);
    RealizePalette(hdcScreen);
    BitBlt(hdcBitmap, 0, 0, gcxScreen, gcyScreen, hdcScreen, 0, 0, SRCCOPY);
    SelectPalette(hdcScreen, hpalScreenOld, FALSE);

/* give the bitmap and palette to the clipboard */
    if (OpenClipboard(hwnd))
    {
        EmptyClipboard();
        SetClipboardData(CF_BITMAP, hbitmap);
        SetClipboardData(CF_PALETTE, hpal);
        CloseClipboard();
        hbitmap = NULL;        // bitmap is now owned by clipboard

        hpal = NULL;        // palette is now owned by clipboard
    }
    else
        dprintf("SnapShot: can't open clipboard\n");

/* fall through */

SnapShot_ERROR:

/* clean up */
    if (hdcBitmap != NULL)
        DeleteDC(hdcBitmap), hdcBitmap = NULL;
    if (hbitmap != NULL)
        DeleteObject(hbitmap), hbitmap = NULL;
    if (hpal != NULL)
        DeleteObject(hpal), hpal = NULL;
    if (hdcScreen != NULL)
        ReleaseDC(NULL, hdcScreen), hdcScreen = NULL;

    dprintf("SnapShot: done.\n");
}


