/*
  File: 4D.h

  Purpose: definitions for a 4D homogeneous coordinate manipulation package
 */

#ifndef MAKELONG
    #define NEAR    near
    #define FAR     far
    #define PASCAL  pascal

    typedef unsigned char	BYTE;
    typedef unsigned short	WORD;
    typedef long		LONG;
    typedef unsigned long	DWORD;
    typedef int 		BOOL;
    typedef char		*PSTR;
    typedef char NEAR		*NPSTR;
    typedef char FAR		*LPSTR;
    typedef int FAR		*LPINT;

    #define MAKELONG(a, b)	((LONG)(((WORD)(a)) | ((DWORD)((WORD)(b))) << 16))
    #define LOWORD(l)		((WORD)(l))
    #define HIWORD(l)		((WORD)(((DWORD)(l) >> 16) & 0xFFFF))
    #define LOBYTE(w)		((BYTE)(w))
    #define HIBYTE(w)		(((WORD)(w) >> 8) & 0xFF)
#endif

typedef unsigned long ULONG;    /* ul  */
typedef long          FIXED4D;    /* fx  */
typedef FIXED4D       *PFIXED4D;    /* pfx */

#define MAKEFIXED4D(intpart,fractpart) MAKELONG(fractpart,intpart)
#define MAKEFX(i,f)                  MAKEFIXED4D(i,(WORD)(65536L * (DWORD)(f) / 10000L))
#define FX(i)			     MAKEFX(i,0)
#define FXFRAC(fx)		     (LOWORD(fx))
#define FXDFRAC(fx)                  (WORD)((DWORD)FXFRAC(fx) * 10000 / 65536L)
#define FXINT(fx)		     ((int)HIWORD(fx))

typedef struct P4D { FIXED4D x,y,z,w; } POINTFX4D, *PPOINTFX4D, far *LPPOINTFX4D;
typedef struct V4D { FIXED4D x,y,z,w; } VECTORFX;
typedef struct T4D { FIXED4D T[4][4]; } MATRIXFX;

extern POINTFX4D	CreatePoint4D(FIXED4D,FIXED4D,FIXED4D,FIXED4D);
extern POINTFX4D  PxT4D(POINTFX4D, MATRIXFX*);
extern void     TransformPoints(MATRIXFX*,POINTFX4D FAR*,POINTFX4D FAR*,int);
extern MATRIXFX Matrix4D();
extern MATRIXFX TxT4D(MATRIXFX*,MATRIXFX*);
extern MATRIXFX Transpose4D(MATRIXFX*);
extern MATRIXFX Identity4D(void);
extern MATRIXFX Translate4D(VECTORFX);
extern MATRIXFX Scale4D(POINTFX4D);
extern MATRIXFX UniformScale4D(FIXED4D);
extern MATRIXFX RotateX(FIXED4D);
extern MATRIXFX RotateY(FIXED4D);
extern MATRIXFX RotateZ(FIXED4D);
extern MATRIXFX Perspective4D(FIXED4D);
extern VECTORFX CreateVector4D(FIXED4D,FIXED4D,FIXED4D);
extern VECTORFX VxT4D(VECTORFX,MATRIXFX*);

extern VECTORFX Cross4D(VECTORFX,VECTORFX);
extern FIXED4D	Dot4D(VECTORFX,VECTORFX);
extern VECTORFX Normalize(VECTORFX);
extern POINTFX4D	PVadd(POINTFX4D,VECTORFX);
extern POINTFX4D	Ratio4D(POINTFX4D,POINTFX4D,FIXED4D);
extern VECTORFX Pdiff(POINTFX4D,POINTFX4D);
extern VECTORFX Vscale(FIXED4D,VECTORFX);
extern VECTORFX Vadd(VECTORFX,VECTORFX);
extern VECTORFX Vsub(VECTORFX,VECTORFX);
extern FIXED4D	Vmag(VECTORFX);

extern FIXED4D	atofx(char *);

/*
 *  Convert a VECTORFX to a point
 */
#define PV(v)  (*(POINTFX4D  *)&(v))
/*
 *  Convert a POINTFX4D to a VECTORFX
 */
#define VP(p)  (*(VECTORFX *)&(p))

#define PrintTransform(t) PrintMatrix(&(t).T)

/* #define PrintFx(fx)   printf("%5d.%04u",FXINT(fx),FXDFRAC(fx)) */

extern void PrintPoint	(POINTFX4D);
extern void PrintVector (VECTORFX);
extern void PrintMatrix (MATRIXFX *);
extern void PrintFx     (FIXED4D fx);


#define POINTX(p) ((p).x)
#define POINTY(p) ((p).y)
#define POINTZ(p) ((p).z)
#define POINTW(p) ((p).w)

#define VECTORX(v) ((v).x)
#define VECTORY(v) ((v).y)
#define VECTORZ(v) ((v).z)


extern POINTFX4D Origin;
extern FIXED4D   epsilon;

#define ONE   FX(1)
#define ZERO  FX(0)
#define FXINF FX(0x7FFF)

/* declarations for MATH.ASM */

extern void  FAR PASCAL fxCross4D (VECTORFX FAR *, VECTORFX, VECTORFX);

extern FIXED4D FAR PASCAL fxsqrt  (FIXED4D);
extern ULONG FAR PASCAL ulsqrt  (ULONG);
extern ULONG FAR PASCAL lDot4D  (VECTORFX, VECTORFX);
extern FIXED4D FAR PASCAL fxLength3D (FIXED4D,FIXED4D,FIXED4D);

//extern FIXED4D FAR PASCAL square_root (FIXED4D);
extern FIXED4D FAR PASCAL multiply    (FIXED4D,FIXED4D);
extern FIXED4D FAR PASCAL divide	    (FIXED4D,FIXED4D);
extern FIXED4D FAR PASCAL fxRound     (FIXED4D);
extern FIXED4D FAR PASCAL CalcSine    (FIXED4D);
extern FIXED4D FAR PASCAL CalcCosine  (FIXED4D);

//#define sin(fx)  CalcSine(fx)
//#define cos(fx)  CalcCosine(fx)
//#define sqrt(fx) square_root(fx)

#define fxsin(fx)  CalcSine(fx)
#define fxcos(fx)  CalcCosine(fx)
//#define fxsqrt(fx) square_root(fx)
#define fxmul(x,y) multiply(x,y)
#define fxdiv(x,y) divide(x,y)
#define fxadd(x,y) ((x) + (y))
#define fxsub(x,y) ((x) - (y))

#define FXMUL(x,y)	(multiply(x,y))
#define FXDIV(x,y)	(divide(x,y))


/*
** Conceptually, a ray is a half-infinite line eminating from
** a point in a particular direction.  In this module a ray
** can be thought of as a pair of points P0 and P1.
** The ray emanates from P0 in the direction of the vector P1-P0.
*/

typedef struct {
      POINTFX4D P0, P1;
}     RAYFX;

extern RAYFX CreateRay	(POINTFX4D,POINTFX4D);
extern RAYFX RxT4D	(RAYFX,MATRIXFX*);
extern void  PrintRay	(RAYFX);


/*
 * Transform Stuff
 */

#define END        0
#define SCALE      1
#define TRANSLATE  2
#define ROTATEX    3
#define ROTATEY    4
#define ROTATEZ    5
#define PROJECT    6
#define VIEWBOX    7

#define PROJECT4D    10
#define SCALE4D      11
#define TRANSLATE4D  12

struct TRANSFORM_STRUCT {
    MATRIXFX T;
    MATRIXFX Tinv;
};

typedef struct TRANSFORM_STRUCT TRANSFORM;

extern TRANSFORM *NewTransform	      (void);
extern TRANSFORM *MakeTM	      ();

extern int	  FreeTransform       (TRANSFORM *);
extern int	  SetTransform	      (TRANSFORM *,...);
extern int	  AppendTransform     (TRANSFORM *,...);
extern void       PreposeTransform    (TRANSFORM *,TRANSFORM *);
extern void       ComposeTransform    (TRANSFORM *,TRANSFORM *);
extern void	  InitTransform       (TRANSFORM *);
extern int	  SetCTM	      (TRANSFORM *);
extern int	  TransformPoint      (POINTFX4D);
extern int	  UnTransformPoint    (POINTFX4D);

/*-----------------------------------------------------------------------------
  Global variables and macros for transforming points.
-----------------------------------------------------------------------------*/

extern	TRANSFORM *CTM;    /* current transform */
#define SetCTM(T)	    (CTM = &(T))
#define TransformPoint(P)   (PxT4D ((P),&CTM->T))
#define UnTransformPoint(P) (PxT4D ((P),&CTM->Tinv))
#define TransformRay(R)     (RayxT4D ((R),&CTM->T))
#define UnTransformRay(R)   (RayxT4D ((R),&CTM->Tinv))
/*
 *  Inplace transform of a ray
 */
#define TRay(R)   ((R).P0=  TransformPoint((R).P0),(R).P1=  TransformPoint((R).P1))
#define UnTRay(R) ((R).P0=UnTransformPoint((R).P0),(R).P1=UnTransformPoint((R).P1))

/*
 * Example:
 *
 *  TRANSFORM T;
 *  POINT     P = {10.0,10.0,10.0,1.0};
 *
 *  MakeTransform (&T,
 *		     ,TRANSLATE,  1.0, 1.0, 1.0
 *		     ,SCALE,	  1.0, 1.0, 1.0
 *		     ,PROJECT,	  1.0, 1.0, 1.0
 *		     ,		  1.0, 1.0, 1.0
 *		     ,		  1.0, 1.0, 1.0
 *		     ,ROTATEX,	  90.0
 *		     ,ROTATEY,	  90.0
 *		     ,ROTATEZ,	  90.0
 *		     ,TRANSLATE4D,1.0, 1.0, 1.0, 1.0
 *		     ,SCALE4D,	  1.0, 1.0, 1.0, 1.0
 *		     ,PROJECT4D,  1.0, 1.0, 1.0, 1.0
 *		     ,		  1.0, 1.0, 1.0, 1.0
 *		     ,		  1.0, 1.0, 1.0, 1.0
 *		     ,END	     <= Dont forget the END!!!
 *		   );
 *
 *  SetCTM (T);
 *  P = TransformPoint (P);
 *  P = UnTransformPoint (P);
 *
 */
