/*----------------------------------------------------------------------------*\
 *
 *  MCIPlay
 *
 *    Sample app showing the use of the 'MCIWnd' window class
 *
 *    the MCIWnd window class is a window class for controling MCI devices
 *    MCI devices include, wave files, midi files, AVI Video, cd audio,
 *    vcr, video disc, and others..
 *
 *    the easiest use of the MCIWnd class is like so:
 *
 *          hwnd = MCIWndCreate(hwndParent, hInstance, 0, "smag.wav");
 *          ...
 *          MCIWndPlay(hwnd);
 *          ....
 *          MCIWndDestroy(hwnd);
 *
 *    this will create a window with a play/pause, stop and a playbar
 *    and start the wave file playing.
 *
 *    mciwnd.h defines macros for all the most common MCI commands, but
 *    any string command can be used if needed.
 *
 *          MCIWndSendString(hwnd, "set hue to bright");
 *
 *
 *    most of this app deals with the overhead of MDI, the real work happens
 *    mostly on WM_COMMAND (look for MCIWnd for real code)
 *
 *		case IDM_PLAY:
 *		    MCIWndPlay(hwndMovie);
 *		    break;
 *
 *    this MDI app handles palettes by letting the topmost MDI child take
 *    control of the palette, all others paint in the background.  what
 *    realy happens is the WM_QUERYNEWPALETTE message is only sent to
 *    the top most MDI window.
 *
 *    In order to open a non-file device (like CDAUDIO) with this sample
 *    application, you will need to launch it from the command line (eg.
 *    "mciplay cdaudio" since the file open dialog will not let you type in
 *    the device name if it does not exist as a filename on your drive.
 *
 *    (C) Copyright Microsoft Corp. 1991, 1992, 1993.  All rights reserved.
 *
 *    You have a royalty-free right to use, modify, reproduce and
 *    distribute the Sample Files (and/or any modified version) in
 *    any way you find useful, provided that you agree that
 *    Microsoft has no warranty obligations or liability for any
 *    Sample Application Files.
 *
 *    If you did not get this from Microsoft Sources, then it may not be the
 *    most current version.  This sample code in particular will be updated
 *    and include more documentation.
 *
 *    Sources are:
 *       CompuServe: WINSDK forum, MDK section.
 *       Anonymous FTP from ftp.uu.net vendor\microsoft\multimedia
 *
 *----------------------------------------------------------------------------*/

#include <windows.h>
#include <commdlg.h>
#include "mciwnd.h"

#include "mciplay.h"

//
// constants
//
#define ACH_BUFFER_LENGTH   128


/*-------------------------------------------------------------------------*\
|                                                                          |
|   g l o b a l   v a r i a b l e s                                        |
|                                                                          |
\*------------------------------------------------------------------------*/

char    szAppName[]  = "MCIPlay";   /* change this to your app's name */
char    szDocClass[] = MCIWND_WINDOW_CLASS;

char    szOpenFilter[] = "Video Files\0*.avi\0"
                         "Wave Files\0*.wav\0"
                         "Midi Files\0*.mid; *.rmi\0"
                         "All Files\0*.*\0";

HANDLE  hInstApp;                   /* Instance handle */
HACCEL  hAccelApp;
HWND    hwndApp;                    /* Handle to parent window */
HWND    hwndMdi;                    /* Handle to MCI client window */

OFSTRUCT     of;
OPENFILENAME ofn;
char         achFileName[ACH_BUFFER_LENGTH];

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

long FAR PASCAL _export AppWndProc(HWND, UINT, WPARAM, LPARAM);
int ErrMsg (LPSTR sz,...);

HWND mdiCreateDoc(LPSTR szClass, LPSTR szTitle, LPARAM l);

/*----------------------------------------------------------------------------*\
|   AppAbout( hDlg, msg, wParam, lParam )                                      |
|                                                                              |
|   Description:                                                               |
|       This function handles messages belonging to the "About" dialog box.    |
|       The only message that it looks for is WM_COMMAND, indicating the use   |
|       has pressed the "OK" button.  When this happens, it takes down         |
|       the dialog box.                                                        |
|                                                                              |
|   Arguments:                                                                 |
|       hDlg            window handle of about dialog window                   |
|       msg             message number                                         |
|       wParam          message-dependent                                      |
|       lParam          message-dependent                                      |
|                                                                              |
|   Returns:                                                                   |
|       TRUE if message has been processed, else FALSE                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
BOOL FAR PASCAL _export AppAbout(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
        case WM_COMMAND:
            EndDialog(hwnd,TRUE);
            return TRUE;

        case WM_INITDIALOG:
	    return TRUE;
    }
    return FALSE;
}

/*----------------------------------------------------------------------------*\
|   AppInit ( hInstance, hPrevInstance )				       |
|                                                                              |
|   Description:                                                               |
|       This is called when the application is first loaded into               |
|       memory.  It performs all initialization that doesn't need to be done   |
|       once per instance.                                                     |
|                                                                              |
|   Arguments:                                                                 |
|	hPrevInstance	instance handle of previous instance		       |
|       hInstance       instance handle of current instance                    |
|                                                                              |
|   Returns:                                                                   |
|       TRUE if successful, FALSE if not                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
BOOL AppInit(HANDLE hInst, HANDLE hPrev, LPSTR szCmd, int sw)
{
    WNDCLASS    cls;

    /* Save instance handle for DialogBox */
    hInstApp = hInst;

    hAccelApp = LoadAccelerators(hInstApp, "AppAccel");

    if (!hPrev) {
        cls.hCursor        = LoadCursor(NULL,IDC_ARROW);
        cls.hIcon          = LoadIcon(hInst,"AppIcon");
        cls.lpszMenuName   = "AppMenu";
        cls.lpszClassName  = szAppName;
        cls.hbrBackground  = (HBRUSH)COLOR_APPWORKSPACE+1;
        cls.hInstance      = hInst;
        cls.style          = 0;
        cls.lpfnWndProc    = (WNDPROC)AppWndProc;
        cls.cbClsExtra     = 0;
	cls.cbWndExtra	   = 0;

        if (!RegisterClass(&cls))
            return FALSE;
    }

    //
    // because this app uses CreateWindow() to make 'MCIWnd' windows
    // we need to register the window class by hand, if you use MCIWndCreate()
    // you dont need to do this.
    //
    MCIWndRegisterClass(hInst);

    hwndApp = CreateWindow(szAppName,szAppName,
	       WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
	       CW_USEDEFAULT,0,
	       CW_USEDEFAULT,0,
	       (HWND)NULL,	  /* no parent */
	       (HMENU)NULL,	  /* use class menu */
               (HANDLE)hInst,     /* handle to window instance */
	       (LPSTR)NULL	  /* no params to pass on */
	     );

    /* Make window visible according to the way the app is activated */
    ShowWindow(hwndApp,sw);

    if (szCmd && szCmd[0])
        mdiCreateDoc(szDocClass, szCmd, 0);

    return TRUE;
}

/*----------------------------------------------------------------------------*\
|   WinMain( hInstance, hPrevInstance, lpszCmdLine, cmdShow )                  |
|                                                                              |
|   Description:                                                               |
|       The main procedure for the App.  After initializing, it just goes      |
|       into a message-processing loop until it gets a WM_QUIT message         |
|       (meaning the app was closed).                                          |
|                                                                              |
|   Arguments:                                                                 |
|       hInstance       instance handle of this instance of the app            |
|       hPrevInstance   instance handle of previous instance, NULL if first    |
|       lpszCmdLine     ->null-terminated command line                         |
|       cmdShow         specifies how the window is initially displayed        |
|                                                                              |
|   Returns:                                                                   |
|       The exit code as specified in the WM_QUIT message.                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR szCmdLine, int sw)
{
    MSG     msg;

    if (!AppInit(hInstance,hPrevInstance,szCmdLine,sw))
       return FALSE;

    /*
     * Polling messages from event queue
     */
    for (;;)
    {
        if (PeekMessage(&msg, NULL, 0, 0,PM_REMOVE))
        {
            if (msg.message == WM_QUIT)
                break;

            if (hAccelApp && TranslateAccelerator(hwndApp, hAccelApp, &msg))
                continue;

            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
        else
        {
            // idle time here, DONT BE A PIG!
            WaitMessage();
        }
    }

    return msg.wParam;
}

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

BOOL fDialog(HWND hwnd,int id,FARPROC fpfn)
{
    BOOL	f;
    HANDLE	hInst;

    hInst = (HINSTANCE)GetWindowWord(hwnd,GWW_HINSTANCE);
    fpfn  = MakeProcInstance(fpfn,hInst);
    f = DialogBox(hInst,MAKEINTRESOURCE(id),hwnd,(DLGPROC)fpfn);
    FreeProcInstance (fpfn);
    return f;
}

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

#define mdiGetCreateParam(lParam) \
    (((LPMDICREATESTRUCT)(((LPCREATESTRUCT)lParam)->lpCreateParams))->lParam)

/*----------------------------------------------------------------------------*\
|   mdiCreateChild()							       |
|									       |
|   Description:                                                               |
|                                                                              |
|   Arguments:                                                                 |
|                                                                              |
|   Returns:                                                                   |
|	HWND if successful, NULL otherwise				       |
|									       |
\*----------------------------------------------------------------------------*/

HWND mdiCreateChild(
    HWND  hwndMdi,
    LPSTR szClass,
    LPSTR szTitle,
    DWORD dwStyle,
    int   x,
    int   y,
    int   dx,
    int   dy,
    WORD  sw,
    HMENU hmenu,
    LPARAM l)
{
    //
    //	this is where we create a MCIWnd, by default a MCIWnd will try to
    //	open it's window text as a filename or device.
    //
    //	so the window text is the file name.
    //
    //	another way to open a file/device is to use MCIWndOpen()
    //
    //	    hwnd = MCIWndCreate(hwndParent, hInstance, 0, "");
    //	    MCIWndOpen(hwnd, "cdaudio", 0);
    //

    MDICREATESTRUCT mdics;

    mdics.szClass   = szClass;
    mdics.szTitle   = szTitle;
    mdics.hOwner    = (HINSTANCE)GetWindowWord(hwndMdi, GWW_HINSTANCE);
    mdics.x         = x;
    mdics.y         = y;
    mdics.cx        = dx;
    mdics.cy        = dy;
    mdics.style     = dwStyle;
    mdics.lParam    = l;

    return (HWND)SendMessage(hwndMdi,WM_MDICREATE,0,(LONG)(LPVOID)&mdics);
}

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

HWND mdiCreateDoc(LPSTR szClass, LPSTR szTitle, LPARAM l)
{
    return mdiCreateChild(hwndMdi,szClass,szTitle,
	WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
        CW_USEDEFAULT,0,CW_USEDEFAULT,0,SW_NORMAL,NULL,l);
}

/*----------------------------------------------------------------------------*\
|   mdiCreateClient()                                                           |
|									       |
|   Description:                                                               |
|                                                                              |
|   Arguments:                                                                 |
|                                                                              |
|   Returns:                                                                   |
|	HWND if successful, NULL otherwise				       |
|									       |
\*----------------------------------------------------------------------------*/
HWND FAR PASCAL mdiCreateClient(HWND hwndP, HMENU hmenuWindow)
{
    CLIENTCREATESTRUCT ccs;

    ccs.hWindowMenu = hmenuWindow;
    ccs.idFirstChild = 942;

    return CreateWindow ("MDICLIENT",NULL,
                WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE,
                0,0,0,0,
                hwndP, 0, (HINSTANCE)GetWindowWord(hwndP,GWW_HINSTANCE),
                (LPVOID)&ccs);
}

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

#define mdiActiveDoc() \
    (HWND)SendMessage(hwndMdi,WM_MDIGETACTIVE,0,0L)

/*----------------------------------------------------------------------------*\
\*----------------------------------------------------------------------------*/

LONG NEAR PASCAL mdiSendMessage(HWND hwndMdi, HWND hwnd, unsigned msg, WORD wParam, LONG lParam)
{
    if (hwnd == (HWND)-1)
    {
        for (hwnd = GetWindow(hwndMdi, GW_CHILD); hwnd; hwnd = GetWindow(hwnd, GW_HWNDNEXT))
            SendMessage(hwnd, msg, wParam, lParam);

        return 0L;
    }
    else
    {
        if (hwnd == NULL)
            hwnd = (HWND)SendMessage(hwndMdi,WM_MDIGETACTIVE,0,0L);

        if (hwnd)
            return SendMessage(hwnd, msg, wParam, lParam);
    }
}

/*----------------------------------------------------------------------------*\
|   AppWndProc( hwnd, msg, wParam, lParam )                                    |
|                                                                              |
|   Description:                                                               |
|       The window proc for the app's main (tiled) window.  This processes all |
|       of the parent window's messages.                                       |
|									       |
|   Arguments:                                                                 |
|       hwnd            window handle for the parent window                    |
|       msg             message number                                         |
|       wParam          message-dependent                                      |
|       lParam          message-dependent                                      |
|                                                                              |
|   Returns:                                                                   |
|       0 if processed, nonzero if ignored                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
long FAR PASCAL _export AppWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    UINT            f;
    PAINTSTRUCT     ps;
    HDC             hdc;
    HMENU           hmenu;
    HWND            hwndMovie;
    HWND            hwndT;

    switch (msg) {
        case WM_COMMAND:
            hwndMovie = mdiActiveDoc();

	    switch(wParam) {
		case MENU_ABOUT:
                    fDialog(hwnd,ABOUTBOX,(FARPROC)AppAbout);
		    break;

		case MENU_EXIT:
                    PostMessage(hwnd,WM_CLOSE,0,0L);
                    break;

                case MENU_CLOSE:
                    SendMessage(hwndMovie,WM_CLOSE,0,0);
                    break;

                case MENU_CLOSEALL:
                    while (hwndT = mdiActiveDoc())
			SendMessage(hwndT,WM_CLOSE,0,0);
                    break;

                case MENU_NEW:
                    mdiCreateDoc(szDocClass, "Untitled", 0);
                    break;

                case MENU_OPEN:
                    /* prompt user for file to open */
                    ofn.lStructSize = sizeof(OPENFILENAME);
                    ofn.hwndOwner = hwnd;
                    ofn.hInstance = NULL;
                    ofn.lpstrFilter = szOpenFilter;
                    ofn.lpstrCustomFilter = NULL;
                    ofn.nMaxCustFilter = 0;
                    ofn.nFilterIndex = 0;
                    ofn.lpstrFile = achFileName;
                    ofn.nMaxFile = sizeof(achFileName);
                    ofn.lpstrFileTitle = NULL;
                    ofn.nMaxFileTitle = 0;
                    ofn.lpstrInitialDir = NULL;
                    ofn.lpstrTitle = "Open";
                    ofn.Flags = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
                    ofn.nFileOffset = 0;
                    ofn.nFileExtension = 0;
                    ofn.lpstrDefExt = NULL;
                    ofn.lCustData = 0;
                    ofn.lpfnHook = NULL;
                    ofn.lpTemplateName = NULL;

                    if (GetOpenFileName(&ofn))
                    {
                        hwndMovie = mdiCreateDoc(szDocClass, achFileName, 0);
                        if (hwndMovie) {
                            char  achFileTitle[ACH_BUFFER_LENGTH];
                    //
                    // Set the window text to the filename without the
                    // directory path information.
                    //
                            GetFileTitle(achFileName,
                                         achFileTitle,
                                         ACH_BUFFER_LENGTH);
                            SetWindowText(hwndMovie,achFileTitle);
                        }
                    }
                    break;

                case WM_MDITILE:
                case WM_MDICASCADE:
                case WM_MDIICONARRANGE:
                    SendMessage(hwndMdi, (UINT)wParam, 0, 0);
                    break;

                /* Movie Menu */
                case IDM_PLAY:
                    MCIWndPlay(hwndMovie);
                    break;
                case IDM_RPLAY:
                    MCIWndPlayReverse(hwndMovie);
                    break;
                case IDM_STOP:
                    MCIWndStop(hwndMovie);
                    break;
                case IDM_HOME:
                    MCIWndHome(hwndMovie);
                    break;
                case IDM_END:
                    MCIWndEnd(hwndMovie);
                    break;
                case IDM_STEP:
                    MCIWndStep(hwndMovie, 1);
                    break;
                case IDM_RSTEP:
                    MCIWndStep(hwndMovie, -1);
                    break;
                default:
                    mdiSendMessage(hwndMdi,NULL,msg,wParam,lParam);
                    break;
	    }
            break;

        case WM_PALETTECHANGED:
            mdiSendMessage(hwndMdi, (HWND)-1, msg, wParam, lParam);
            break;

        case WM_QUERYNEWPALETTE:
            return mdiSendMessage(hwndMdi, NULL, msg, wParam, lParam);

        case WM_INITMENU:
            hwndMovie = mdiActiveDoc();

            if (!hwndMovie || MCIWndGetMode(hwndMovie) == MCI_MODE_NOT_READY) {
		f = hwndMovie ? MF_ENABLED : MF_GRAYED;
                EnableMenuItem((HMENU)wParam, MENU_CLOSE, f);
                EnableMenuItem((HMENU)wParam, MENU_CLOSEALL, f);

                EnableMenuItem((HMENU)wParam, IDM_STOP, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_PLAY, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_RPLAY, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_HOME, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_END, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_STEP, MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_RSTEP, MF_GRAYED);
            }
            else {
                EnableMenuItem((HMENU)wParam, MENU_CLOSE, MF_ENABLED);
                EnableMenuItem((HMENU)wParam, MENU_CLOSEALL, MF_ENABLED);

                f = MCIWndGetMode(hwndMovie) != MCI_MODE_STOP;
                EnableMenuItem((HMENU)wParam, IDM_PLAY, !f ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_RPLAY,!f ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem((HMENU)wParam, IDM_STOP, f  ? MF_ENABLED : MF_GRAYED);

                EnableMenuItem((HMENU)wParam, IDM_HOME, MF_ENABLED);
                EnableMenuItem((HMENU)wParam, IDM_END,  MF_ENABLED);
                EnableMenuItem((HMENU)wParam, IDM_STEP, MF_ENABLED);
                EnableMenuItem((HMENU)wParam, IDM_RSTEP,MF_ENABLED);
            }

            return mdiSendMessage(hwndMdi, NULL, msg, wParam, lParam);
            break;

       case WM_CREATE:
            hmenu = GetMenu(hwnd);
            hwndMdi = mdiCreateClient(hwnd, GetSubMenu(hmenu, GetMenuItemCount(hmenu)-1));
            break;

       case WM_SIZE:
            MoveWindow(hwndMdi,0,0,LOWORD(lParam),HIWORD(lParam),TRUE);
            break;

       case WM_DESTROY:
	    PostQuitMessage(0);
	    break;

       case WM_PAINT:
            hdc = BeginPaint(hwnd, &ps);
            EndPaint(hwnd, &ps);
            return 0;
    }
    return DefFrameProc(hwnd,hwndMdi,msg,wParam,lParam);
}

/*----------------------------------------------------------------------------*\
|   ErrMsg - Opens a Message box with a error message in it.  The user can     |
|	     select the OK button to continue or the CANCEL button to kill     |
|	     the parent application.					       |
\*----------------------------------------------------------------------------*/
int ErrMsg (LPSTR sz,...)
{
    char ach[ACH_BUFFER_LENGTH];

    wvsprintf(ach,sz,(LPSTR)(&sz+1));   /* Format the string */
    MessageBox (NULL,ach,NULL,MB_OK|MB_ICONEXCLAMATION);
    return FALSE;
}
