/*
 * CLASSLIB.H
 * Sample Code Class Libraries
 *
 * Master include file for our own Class Library containing:
 *  CStringTable    Class that manages a stringtable.
 *
 *  CWindow         Basic window class
 *  CFrame          Frame window derived from CWindow
 *
 *  CGizmoBar       Wrapper for GizmoBar control derived from CWindow.
 *  CStatStrip      Wrapper for StatStrip control derived from CWindow.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#ifndef _CLASSLIB_H_
#define _CLASSLIB_H_

//Get default resource definitions
#include <classres.h>

class __far CStringTable;
class __far CGizmoBar;
class __far CStatStrip;
class __far CWindow;
class __far CFrame;
class __far CClient;
class __far CDocument;
class __far CDocumentAdviseSink;



/**
 ** CStringTable providing string table management for anyone.
 **/

class __far CStringTable
    {
    protected:
        HINSTANCE       m_hInst;
        UINT            m_idsMin;
        UINT            m_idsMax;
        UINT            m_cStrings;
        LPSTR           m_pszStrings;
        LPSTR FAR      *m_ppszTable;

    public:
        CStringTable(HINSTANCE);
        ~CStringTable(void);

        BOOL FInit(UINT, UINT);

        //Function to resolve an ID into a string pointer.
        const LPSTR operator [](const UINT) const;
    };


typedef CStringTable FAR * LPCStringTable;

/*
 * Typical classes that initialize a stringtable should have an m_pST
 * member, then they can use this macro conveniently.
 */

#define PSZ(i) ((*m_pST)[i])





/**
 ** CWindow:  Our basic window class.
 **/

class __far CWindow
    {
    protected:
        HINSTANCE   m_hInst;            //Task instance
        HWND        m_hWnd;             //Window handle of the window

    public:
        //Standard Class Functions
        CWindow(HINSTANCE);
        ~CWindow(void);

        //Derived classes should have no need to modify this.
        HWND Window(void);
    };


typedef CWindow FAR * LPCWindow;




/**
 ** CGizmoBar encapculates a GizmoBar control
 **/

#include <bttncur.h>
#include <gizmobar.h>


class __far CGizmoBar : public CWindow
    {
    protected:
        UINT            m_cyBar;

    public:
        CGizmoBar(HINSTANCE);
        ~CGizmoBar(void);

        BOOL FInit(HWND, UINT, UINT);

        //Window message wrappers
        void   OnSize(HWND);
        void   FontSet(HFONT, BOOL);
        HFONT  FontGet(void);
        void   Enable(BOOL);

        //GizmoBar function wrappers
        HWND   HwndAssociateSet(HWND);
        HWND   HwndAssociateGet(void);

        BOOL   Add(UINT, UINT, UINT, UINT, UINT, LPSTR, HBITMAP, UINT, UINT);
        BOOL   Remove(UINT);

        LONG   SendMessage(UINT, UINT, WPARAM, LPARAM);

        BOOL   Show(UINT, BOOL);
        BOOL   Enable(UINT, BOOL);
        BOOL   Check(UINT, BOOL);
        UINT   FocusSet(UINT);
        BOOL   Exist(UINT);

        int    TypeGet(UINT);

        DWORD  DataSet(UINT, DWORD);
        DWORD  DataGet(UINT);
        BOOL   NotifySet(UINT, BOOL);
        BOOL   NotifyGet(UINT);

        int    TextGet(UINT, LPSTR, UINT);
        void   TextSet(UINT, LPSTR);
        UINT   IntGet(UINT, BOOL FAR *, BOOL);
        void   IntSet(UINT, int, BOOL);
    };


typedef CGizmoBar FAR * LPCGizmoBar;





/**
 ** CStatStrip encapsulates a StatStrip control.
 **/

#include <stastrip.h>


class __far CStatStrip : public CWindow
    {
    protected:
        UINT        m_cy;           //Control height

    public:
        CStatStrip(HINSTANCE);
        ~CStatStrip(void);

        BOOL FInit(HWND, UINT, UINT);

        void  OnSize(HWND);
        void  MessageSet(LPSTR);
        UINT  MessageGet(LPSTR, UINT);
        UINT  MessageGetLength(void);
        void  FontSet(HFONT, BOOL);
        HFONT FontGet(void);
        void  Enable(BOOL);

        BOOL  MessageMap(HWND, HINSTANCE, UINT, UINT, UINT, UINT, UINT, UINT, UINT, UINT, UINT);
        void  MenuSelect(WPARAM, LPARAM);
        void  MessageDisplay(UINT);
    };

typedef CStatStrip FAR * LPCStatStrip;





/**
 ** CFrame encapsulates a main application window
 **/

//FRAMEWIN.CPP:  Standard window procedure and AboutProc
LRESULT __export FAR PASCAL FrameWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL    __export FAR PASCAL AboutProc(HWND, UINT, WPARAM, LPARAM);

#define CBFRAMEWNDEXTRA     sizeof(LONG)
#define FRAMEWL_STRUCTURE   0


/*
 * Structure containing resource ranges for initialization of
 * a CFrame object through its FInit member.
 */

typedef struct tagFRAMEINIT
    {
    UINT    idsMin;         //Stringtable start and end
    UINT    idsMax;

    UINT    idsStatMin;     //StatStrip stringtable start and end
    UINT    idsStatMax;

    UINT    idStatMenuMin;  //Start and end IDs for StatStrip popup menu IDs
    UINT    idStatMenuMax;

    UINT    iPosWindowMenu; //Position of the Window menu (for MDI init)
    UINT    cMenus;         //Number of popup menus we have.
    } FRAMEINIT, FAR *LPFRAMEINIT;



//FRAME.CPP:  Frame object that creates a main window

class __far CFrame : public CWindow
    {
    //Let our window procedure look in our private variables.
    friend LRESULT __export FAR PASCAL FrameWndProc(HWND, UINT, WPARAM, LPARAM);
    friend class CDocumentAdviseSink;

    protected:
        HINSTANCE       m_hInstPrev;        //WinMain parameters
        LPSTR           m_pszCmdLine;
        int             m_nCmdShow;

        BOOL            m_fInit;            //TRUE when initializing or closing
        BOOL            m_fSizing;          //TRUE when inside WM_SIZE
        BOOL            m_fClosing;         //TRUE when in WM_CLOSE
        HCURSOR         m_hCurHold;         //Cursor held during HourGlass

        HMENU FAR      *m_phMenu;           //Popup menu handles
        HMENU           m_hMenuWindow;      //Position of the window menu.

        HACCEL          m_hAccel;           //Accelerators
        HBITMAP         m_hBmp;             //Gizmobar images
        UINT            m_cyBar;            //Gizmobar height
        UINT            m_dxB;              //Gizmobar button sizes.
        UINT            m_dyB;

        LPCGizmoBar     m_pGB;              //GizmoBar
        LPCStatStrip    m_pSS;              //StatStrip

        LPCStringTable  m_pST;              //Stringtable.
        CClient FAR    *m_pCL;              //SDI OR MDI client window

        CDocumentAdviseSink FAR *m_pAdv;    //Advise sink for document codes.

    protected:
        virtual CClient FAR *CreateCClient(void);

        virtual BOOL     FRegisterAllClasses(void);
        virtual BOOL     FPreShowInit(void);
        virtual void     ParseCommandLine(void);
        virtual UINT     CreateGizmos(void);

        virtual BOOL     FMessageHook(HWND, UINT, WPARAM, LPARAM, LRESULT FAR *);
        virtual LRESULT  OnCommand(HWND, WPARAM, LPARAM);
        virtual void     HourGlass(BOOL);
        virtual void     OnDocumentDataChange(CDocument FAR *);
        virtual void     OnDocumentActivate(CDocument FAR *);

        virtual BOOL     FSaveOpenDialog(LPSTR, UINT, UINT, BOOL, UINT FAR *);
        virtual UINT     ReplaceCharWithNull(LPSTR, int);
        virtual LPSTR    PszWhiteSpaceScan(LPSTR, BOOL);

    public:
        CFrame(HINSTANCE, HINSTANCE, LPSTR, int);
        virtual ~CFrame(void);

        virtual BOOL     FInit(LPFRAMEINIT);

        virtual WPARAM   MessageLoop(void);
        virtual BOOL     FAskAndSave(LPSTR);
        virtual void     UpdateMenus(HMENU, UINT);
        virtual void     UpdateGizmos(void);
        virtual void     WindowTitleSet(CDocument FAR *, BOOL);
    };


typedef CFrame FAR * LPCFrame;


//Other Miscellaneous CFrame definitions

//GizmoBar and StatStrip IDs used in this frame
#define ID_GIZMOBAR         11
#define ID_STATSTRIP        12

//Classname
#define SZCLASSFRAME        "Frame"





/**
 ** CClient encapsulates an MDI or SDI client window
 **/


//CLIENT.CPP
LRESULT __export FAR PASCAL SDIClientWndProc(HWND, UINT, WPARAM, LPARAM);



#define CBCLIENTWNDEXTRA    sizeof(HWND)
#define CLIENTWW_HWNDDOC    0


class __far CClient : public CWindow
    {
    friend LRESULT __export FAR PASCAL SDIClientWndProc(HWND, UINT, WPARAM, LPARAM);

    protected:
        LPCFrame        m_pFR;              //Frame window information
        UINT            m_cDoc;             //Count of open documents

    protected:
        //Overridable for creating a new CDocument
        virtual CDocument FAR * CreateCDocument();

    public:
        CClient(HINSTANCE);
        virtual ~CClient(void);

        virtual BOOL            FInit(HMENU, LPRECT, LPCFrame);

        virtual BOOL            TranslateAccelerator(LPMSG);
        virtual LRESULT         DefaultFrameProc(HWND, UINT, WPARAM, LPARAM);
        virtual void            OnWindowCommand(UINT, UINT);
        virtual void            OnSize(UINT, UINT, UINT, UINT);

        virtual CDocument FAR * NewDocument(BOOL, CDocumentAdviseSink FAR *);
        virtual CDocument FAR * ActiveDocument(void);
        virtual BOOL            ShowDocument(CDocument FAR *, BOOL);
        virtual BOOL            SDIVerify(void);
        virtual UINT            CloseDocument(CDocument FAR *);
        virtual BOOL            QueryCloseAllDocuments(BOOL);
        virtual BOOL            FCleanVerify(CDocument FAR *);
    };


typedef CClient FAR * LPCClient;


//Other Miscellaneous CFrame definitions

//Document window identifiers
#define ID_DOCUMENT         999     //Internal ID for all documents
#define ID_MDICHILDMIN      1000    //Starting point for MDI document IDs

#define SZCLASSSDICLIENT    "SDIClient"




/**
 ** CDocument encapsulates an MDI or SDI child window
 **/


//DOCUMENT.CPP
LRESULT __export FAR PASCAL DocumentWndProc(HWND, UINT, WPARAM, LPARAM);


//File-related string lengths.
#define CCHPATHMAX          256
#define CCHFILENAMEMAX      15


//Window extra bytes and offsets
#define CBDOCUMENTWNDEXTRA  sizeof(LONG)
#define DOCWL_STRUCTURE     0


//Error codes from Load and Save functions
#define DOCERR_NONE                     0
#define DOCERR_NOFILE                   1
#define DOCERR_COULDNOTOPEN             2
#define DOCERR_READFAILURE              3
#define DOCERR_UNSUPPORTEDVERSION       4
#define DOCERR_WRITEFAILURE             5
#define DOCERR_CANCELLED                6
#define DOCERR_STDMAX                   6




/*
 * Structure containing resource ranges and other data for
 * initialization of a CDocument object through its FInit member.
 */

typedef struct tagDOCUMENTINIT
    {
    UINT        idsMin;    //Stringtable start and end
    UINT        idsMax;
    HWND        hWndDoc;   //HWND of this document, created in client

    CDocumentAdviseSink FAR * pAdv;
    } DOCUMENTINIT, FAR *LPDOCUMENTINIT;



class __far CDocument : public CWindow
    {
    friend LRESULT __export FAR PASCAL DocumentWndProc(HWND, UINT, WPARAM, LPARAM);

    protected:
        UINT            m_cf;                 //Document clipboard format.
        BOOL            m_fDirty;             //Is file dirty?
        BOOL            m_fNoDirty;           //If TRUE, don't touch dirty flag.
        BOOL            m_fNoSize;            //Prevent ourselves from sizing.
        BOOL            m_fFileKnown;         //FALSE if File/New used until saved.
        char            m_szFile[CCHPATHMAX]; //Filename for Save command.

        LPCStringTable  m_pST;                //Document string.

        //If someone above us wants information, they'll give us this object.
        class CDocumentAdviseSink FAR * m_pAdv;

    protected:
        virtual BOOL    FMessageHook(HWND, UINT, WPARAM, LPARAM, LRESULT FAR *);

    public:
        CDocument(HINSTANCE);
        virtual ~CDocument(void);

        virtual BOOL     FInit(LPDOCUMENTINIT);

        virtual BOOL     FDirtySet(BOOL);
        virtual BOOL     FDirtyGet(void);
        virtual void     Clear(void);

        virtual UINT     ULoad(BOOL, LPSTR);
        virtual UINT     USave(UINT, LPSTR);
        virtual void     ErrorMessage(UINT);

        virtual BOOL     FClip(HWND, BOOL);
        virtual HGLOBAL  RenderFormat(UINT);
        virtual BOOL     FQueryPaste(void);
        virtual BOOL     FPaste(HWND);
        virtual void     Undo(void);

        virtual BOOL     FQuerySave(void);
        virtual void     Rename(LPSTR);
        virtual UINT     FilenameGet(LPSTR, UINT);
    };

typedef CDocument FAR * LPCDocument;


//Classname for documents, not localized
#define SZCLASSDOCUMENT     "document"


//Macro that isolates us from MDI or SDI Def procs
#ifdef MDI
#define DEFDOCUMENTPROC     DefMDIChildProc
#else
#define DEFDOCUMENTPROC     DefWindowProc
#endif


//Message to get PCDocument from hWnd
#define DOCM_PDOCUMENT      (WM_USER+0)




class __far CDocumentAdviseSink
    {
    protected:
        LPVOID      m_pv;                       //Customizable structure

    public:
        CDocumentAdviseSink(LPVOID);

        void OnDataChange(LPCDocument);         //Data change may affect menus
        void OnCloseRequest(LPCDocument);       //Document should be closed
        void OnSizeChange(LPCDocument, LPRECT); //Document was sized
        void OnCaptionChange(LPCDocument);      //Update caption should happen
        void OnActivate(LPCDocument);           //Document is becoming active.
    };


typedef CDocumentAdviseSink FAR * LPCDocumentAdviseSink;



#endif _CLASSLIB_H_
