/*
 * IPERSTOR.CPP
 * Schmoo Figure Handler Chapter 11
 *
 * Implementation of the IPersistStorage interface that we expose on the
 * Figure object.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "hschmoo.h"


/*
 * CImpIPersistStorage:CImpIPersistStorage
 * CImpIPersistStorage::~CImpIPersistStorage
 *
 * Constructor Parameters:
 *  pObj            LPVOID pointing to the object we live in.
 *  punkOuter       LPUNKNOWN of the controlling unknown.
 */

CImpIPersistStorage::CImpIPersistStorage(LPCFigure pObj, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pObj=pObj;
    m_punkOuter=punkOuter;
    return;
    }


CImpIPersistStorage::~CImpIPersistStorage(void)
    {
    return;
    }




/*
 * CImpIPersistStorage::QueryInterface
 * CImpIPersistStorage::AddRef
 * CImpIPersistStorage::Release
 *
 * Purpose:
 *  Standard set of IUnknown members for this interface
 */

STDMETHODIMP CImpIPersistStorage::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    return m_punkOuter->QueryInterface(riid, ppv);
    }

STDMETHODIMP_(ULONG) CImpIPersistStorage::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIPersistStorage::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }





/*
 * CImpIPersistStorage::GetClassID
 *
 * Purpose:
 *  Returns the CLSID of the object represented by this interface.
 */

STDMETHODIMP CImpIPersistStorage::GetClassID(LPCLSID pClsID)
    {
    *pClsID=m_pObj->m_clsID;
    return NOERROR;
    }




/*
 * CImpIPersistStorage::IsDirty
 *
 * Purpose:
 *  Tells the caller if we have made changes to this object since
 *  it was loaded or initialized new.
 */

STDMETHODIMP CImpIPersistStorage::IsDirty(void)
    {
    /*
     * Since we don't edit, we have no idea if this data is dirty.
     * Delegate to the default handler in case it wants to ask the server.
     */
    return m_pObj->m_pDefIPersistStorage->IsDirty();
    }





/*
 * CImpIPersistStorage::InitNew
 *
 * Purpose:
 *  Provides the object with the IStorage they can hold on to while
 *  they are running.  Here we can initialize the structure of the
 *  storage and AddRef it for incremental access.  This function will
 *  only be called once in the object's lifetime in lieu of ::Load.
 */

STDMETHODIMP CImpIPersistStorage::InitNew(LPSTORAGE pIStorage)
    {
    //Good time to initilize our data
    m_pObj->m_pl.wVerMaj=VERSIONMAJOR;
    m_pObj->m_pl.wVerMin=VERSIONMINOR;
    m_pObj->m_pl.cPoints=0;
    m_pObj->m_pl.rgbBackground=GetSysColor(COLOR_WINDOW);
    m_pObj->m_pl.rgbLine=GetSysColor(COLOR_WINDOWTEXT);
    m_pObj->m_pl.iLineStyle=PS_SOLID;

    //Make sure these aren't filled with trash.
    _fmemcpy(&m_pObj->m_plContent,   &m_pObj->m_pl, CBPOLYLINEDATA);
    _fmemcpy(&m_pObj->m_plThumbnail, &m_pObj->m_pl, CBPOLYLINEDATA);


    m_pObj->m_pDefIPersistStorage->InitNew(pIStorage);
    return NOERROR;
    }





/*
 * CImpIPersistStorage::Load
 *
 * Purpose:
 *  Instructs the object to load itself from a previously saved IStorage
 *  that was handled by ::Save in another object lifetime.  This function
 *  will only be called once in the object's lifetime in lieu of ::InitNew.
 */

STDMETHODIMP CImpIPersistStorage::Load(LPSTORAGE pIStorage)
    {
    POLYLINEDATA    pl;
    ULONG           cb;
    LPSTREAM        pIStream;
    HRESULT         hr;

    if (NULL==pIStorage)
        return ResultFromScode(STG_E_INVALIDPOINTER);

    //Open the CONTENTS stream
    hr=pIStorage->OpenStream("CONTENTS", 0, STGM_DIRECT | STGM_READ
        | STGM_SHARE_EXCLUSIVE, 0, &pIStream);

    if (FAILED(hr))
        return ResultFromScode(STG_E_READFAULT);

    //Read all the data into the POLYLINEDATA structure.
    hr=pIStream->Read((LPVOID)&pl, CBPOLYLINEDATA, &cb);
    pIStream->Release();

    if (CBPOLYLINEDATA!=cb)
        return ResultFromScode(STG_E_READFAULT);

    //Copy into the actual object now.
    _fmemcpy(&m_pObj->m_pl, &pl, CBPOLYLINEDATA);

    m_pObj->m_pDefIPersistStorage->Load(pIStorage);
    return NOERROR;
    }





/*
 * CImpIPersistStorage::Save
 *
 * Purpose:
 *  Saves the data for this object to an IStorage.
 */

STDMETHODIMP CImpIPersistStorage::Save(LPSTORAGE pIStorage, BOOL fSameAsLoad)
    {
    ULONG           cb;
    LPSTREAM        pIStream;
    HRESULT         hr;

    if (NULL==pIStorage)
        return ResultFromScode(STG_E_INVALIDPOINTER);

    /*
     * If the server is running, don't do the save ourselves since
     * we'd end up writing the storage twice with possible conflicts.
     */
    if (OleIsRunning(m_pObj->m_pDefIOleObject))
        return m_pObj->m_pDefIPersistStorage->Save(pIStorage, fSameAsLoad);

    //Rewrite the entire stream
    WriteClassStg(pIStorage, m_pObj->m_clsID);
    WriteFmtUserTypeStg(pIStorage, m_pObj->m_cf, "Polyline Figure");

    hr=pIStorage->CreateStream("CONTENTS", STGM_DIRECT | STGM_CREATE
        | STGM_WRITE | STGM_SHARE_EXCLUSIVE, 0, 0, &pIStream);

    if (FAILED(hr))
        return ResultFromScode(STG_E_WRITEFAULT);

    hr=pIStream->Write((LPVOID)&m_pObj->m_pl, CBPOLYLINEDATA, &cb);
    pIStream->Release();

    m_pObj->m_pDefIPersistStorage->Save(pIStorage, fSameAsLoad);

    return (SUCCEEDED(hr) && CBPOLYLINEDATA==cb) ?
        NOERROR : ResultFromScode(STG_E_WRITEFAULT);
    }




/*
 * CImpIPersistStorage::SaveCompleted
 * CImpIPersistStorage::HandsOffStorage
 *
 * Purpose:
 *  Pass throughs.
 */

STDMETHODIMP CImpIPersistStorage::SaveCompleted(LPSTORAGE pIStorage)
    {
    return m_pObj->m_pDefIPersistStorage->SaveCompleted(pIStorage);
    }

STDMETHODIMP CImpIPersistStorage::HandsOffStorage(void)
    {
    return m_pObj->m_pDefIPersistStorage->HandsOffStorage();
    }
