/*
 * IPERSTOR.CPP
 * Polyline Component Object Chapter 11
 *
 * Implementation of the IPersistStorage interface that we expose on the
 * Polyline object.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "polyline.h"


/*
 * CImpIPersistStorage:CImpIPersistStorage
 * CImpIPersistStorage::~CImpIPersistStorage
 *
 * Constructor Parameters:
 *  pObj            LPCPolyline pointing to the object we live in.
 *  punkOuter       LPUNKNOWN of the controlling unknown.
 */

CImpIPersistStorage::CImpIPersistStorage(LPCPolyline pObj, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pObj=pObj;
    m_punkOuter=punkOuter;
    return;
    }


CImpIPersistStorage::~CImpIPersistStorage(void)
    {
    return;
    }




/*
 * CImpIPersistStorage::QueryInterface
 * CImpIPersistStorage::AddRef
 * CImpIPersistStorage::Release
 *
 * Purpose:
 *  Standard set of IUnknown members for this interface
 */

STDMETHODIMP CImpIPersistStorage::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    return m_punkOuter->QueryInterface(riid, ppv);
    }

STDMETHODIMP_(ULONG) CImpIPersistStorage::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIPersistStorage::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }





/*
 * CImpIPersistStorage::GetClassID
 *
 * Purpose:
 *  Returns the CLSID of the object represented by this interface.
 *
 * Parameters:
 *  pClsID          LPCLSID in which to store our CLSID.
 *
 * Return Value:
 *  HRESULT         NOERROR on success, error code otherwise.
 */

STDMETHODIMP CImpIPersistStorage::GetClassID(LPCLSID pClsID)
    {
    *pClsID=m_pObj->m_clsID;
    return NOERROR;
    }





/*
 * CImpIPersistStorage::IsDirty
 *
 * Purpose:
 *  Tells the caller if we have made changes to this object since
 *  it was loaded or initialized new.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  HRESULT         Contains S_OK if we ARE dirty, S_FALSE if NOT dirty,
 *                  that is, "Yes I AM dirty, or NO, I'm clean."
 */

STDMETHODIMP CImpIPersistStorage::IsDirty(void)
    {
    return ResultFromScode(m_pObj->m_fDirty ? S_OK : S_FALSE);
    }





/*
 * CImpIPersistStorage::Load
 *
 * Purpose:
 *  Instructs the object to load itself from a previously saved IStorage
 *  that was handled by ::Save in another object lifetime.  This function
 *  will only be called once in the object's lifetime in lieu of ::InitNew.
 *  The object may hold on to pIStorage here for incremental access.
 *
 * Parameters:
 *  pIStorage       LPSTORAGE from which to load.
 *
 * Return Value:
 *  HRESULT         NOERROR on success, error code otherwise.
 */

STDMETHODIMP CImpIPersistStorage::Load(LPSTORAGE pIStorage)
    {
    POLYLINEDATA    pl;
    ULONG           cb;
    LPSTREAM        pIStream;
    HRESULT         hr;

    if (NULL==pIStorage)
        return ResultFromScode(STG_E_INVALIDPOINTER);

    //We don't check ClassStg to remain compatible with other chatpers.

    //Open the CONTENTS stream
    hr=pIStorage->OpenStream("CONTENTS", 0, STGM_DIRECT | STGM_READ
        | STGM_SHARE_EXCLUSIVE, 0, &pIStream);

    if (FAILED(hr))
        return ResultFromScode(STG_E_READFAULT);

    //Read all the data into the POLYLINEDATA structure.
    hr=pIStream->Read((LPVOID)&pl, CBPOLYLINEDATA, &cb);
    pIStream->Release();

    if (CBPOLYLINEDATA!=cb)
        return ResultFromScode(STG_E_READFAULT);

    //DataSet now internal on CPolyline
    m_pObj->DataSet(&pl, TRUE, TRUE);

    //CHAPTER11MOD
    //We also need to tell the cache to load cached graphics
    m_pObj->m_pDefIPersistStorage->Load(pIStorage);
    //End CHAPTER11MOD
    return NOERROR;
    }





/*
 * CImpIPersistStorage::Save
 *
 * Purpose:
 *  Saves the data for this object to an IStorage which may
 *  or may not be the same as the one previously passed to
 *  ::Load, indicated with fSameAsLoad.
 *
 * Parameters:
 *  pIStorage       LPSTORAGE in which to save our data.
 *  fSameAsLoad     BOOL indicating if this is the same pIStorage
 *                  that was passed to ::Load.  If it was, then
 *                  objects that built up a structure in that storage
 *                  do not have to regenerate the entire thing.
 *
 * Return Value:
 *  HRESULT         NOERROR on success, error code otherwise.
 */

STDMETHODIMP CImpIPersistStorage::Save(LPSTORAGE pIStorage, BOOL fSameAsLoad)
    {
    POLYLINEDATA    pl;
    ULONG           cb;
    LPSTREAM        pIStream;
    HRESULT         hr;

    if (NULL==pIStorage)
        return ResultFromScode(STG_E_INVALIDPOINTER);

    /*
     * fSameAsLoad is not important to us since we always rewrite
     * an entire stream as well as the identification tags for this
     * object.  Note that we don't bother to check the ClassStg
     * above in ::Load to remain compatible with other revisions
     * of Polyline in other chapters.
     */

    WriteClassStg(pIStorage, m_pObj->m_clsID);
    WriteFmtUserTypeStg(pIStorage, m_pObj->m_cf, (*m_pObj->m_pST)[IDS_USERTYPE]);

    hr=pIStorage->CreateStream("CONTENTS", STGM_DIRECT | STGM_CREATE
        | STGM_WRITE | STGM_SHARE_EXCLUSIVE, 0, 0, &pIStream);

    if (FAILED(hr))
        return ResultFromScode(STG_E_WRITEFAULT);

    //DataGet now internal on CPolyline
    m_pObj->DataGet(&pl);

    hr=pIStream->Write((LPVOID)&pl, CBPOLYLINEDATA, &cb);
    pIStream->Release();

    //CHAPTER11MOD
    //We also need to tell the cache to save cached graphics
    m_pObj->m_pDefIPersistStorage->Save(pIStorage, fSameAsLoad);
    //End CHAPTER11MOD

    return (SUCCEEDED(hr) && CBPOLYLINEDATA==cb) ?
        NOERROR : ResultFromScode(STG_E_WRITEFAULT);
    }





//CHAPTER11MOD

/*
 * CImpIPersistStorage::InitNew
 * CImpIPersistStorage::SaveCompleted
 * CImpIPersistStorage::HandsOffStorage
 *
 * Passes through to the cache.
 */

STDMETHODIMP CImpIPersistStorage::InitNew(LPSTORAGE pIStorage)
    {
    m_pObj->m_pDefIPersistStorage->InitNew(pIStorage);
    return NOERROR;
    }

STDMETHODIMP CImpIPersistStorage::SaveCompleted(LPSTORAGE pIStorage)
    {
    m_pObj->m_pDefIPersistStorage->SaveCompleted(pIStorage);
    return NOERROR;
    }

STDMETHODIMP CImpIPersistStorage::HandsOffStorage(void)
    {
    m_pObj->m_pDefIPersistStorage->HandsOffStorage();
    return NOERROR;
    }

//End CHAPTER11MOD
