/*
 * DOCUMENT.CPP
 *
 * Implementation of the CSchmooDoc derivation of CDocument as
 * well as an implementation of CPolylineAdviseSink.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "schmoo.h"



/*
 * CSchmooDoc::CSchmooDoc
 * CSchmooDoc::~CSchmooDoc
 *
 * Constructor Parameters:
 *  hInst           HINSTANCE of the application.
 */

CSchmooDoc::CSchmooDoc(HINSTANCE hInst)
    : CDocument(hInst)
    {
    m_pPL=NULL;
    m_pPLAdv=NULL;
    m_uPrevSize=SIZE_RESTORED;

    m_pDropTarget=NULL;
    m_fDragSource=FALSE;

    m_cfEmbedSource=RegisterClipboardFormat(CF_EMBEDSOURCE);
    m_cfObjectDescriptor=RegisterClipboardFormat(CF_OBJECTDESCRIPTOR);

    m_pFigure=NULL;

    //CHAPTER13MOD
    m_pMoniker=NULL;
    m_dwRegROT=0L;

    m_cfLinkSource=RegisterClipboardFormat(CF_LINKSOURCE);
    m_cfLinkSrcDescriptor=RegisterClipboardFormat(CF_LINKSRCDESCRIPTOR);
    //End CHAPTER13MOD
    return;
    }


CSchmooDoc::~CSchmooDoc(void)
    {
    //Make sure the object is saved, the disconnect entirerly
    m_pFigure->SendAdvise(OBJECTCODE_SAVEOBJECT);
    m_pFigure->SendAdvise(OBJECTCODE_HIDEWINDOW);
    m_pFigure->SendAdvise(OBJECTCODE_CLOSED);

    //CHAPTER13MOD
    //If the document is reg'd as running, revoke and free the moniker
    if (0L!=m_dwRegROT)
        OleStdRevokeAsRunning(&m_dwRegROT);

    if (NULL!=m_pMoniker)
        m_pMoniker->Release();
    //End CHAPTER13MOD

    CoDisconnectObject((LPUNKNOWN)m_pFigure, 0L);

    if (NULL!=m_pFigure)
        m_pFigure->Release();   //So it can start shutdown if necessary.

    //Clean up the allocations we did in FInit
    if (NULL!=m_pDropTarget)
        {
        RevokeDragDrop(m_hWnd);

        //This lets OLE forget about the object.
        CoLockObjectExternal((LPUNKNOWN)m_pDropTarget, FALSE, TRUE);

        m_pDropTarget->Release();
        }

    if (NULL!=m_pPL)
        delete m_pPL;

    if (NULL!=m_pPLAdv)
        delete m_pPLAdv;

    return;
    }






/*
 * CSchmooDoc::FInit
 *
 * Purpose:
 *  Initializes an already created document window.  The client actually
 *  creates the window for us, then passes that here for further
 *  initialization.
 *
 * Parameters:
 *  pDI             LPDOCUMENTINIT containing initialization parameters.
 *
 * Return Value:
 *  BOOL            TRUE if the function succeeded, FALSE otherwise.
 */

BOOL CSchmooDoc::FInit(LPDOCUMENTINIT pDI)
    {
    RECT        rc;

    //Change the stringtable range to our customization.
    pDI->idsMin=IDS_DOCUMENTMIN;
    pDI->idsMax=IDS_DOCUMENTMAX;

    //Do default initialization
    if (!CDocument::FInit(pDI))
        return FALSE;

    //Add the Polyline stuff we need.
    m_pPLAdv=new CPolylineAdviseSink((LPVOID)this);
    m_pPL   =new CPolyline(m_hInst);

    //Attempt to create our contained Polyline.
    GetClientRect(m_hWnd, &rc);
    InflateRect(&rc, -8, -8);

    if (!m_pPL->FInit(m_hWnd, &rc, WS_CHILD | WS_VISIBLE
        , ID_POLYLINE, m_pPLAdv))
        return FALSE;

    m_pDropTarget=new CDropTarget(this);

    if (NULL!=m_pDropTarget)
        {
        m_pDropTarget->AddRef();
        RegisterDragDrop(m_hWnd, (LPDROPTARGET)m_pDropTarget);

        /*
         * For some forsaken reason you must call CoLockObjectExternal
         * on the drop target or else OLE will forget about it and lose
         * the pointer after the first drag-drop in which this target
         * is involved.  To prevent that loss, that is, work around
         * an OLE2.DLL but, you can make sure OLE can't forget.
         */
        CoLockObjectExternal((LPUNKNOWN)m_pDropTarget, TRUE, FALSE);
        }


    m_pFigure=new CFigure(ObjectDestroyed, this);

    if (NULL==m_pFigure)
        return FALSE;

    /*
     * These two lines are 1) so we can ::Release on document closure
     * and 2) if we fail FInit then we'll destroy the document which
     * calls ::Release which will call ObjectDestroyed which will
     * decrement g_cObj and shut down if necessary.
     */
    m_pFigure->AddRef();
    g_cObj++;

    if (!m_pFigure->FInit())
        return FALSE;

    return TRUE;
    }







/*
 * CSchmooDoc::FMessageHook
 *
 * Purpose:
 *  Processes WM_SIZE for the document so we can resize the Polyline.
 *
 * Parameters:
 *  <WndProc Parameters>
 *  pLRes           LRESULT FAR * in which to store the return value
 *                  for the message.
 *
 * Return Value:
 *  BOOL            TRUE to prevent further processing, FALSE otherwise.
 */

BOOL CSchmooDoc::FMessageHook(HWND hWnd, UINT iMsg, WPARAM wParam
    , LPARAM lParam, LRESULT FAR *pLRes)
    {
    UINT        dx, dy;
    RECT        rc;

    if (WM_SIZE==iMsg)
        {
        //Don't effect the Polyline size to or from minimized state.
        if (SIZE_MINIMIZED!=wParam && SIZE_MINIMIZED !=m_uPrevSize)
            {
            //When we change size, resize any Polyline we hold.
            dx=LOWORD(lParam);
            dy=HIWORD(lParam);

            /*
             * If we are getting WM_SIZE in response to a Polyline
             * notification, then don't resize the Polyline window again.
             */
            if (!m_fNoSize && NULL!=m_pPL)
                {
                //Resize the polyline to fit the new client
                SetRect(&rc, 8, 8, dx-8, dy-8);
                m_pPL->RectSet(&rc, FALSE);

                /*
                 * We consider sizing something that makes the file dirty,
                 * but not until we've finished the create process, which
                 * is why we set fNoDirty to FALSE in WM_CREATE since we
                 * get a WM_SIZE on the first creation.
                 */
                if (!m_fNoDirty)
                    FDirtySet(TRUE);

                SetRect(&rc, 0, 0, dx, dy);

                if (NULL!=m_pAdv)
                    m_pAdv->OnSizeChange((LPCDocument)this, &rc);

                m_fNoDirty=FALSE;
                }
            }

        m_uPrevSize=wParam;
        }

    if (WM_LBUTTONDOWN==iMsg)
            {
            LPDROPSOURCE    pIDropSource;
            LPDATAOBJECT    pIDataObject;
            HRESULT         hr;
            SCODE           sc;
            DWORD           dwEffect;

            /*
             * The document has an 8 pixel border around the polyline
             * window where we'll see mouse clicks.  A left mouse button
             * click here means the start of a drag-drop operation.
             *
             * Since this is a modal operation, the IDropSource we need
             * here is entirely local.
             */

            pIDropSource=(LPDROPSOURCE)new CDropSource(this);

            if (NULL==pIDropSource)
                return FALSE;

            pIDropSource->AddRef();
            m_fDragSource=TRUE;

            //Go get the data and start the ball rolling.
            pIDataObject=TransferObjectCreate(FALSE);

            hr=DoDragDrop(pIDataObject, pIDropSource
                , DROPEFFECT_COPY | DROPEFFECT_MOVE, &dwEffect);

            pIDataObject->Release();


            /*
             * When we return from DoDragDrop, we either cancelled or dropped.
             * First we can toss the IDropSource we have here, then bail out
             * on cancel, and possibly clear our data on a move drop.
             */

            pIDropSource->Release();

            /*
             * If there was a drop on the same document (determined using
             * this flag, then dwEffect will be DROPEFFECT_NONE (see
             * IDropTarget_Drop in IDROPTGT.CPP).  In any case, we need
             * to reset this since the operation is done.
             */

            m_fDragSource=FALSE;

            sc=GetScode(hr);

            if (DRAGDROP_S_DROP==sc && DROPEFFECT_MOVE==dwEffect)
                {
                m_pPL->New();
                FDirtySet(TRUE);
                }

            //On a canceled drop or a copy we don't need to do anything else
            return TRUE;
            }


    /*
     * We return FALSE even on WM_SIZE so we can let the default procedure
     * handle maximized MDI child windows appropriately.
     */
    return FALSE;
    }








/*
 * CSchmooDoc::Clear
 *
 * Purpose:
 *  Sets all contents in the document back to defaults with no filename.
 *
 * Paramters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Clear(void)
    {
    //Completely reset the polyline
    m_pPL->New();

    CDocument::Clear();
    m_lVer=0;
    return;
    }



/*
 * CSchmooDoc::FDirtySet
 *
 * Purpose:
 *  Sets or clears the document 'dirty' flag returning the previous state
 *  of that same flag.  We override this in Schmoo server to send the
 *  OnDataChange notification as necessary.
 *
 * Parameters:
 *  fDirty          BOOL indicating the new contents of the dirty flag.
 *
 * Return Value:
 *  BOOL            Previous value of the dirty flag.
 */

BOOL CSchmooDoc::FDirtySet(BOOL fDirty)
    {
    BOOL        fRet;

    fRet=CDocument::FDirtySet(fDirty);
    m_pFigure->SendAdvise(OBJECTCODE_DATACHANGED);

    return fRet;
    }




/*
 * CSchmooDoc::FDirtyGet
 *
 * Purpose:
 *  Override of the normal FDirtyGet such that we never return dirty
 *  for an embedded object we're serving since updates constantly happen
 *  and since the object will be saved on closure.  This then prevents
 *  any message boxes asking the user to save.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if the document is dirty, FALSE otherwise.
 */

BOOL CSchmooDoc::FDirtyGet(void)
    {
    if (m_pFigure->FIsEmbedded())
        return FALSE;

    return m_fDirty;
    }




/*
 * CSchmooDoc::ULoad
 *
 * Purpose:
 *  Loads a given document without any user interface overwriting the
 *  previous contents of the Polyline window.  We do this by opening
 *  the file and telling the Polyline to load itself from that file.
 *
 * Parameters:
 *  fChangeFile     BOOL indicating if we're to update the window title
 *                  and the filename from using this file.
 *  pszFile         LPSTR to the filename to load, NULL if the file is
 *                  new and untitled.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::ULoad(BOOL fChangeFile, LPSTR pszFile)
    {
    HRESULT         hr;
    LPSTORAGE       pIStorage;

    if (NULL==pszFile)
        {
        //For a new untitled document, just rename ourselved.
        Rename(NULL);
        m_lVer=VERSIONCURRENT;
        return DOCERR_NONE;
        }

    /*
     * If this is not a Compound File, open the file using STGM_CONVERT
     * in TRANSACTED mode to effectively see old files as a storage with
     * one stream called "CONTENTS" (which is conveniently the name we use
     * in the new files).  We must use STGM_TRANSACTED here or else
     * the old file will be immediately converted on disk:  we only want
     * a converted image in memory from which to read.  In addition,
     * note that we need STGM_READWRITE as well since conversion is
     * inherently a write operation.
     */

    pIStorage=NULL;

    if (NOERROR!=StgIsStorageFile(pszFile))
        {
        hr=StgCreateDocfile(pszFile, STGM_TRANSACTED | STGM_READWRITE
            | STGM_CONVERT | STGM_SHARE_EXCLUSIVE, 0, &pIStorage);

        if (FAILED(hr))
            {
            //If we were denied write access, try to load the old way
            if (STG_E_ACCESSDENIED==GetScode(hr))
                m_lVer=m_pPL->ReadFromFile(pszFile);
            else
                return DOCERR_COULDNOTOPEN;
            }
        }
    else
        {
        hr=StgOpenStorage(pszFile, NULL, STGM_DIRECT | STGM_READ
            | STGM_SHARE_EXCLUSIVE, NULL, 0, &pIStorage);

        if (FAILED(hr))
            return DOCERR_COULDNOTOPEN;
        }

    if (NULL!=pIStorage)
        {
        m_lVer=m_pPL->ReadFromStorage(pIStorage);
        pIStorage->Release();
        }

    if (POLYLINE_E_READFAILURE==m_lVer)
        return DOCERR_READFAILURE;

    if (POLYLINE_E_UNSUPPORTEDVERSION==m_lVer)
        return DOCERR_UNSUPPORTEDVERSION;

    if (fChangeFile)
        Rename(pszFile);

    //Importing a file makes things dirty
    FDirtySet(!fChangeFile);

    return DOCERR_NONE;
    }







/*
 * CSchmooDoc::USave
 *
 * Purpose:
 *  Writes the file to a known filename, requiring that the user has
 *  previously used FileOpen or FileSaveAs in order to have a filename.
 *
 * Parameters:
 *  uType           UINT indicating the type of file the user requested
 *                  to save in the File Save As dialog.
 *  pszFile         LPSTR under which to save.  If NULL, use the current name.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::USave(UINT uType, LPSTR pszFile)
    {
    LONG        lVer, lRet;
    UINT        uTemp;
    BOOL        fRename=TRUE;
    HRESULT     hr;
    LPSTORAGE   pIStorage;
    BOOL        fEmbedding;

    fEmbedding=m_pFigure->FIsEmbedded();

    if (NULL==pszFile)
        {
        fRename=FALSE;
        pszFile=m_szFile;
        }

    /*
     * Type 1 is the current version, type 2 is version 1.0 of the Polyline
     * so we use this to send the right version to CPolyline::WriteToFile.
     */

    switch (uType)
        {
        case 0:         //From Save, use loaded version.
            lVer=m_lVer;
            break;

        case 1:
            lVer=VERSIONCURRENT;
            break;

        case 2:
            lVer=MAKELONG(0, 1);    //1.0
            break;

        default:
            return DOCERR_UNSUPPORTEDVERSION;
        }

    /*
     * If the version the user wants to save is different than the
     * version that we loaded, and m_lVer is not zero (new document),
     * then inform the user of the version change and verify.
     */

    //For embedding, this is Save Copy As, so don't ask about versions.
    if (0!=m_lVer && m_lVer!=lVer && !fEmbedding)
        {
        char        szMsg[128];

        wsprintf(szMsg, PSZ(IDS_VERSIONCHANGE), (UINT)HIWORD(m_lVer)
            , (UINT)LOWORD(m_lVer), (UINT)HIWORD(lVer), (UINT)LOWORD(lVer));

        uTemp=MessageBox(m_hWnd, szMsg, PSZ(IDS_DOCUMENTCAPTION), MB_YESNOCANCEL);

        if (IDCANCEL==uTemp)
            return DOCERR_CANCELLED;

        //If the user won't upgrade versions, revert to loaded version.
        if (IDNO==uTemp)
            lVer=m_lVer;
        }

    /*
     * For 1.0 files, still use the old code.  For new files, use
     * storages instead
     */
    if (lVer==MAKELONG(0, 1))
        lRet=m_pPL->WriteToFile(pszFile, lVer);
    else
        {
        hr=StgCreateDocfile(pszFile, STGM_DIRECT | STGM_READWRITE
            | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0, &pIStorage);

        if (FAILED(hr))
            return DOCERR_COULDNOTOPEN;

        //Mark this as one of our class
        WriteClassStg(pIStorage, CLSID_Schmoo2Figure);

        //Write user-readable class information
        WriteFmtUserTypeStg(pIStorage, m_cf, PSZ(IDS_CLIPBOARDFORMAT));

        lRet=m_pPL->WriteToStorage(pIStorage, lVer);
        pIStorage->Release();
        }

    if (POLYLINE_E_NONE!=lRet)
        return DOCERR_WRITEFAILURE;

    //Saving makes us clean, but this doesn't apply to embedding.
    if (!fEmbedding)
        FDirtySet(FALSE);

    //Update the known version of this document.
    m_lVer=lVer;

    /*
     * If we're embedding, this is Save Copy As, so no rename.
     * Note that we also don't care about having been set to clean
     * since we're always 'clean' as an embedded object from
     * the user's perspective.
     */
    if (fRename && !fEmbedding)
        Rename(pszFile);

    return DOCERR_NONE;
    }


//CHAPTER13MOD
/*
 * CSchmooDoc::Rename
 *
 * Purpose:
 *  Overrides the normal rename to manage a moniker for the open
 *  file.  We maintain no moniker for untitled documents, and therefore
 *  do not allow linking to such documents.
 *
 * Parameters:
 *  pszFile         LPSTR to the new filename.
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Rename(LPSTR pszFile)
    {
    LPMONIKER   pmk;

    //We don't need to change the base class, just augment...
    CDocument::Rename(pszFile);

    //Get rid of the old moniker.  The OldStd function sets the key to zero.
    if (0L!=m_dwRegROT)
        OleStdRevokeAsRunning(&m_dwRegROT);

    if (NULL!=m_pMoniker)
        {
        m_pMoniker->Release();
        m_pMoniker=NULL;
        }

    if (NULL!=pszFile)
        {
        CreateFileMoniker(pszFile, &pmk);

        if (NULL!=pmk)
            {
            m_pMoniker=pmk;     //pmk was AddRef'd in CreateFileMoniker
            OleStdRegisterAsRunning((LPUNKNOWN)m_pFigure, m_pMoniker
                , &m_dwRegROT);

            m_pFigure->SendAdvise(OBJECTCODE_RENAMED);
            }
        }

    return;
    }
//End CHAPTER13MOD






/*
 * CSchmooDoc::Undo
 *
 * Purpose:
 *  Reverses a previous action.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Undo(void)
    {
    m_pPL->Undo();
    return;
    }







/*
 * CSchmooDoc::FClip
 *
 * Purpose:
 *  Places a private format, a metafile, and a bitmap of the display
 *  on the clipboard, optionally implementing Cut by deleting the
 *  data in the current window after rendering.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window
 *  fCut            BOOL indicating cut (TRUE) or copy (FALSE).
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FClip(HWND hWndFrame, BOOL fCut)
    {
    BOOL            fRet=TRUE;
    LPDATAOBJECT    pIDataObject;

    pIDataObject=TransferObjectCreate(fCut);

    if (NULL==pIDataObject)
        return FALSE;

    fRet=SUCCEEDED(OleSetClipboard(pIDataObject));
    pIDataObject->Release();

    //Delete our current data if copying succeeded.
    if (fRet && fCut)
        {
        m_pPL->New();
        FDirtySet(TRUE);
        }

    return fRet;
    }





/*
 * CSchmooDoc::RenderFormat
 *
 * Purpose:
 *  Renders a specific clipboard format into global memory.  We have this
 *  function split out because we'll eventually move to delayed rendering
 *  and this will then be immediately callable from the frame.
 *
 * Parameters:
 *  cf              UINT format to render.
 *
 * Return Value:
 *  HGLOBAL         Global memory handle containing the data.
 */

HGLOBAL CSchmooDoc::RenderFormat(UINT cf)
    {
    HGLOBAL     hMem;

    if (cf==m_cf)
        {
        m_pPL->DataGetMem(VERSIONCURRENT, &hMem);
        return hMem;
        }

    switch (cf)
        {
        case CF_METAFILEPICT:
            return m_pPL->RenderMetafilePict();

        case CF_BITMAP:
            return (HGLOBAL)m_pPL->RenderBitmap();
        }

    return NULL;
    }





/*
 * CSchmooDoc::FRenderMedium
 *
 * Purpose:
 *  Like RenderFormat, this function creates a specific data format
 *  based on the cf parameter.  Unlike RenderFormat, we store the
 *  result in a STGMEDIUM in case it has a medium other than TYMED_HGLOBAL.
 *  For conveniece we'll centralize all compound document formats here,
 *  hGlobal or not.
 *
 * Parameters:
 *  cf              UINT clipboard format of interest.
 *  pSTM            LSTGMEDIUM to fill.  We only fill the union and tymed.
 *
 * Return Value:
 *  BOOL            TRUE if we could render the format, FALSE otherwise.
 */

BOOL CSchmooDoc::FRenderMedium(UINT cf, LPSTGMEDIUM pSTM)
    {
    if (NULL==pSTM)
        return FALSE;

    if (cf==m_cfEmbedSource)
        {
        /*
         * Embed Source data is an IStorage containing the native
         * data (same as Embedded Object).  Since our data is small,
         * it makes the most sense to create an IStorage in memory
         * and put transfer that instead of a disk-based IStorage.
         */

        pSTM->pstg=OleStdCreateStorageOnHGlobal(NULL, TRUE, STGM_DIRECT
            | STGM_READWRITE | STGM_SHARE_EXCLUSIVE);

        if (NULL==pSTM->pstg)
            return FALSE;

        //Now save the data to the storage.
        WriteClassStg(pSTM->pstg, CLSID_Schmoo2Figure);
        WriteFmtUserTypeStg(pSTM->pstg, m_cf, PSZ(IDS_CLIPBOARDFORMAT));

        if (POLYLINE_E_NONE!=m_pPL->WriteToStorage(pSTM->pstg, VERSIONCURRENT))
            {
            /*
             * When someone releases the IStorage, STORAGE.DLL will release
             * the ILockBytes which, having fDeleteOnRelease=TRUE (second
             * parameter) will release the memory.  That's why we don't have
             * STGM_DELETEONRELEASE on the IStorage.
             */
            pSTM->pstg->Release();
            return FALSE;
            }

        pSTM->tymed=TYMED_ISTORAGE;
        return TRUE;
        }

    //CHAPTER13MOD
    /*
     * CF_OBJECTDESCRIPTOR and CF_LINKSRCDESCRIPTOR are the same
     * formats, but only copy link source if we have a moniker.
     */
    if (cf==m_cfLinkSrcDescriptor && NULL==m_pMoniker)
        return FALSE;

    if (cf==m_cfObjectDescriptor || cf==m_cfLinkSrcDescriptor)
    //End CHAPTER13MOD
        {
        SIZEL   szl, szlT;
        POINTL  ptl;
        RECT    rc;
        //CHAPTER13MOD
        LPSTR   psz=NULL;
        //End CHAPTER13MOD

        m_pPL->SizeGet(&rc);
        SETSIZEL(szlT, rc.right, rc.bottom);
        XformSizeInPixelsToHimetric(NULL, &szlT, &szl);

        SETPOINTL(ptl, 0, 0);

        //CHAPTER13MOD
        //Include the moniker display name now, if we have one.
        if (m_pMoniker)
            {
            LPBC    pbc;

            CreateBindCtx(0, (LPBC FAR*)&pbc);
            m_pMoniker->GetDisplayName(pbc, NULL, &psz);
            pbc->Release();
            }

        pSTM->hGlobal=OleStdGetObjectDescriptorData(CLSID_Schmoo2Figure
            , DVASPECT_CONTENT, szl, ptl, OLEMISC_RECOMPOSEONRESIZE
            , PSZ(IDS_OBJECTDESCRIPTION), psz);
        //End CHAPTER13MOD

        pSTM->tymed=TYMED_HGLOBAL;
        return (NULL!=pSTM->hGlobal);
        }

    //CHAPTER13MOD
    if (cf==m_cfLinkSource)
        {
        if (NULL!=m_pMoniker)
            {
            FORMATETC   fe;
            HRESULT     hr;

            pSTM->tymed=TYMED_NULL;     //Necessary for OleStdGetLinkSourceData
            SETDefFormatEtc(fe, cf, TYMED_ISTREAM);
            hr=OleStdGetLinkSourceData(m_pMoniker
                , (LPCLSID)&CLSID_Schmoo2Figure, &fe, pSTM);

            return SUCCEEDED(hr);
            }
        }
    //End CHAPTER13MOD

    return FALSE;
    }







/*
 * CSchmooDoc::FQueryPaste
 *
 * Purpose:
 *  Determines if we can paste data from the clipboard.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if data is available, FALSE otherwise.
 */

BOOL CSchmooDoc::FQueryPaste(void)
    {
    LPDATAOBJECT    pIDataObject;
    BOOL            fRet;

    if (FAILED(OleGetClipboard(&pIDataObject)))
        return FALSE;

    fRet=FQueryPasteFromData(pIDataObject);
    pIDataObject->Release();
    return fRet;
    }





/*
 * CSchmooDoc::FQueryPasteFromData
 * (Protected)
 *
 * Purpose:
 *  Determines if we can paste data from a data object.
 *
 * Parameters:
 *  pIDataObject    LPDATAOBJECT from which we might want to paste.
 *
 * Return Value:
 *  BOOL            TRUE if data is available, FALSE otherwise.
 */

BOOL CSchmooDoc::FQueryPasteFromData(LPDATAOBJECT pIDataObject)
    {
    FORMATETC       fe;

    SETDefFormatEtc(fe, m_cf, TYMED_HGLOBAL);
    return (NOERROR==pIDataObject->QueryGetData(&fe));
    }




/*
 * CSchmooDoc::FPaste
 *
 * Purpose:
 *  Retrieves the private data format from the clipboard and sets it
 *  to the current figure in the editor window.
 *
 *  Note that if this function is called, then the clipboard format
 *  is available because the Paste menu item is only enabled if the
 *  format is present.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FPaste(HWND hWndFrame)
    {
    LPDATAOBJECT    pIDataObject;
    BOOL            fRet;

    if (FAILED(OleGetClipboard(&pIDataObject)))
        return FALSE;

    fRet=FPasteFromData(pIDataObject);
    pIDataObject->Release();
    return fRet;
    }





/*
 * CSchmooDoc::FPasteFromData
 * (Protected)
 *
 * Purpose:
 *  Retrieves the private data format from a data object and sets it
 *  to the current figure in the editor window.
 *
 * Parameters:
 *  pIDataObject    LPDATAOBJECT from which to paste.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FPasteFromData(LPDATAOBJECT pIDataObject)
    {
    FORMATETC       fe;
    STGMEDIUM       stm;
    BOOL            fRet;

    SETDefFormatEtc(fe, m_cf, TYMED_HGLOBAL);
    fRet=SUCCEEDED(pIDataObject->GetData(&fe, &stm));

    if (fRet && NULL!=stm.hGlobal)
        {
        m_pPL->DataSetMem(stm.hGlobal, FALSE, FALSE, TRUE);
        ReleaseStgMedium(&stm);
        FDirtySet(TRUE);
        }

    return fRet;
    }




/*
 * CSchmooDoc::TransferObjectCreate
 * (Protected)
 *
 * Purpose:
 *  Creates a DataTransferObject and stuffs the current Polyline
 *  data into it, used for both clipboard and drag-drop operations.
 *
 * Parameters:
 *  fCut            BOOL TRUE if we're cutting, FALSE for copying.
 *
 * Return Value:
 *  LPDATAOBJECT    Pointer to the object created, NULL on failure
 */

LPDATAOBJECT CSchmooDoc::TransferObjectCreate(BOOL fCut)
    {
    UINT            i;
    HRESULT         hr;
    STGMEDIUM       stm;
    FORMATETC       fe;
    LPDATAOBJECT    pIDataObject=NULL;
    //CHPATER13MOD
    const UINT      cFormats=7;
    static UINT     rgcf[7]={0, 0, 0, CF_METAFILEPICT, CF_BITMAP, 0, 0};
    static DWORD    rgtm[7]={TYMED_HGLOBAL, TYMED_ISTORAGE, TYMED_HGLOBAL
        , TYMED_MFPICT, TYMED_GDI, TYMED_ISTREAM, TYMED_HGLOBAL};
    //End CHAPTER13MOD

    hr=CoCreateInstance(CLSID_DataTransferObject, NULL, CLSCTX_INPROC_SERVER
        , IID_IDataObject, (LPVOID FAR *)&pIDataObject);

    if (FAILED(hr))
        return NULL;

    rgcf[0]=m_cf;
    rgcf[1]=m_cfEmbedSource;
    rgcf[2]=m_cfObjectDescriptor;
    //CHAPTER13MOD
    //Don't include link stuff for cutting (0 format always fails)
    if (!fCut)
        {
        rgcf[5]=m_cfLinkSource;
        rgcf[6]=m_cfLinkSrcDescriptor;
        }
    //End CHAPTER13MOD

    for (i=0; i < cFormats; i++)
        {
        /*
         * RenderFormat handles memory handles, but for compound doc
         * formats we need something more.  So if RenderFormat fails
         * (which it will for i=1, try our latest addition which
         * writes to a different field in the STGMEDIUM.
         */
        stm.hGlobal=RenderFormat(rgcf[i]);

        if (NULL==stm.hGlobal)
            {
            if (!FRenderMedium(rgcf[i], &stm))
                continue;
            }

        stm.tymed=rgtm[i];
        stm.pUnkForRelease=NULL;
        SETDefFormatEtc(fe, rgcf[i], rgtm[i]);
        pIDataObject->SetData(&fe, &stm, TRUE);
        }

    return pIDataObject;    //Caller now responsible
    }



/*
 * CSchmooDoc::DropSelectTargetWindow
 * (Protected)
 *
 * Purpose:
 *  Creates a thin inverted frame around a window that we use to show
 *  the window as a drop target.  This is a toggle function:  it uses
 *  XOR to create the effect so it must be called twice to leave the
 *  window as it was.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::DropSelectTargetWindow(void)
    {
    HDC         hDC;
    RECT        rc;
    UINT        dd=3;
    HWND        hWnd;

    hWnd=m_pPL->Window();
    hDC=GetWindowDC(hWnd);
    GetClientRect(hWnd, &rc);

    //Frame this window with inverted pixels

    //Top
    PatBlt(hDC, rc.left, rc.top, rc.right-rc.left, dd, DSTINVERT);

    //Bottom
    PatBlt(hDC, rc.left, rc.bottom-dd, rc.right-rc.left, dd, DSTINVERT);

    //Left excluding regions already affected by top and bottom
    PatBlt(hDC, rc.left, rc.top+dd, dd, rc.bottom-rc.top-(2*dd), DSTINVERT);

    //Right excluding regions already affected by top and bottom
    PatBlt(hDC, rc.right-dd, rc.top+dd, dd, rc.bottom-rc.top-(2*dd), DSTINVERT);

    ReleaseDC(hWnd, hDC);
    return;
    }






/*
 * CSchmooDoc::ColorSet
 *
 * Purpose:
 *  Changes a color used in our contained Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to change.
 *  cr              COLORREF new color.
 *
 * Return Value:
 *  COLORREF        Previous color for the given index.
 */

COLORREF CSchmooDoc::ColorSet(UINT iColor, COLORREF cr)
    {
    return m_pPL->ColorSet(iColor, cr);
    }





/*
 * CSchmooDoc::ColorGet
 *
 * Purpose:
 *  Retrieves a color currently in use in the Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to retrieve.
 *
 * Return Value:
 *  COLORREF        Current color for the given index.
 */

COLORREF CSchmooDoc::ColorGet(UINT iColor)
    {
    return m_pPL->ColorGet(iColor);
    }






/*
 * CSchmooDoc::LineStyleSet
 *
 * Purpose:
 *  Changes the line style currently used in the Polyline
 *
 * Parameters:
 *  iStyle          UINT index of the new line style to use.
 *
 * Return Value:
 *  UINT            Previous line style.
 */


UINT CSchmooDoc::LineStyleSet(UINT iStyle)
    {
    return m_pPL->LineStyleSet(iStyle);
    }







/*
 * CSchmooDoc::LineStyleGet
 *
 * Purpose:
 *  Retrieves the line style currently used in the Polyline
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  UINT            Current line style.
 */


UINT CSchmooDoc::LineStyleGet(void)
    {
    return m_pPL->LineStyleGet();
    }








/*
 * CPolylineAdviseSink::CPolylineAdviseSink
 * CPolylineAdviseSink::~CPolylineAdviseSink
 *
 * Constructor Parameters:
 *  pv              LPVOID to store in this object
 */

CPolylineAdviseSink::CPolylineAdviseSink(LPVOID pv)
    {
    m_pv=pv;
    return;
    }


CPolylineAdviseSink::~CPolylineAdviseSink(void)
    {
    return;
    }





/*
 * CPolylineAdviseSink::OnPointChange
 *
 * Purpose:
 *  Informs the document that the polyline added or removed a point.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnPointChange(void)
    {
    LPCDocument pDoc=(LPCDocument)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }






/*
 * CPolylineAdviseSink::OnSizeChange
 *
 * Purpose:
 *  Informs the document that the polyline changed size.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnSizeChange(void)
    {
    LPCSchmooDoc pDoc=(LPCSchmooDoc)m_pv;
    RECT         rc;
    DWORD        dwStyle;

    /*
     * Polyline window is informing us that it changed size in
     * response to setting it's data.  Therefore we have to
     * size ourselves accordingly but without moving the screen
     * position of the polyline window.
     */

    pDoc->m_fNoSize=TRUE;

    //Set the document window size.
    GetWindowRect(pDoc->m_pPL->Window(), &rc);
    InflateRect(&rc, 8, 8);

    //Adjust for a window sans menu
    dwStyle=GetWindowLong(pDoc->m_hWnd, GWL_STYLE);
    AdjustWindowRect(&rc, dwStyle, FALSE);

    SetWindowPos(pDoc->m_hWnd, NULL, 0, 0, rc.right-rc.left
        , rc.bottom-rc.top, SWP_NOMOVE | SWP_NOZORDER);

    if (NULL!=pDoc->m_pAdv)
        pDoc->m_pAdv->OnSizeChange(pDoc, &rc);

    pDoc->m_fNoSize=FALSE;
    pDoc->FDirtySet(TRUE);

    return;
    }





/*
 * CPolylineAdviseSink::OnDataChange
 *
 * Purpose:
 *  Informs the document that the polyline data changed.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnDataChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    if (NULL!=pDoc->m_pAdv)
        pDoc->m_pAdv->OnDataChange(pDoc);

    pDoc->FDirtySet(TRUE);
    return;
    }





/*
 * CPolylineAdviseSink::OnColorChange
 *
 * Purpose:
 *  Informs the document that the polyline data changed a color.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnColorChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }





/*
 * CPolylineAdviseSink::OnLineStyleChange
 *
 * Purpose:
 *  Informs the document that the polyline changed its line style.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnLineStyleChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }
