/*
 * IPERFILE.CPP
 *
 * Implementation of the IPersistFile interface for Patron's documents.
 *
 * Copyright (c)1992 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "patron.h"


/*
 * CImpIPersistFile:CImpIPersistFile
 * CImpIPersistFile::~CImpIPersistFile
 *
 * Constructor Parameters:
 *  pDoc            LPCPatronDoc associated with this interface
 *  punkOuter       LPUNKNOWN of the controlling unknown.
 */

CImpIPersistFile::CImpIPersistFile(LPCPatronDoc pDoc, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pDoc=pDoc;
    m_punkOuter=punkOuter;
    return;
    }


CImpIPersistFile::~CImpIPersistFile(void)
    {
    return;
    }




/*
 * CImpIPersistFile::QueryInterface
 * CImpIPersistFile::AddRef
 * CImpIPersistFile::Release
 *
 * Purpose:
 *  Standard set of IUnknown members for this interface
 */

STDMETHODIMP CImpIPersistFile::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    return m_punkOuter->QueryInterface(riid, ppv);
    }

STDMETHODIMP_(ULONG) CImpIPersistFile::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIPersistFile::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }




/*
 * CImpIPersistFile::GetClassID
 *
 * Purpose:
 *  Returns the CLSID of the file represented by this interface.
 *
 * Parameters:
 *  pClsID          LPCLSID in which to store our CLSID.
 *
 * Return Value:
 *  HRESULT         Standard
 */

STDMETHODIMP CImpIPersistFile::GetClassID(LPCLSID pClsID)
    {
    *pClsID=CLSID_PatronPages;
    return NOERROR;
    }





/*
 * CImpIPersistFile::IsDirty
 *
 * Purpose:
 *  Tells the caller if we have made changes to this file since
 *  it was loaded or initialized new.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  HRESULT         Contains S_OK if we ARE dirty, S_FALSE if NOT dirty,
 *                  that is, "Yes I AM dirty, or NO, I'm clean."
 */

STDMETHODIMP CImpIPersistFile::IsDirty(void)
    {
    return ResultFromScode(m_pDoc->FDirtyGet() ? S_OK : S_FALSE);
    }





/*
 * IPersistFile::Load
 *
 * Purpose:
 *  Asks the server to load the document for the given filename.
 *
 * Parameters:
 *  pszFile         LPCSTR to the filename to load.
 *  grfMode         DWORD containing open flags requested from the caller.
 *                  Currently these are safely ignored.
 *
 * Return Value:
 *  HRESULT         Standard
 */

STDMETHODIMP CImpIPersistFile::Load(LPCSTR pszFile, DWORD grfMode)
    {
    UINT        uRet;

    uRet=m_pDoc->ULoad(TRUE, (LPSTR)pszFile);
    return (DOCERR_NONE==uRet) ? NOERROR : ResultFromScode(STG_E_READFAULT);
    }





/*
 * IPersistFile::Save
 *
 * Purpose:
 *  Instructs the server to write the current file into a new filename,
 *  possibly then using that filename as the current one.
 *
 * Parameters:
 *  pszFile         LPCSTR of the file into which we save.  If NULL,
 *                  this means save the current file.
 *  fRemember       BOOL indicating if we're to use this filename as
 *                  the current file now (Save As instead of Save Copy As).
 *
 * Return Value:
 *  HRESULT         Standard
 */

STDMETHODIMP CImpIPersistFile::Save(LPCSTR pszFile, BOOL fRemember)
    {
    UINT        uRet;

    /*
     * Since we don't want to mess with changing USave (which would
     * require changes to CLASSLIB (urk) we instead save fRemember in
     * the document before calling USave which supresses the call
     * to CPatronDoc::Rename if FALSE.
     */

    m_pDoc->m_fRename=fRemember;
    uRet=m_pDoc->USave(0, (LPSTR)pszFile);
    m_pDoc->m_fRename=TRUE;

    return (DOCERR_NONE==uRet) ? NOERROR : ResultFromScode(STG_E_WRITEFAULT);
    }





/*
 * IPersistFile::SaveCompleted
 *
 * Purpose:
 *  Informs us that the operation that called Save is now finished and
 *  we can access the file again.
 *
 * Parameters:
 *  pszFile         LPCSTR to the file that was saved.
 *
 * Return Value:
 *  HRESULT         Standard
 */

STDMETHODIMP CImpIPersistFile::SaveCompleted(LPCSTR pszFile)
    {
    return NOERROR;
    }




/*
 * IPersistFile::GetCurFile
 *
 * Purpose:
 *  Retrieves the name of the current file.
 *
 * Parameters:
 *  ppszFile        LPSTR FAR * into which we store a pointer to the
 *                  filename that should be allocated with the shared
 *                  IMalloc.
 *
 * Return Value:
 *  HRESULT         Standard
 */

STDMETHODIMP CImpIPersistFile::GetCurFile(LPSTR FAR * ppszFile)
    {
    LPMALLOC    pIMalloc;
    LPSTR       psz;
    UINT        uRet;

    *ppszFile=NULL;

    if (FAILED(CoGetMalloc(MEMCTX_SHARED, &pIMalloc)))
        return ResultFromScode(E_FAIL);

    psz=(LPSTR)pIMalloc->Alloc(OLEUI_CCHPATHMAX);
    pIMalloc->Release();

    uRet=m_pDoc->FilenameGet(psz, OLEUI_CCHPATHMAX);

    //If we have no filename, return the prompt for a File Open/Save.
    if (0==uRet)
        {
        //The load string at psz+2 puts the extension past the "*."
        lstrcpy(psz, "*.");
        LoadString(m_pDoc->m_hInst, IDS_DEFEXT
            , (LPSTR)(psz+2), OLEUI_CCHPATHMAX-2);
        }

    *ppszFile=psz;
    return (0==uRet) ? ResultFromScode(S_FALSE) : NOERROR;
    }
