/*
 * FIGURE.CPP
 * Schmoo Figure Handler Chapter 11
 *
 * Implementation of the CFigure class that we expose as a SchmooFigure
 * Object in this handler.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "hschmoo.h"


/*
 * CFigure:CFigure
 * CFigure::~CFigure
 *
 * Constructor Parameters:
 *  punkOuter       LPUNKNOWN of the controlling unknown.
 *  pfnDestroy      LPFNDESTROYED to call when an object is destroyed.
 *  hInst           HINSTANCE of the application we're in.
 */

CFigure::CFigure(LPUNKNOWN punkOuter, LPFNDESTROYED pfnDestroy
    , HINSTANCE hInst)
    {
    m_cRef=0;
    m_punkOuter=punkOuter;
    m_pfnDestroy=pfnDestroy;
    m_clsID=CLSID_Schmoo2Figure;

    m_cf=RegisterClipboardFormat(SZPOLYLINECLIPFORMAT);

    //NULL any contained interfaces initially.
    m_pIOleObject        =NULL;
    m_pIViewObject       =NULL;
    m_pIPersistStorage   =NULL;
    m_pIAdviseSink       =NULL;

    m_pDefIUnknown       =NULL;
    m_pDefIOleObject     =NULL;
    m_pDefIViewObject    =NULL;
    m_pDefIPersistStorage=NULL;
    m_pDefIDataObject    =NULL;

    m_pIAdvSinkView      =NULL;
    m_dwAdviseFlags      =0;
    m_dwAdviseAspects    =0;
    m_dwFrozenAspects    =0;

    return;
    }


CFigure::~CFigure(void)
    {
    if (NULL!=m_pIAdvSinkView)
        m_pIAdvSinkView->Release();

    if (NULL!=m_pDefIDataObject)
        m_pDefIDataObject->Release();

    if (NULL!=m_pDefIPersistStorage)
        m_pDefIPersistStorage->Release();

    if (NULL!=m_pDefIViewObject)
        m_pDefIViewObject->Release();

    if (NULL!=m_pDefIOleObject)
        m_pDefIOleObject->Release();

    if (NULL!=m_pDefIUnknown)
        m_pDefIUnknown->Release();

    if (NULL!=m_pIAdviseSink)
        delete m_pIAdviseSink;

    if (NULL!=m_pIPersistStorage)
        delete m_pIPersistStorage;

    if (NULL!=m_pIViewObject)
        delete m_pIViewObject;

    if (NULL!=m_pIOleObject)
        delete m_pIOleObject;

    return;
    }




/*
 * CFigure::FInit
 *
 * Purpose:
 *  Performs any intiailization of a CFigure that's prone to failure
 *  that we also use internally before exposing the object outside
 *  this DLL.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if the function is successful, FALSE otherwise.
 */

BOOL CFigure::FInit(void)
    {
    LPUNKNOWN       pIUnknown=(LPUNKNOWN)this;
    HRESULT         hr;
    DWORD           dwConn;
    FORMATETC       fe;

    if (NULL!=m_punkOuter)
        pIUnknown=m_punkOuter;

    //First create our interfaces.
    m_pIOleObject=new CImpIOleObject(this, pIUnknown);

    if (NULL==m_pIOleObject)
        return FALSE;

    m_pIViewObject=new CImpIViewObject(this, pIUnknown);

    if (NULL==m_pIViewObject)
        return FALSE;

    m_pIPersistStorage=new CImpIPersistStorage(this, pIUnknown);

    if (NULL==m_pIPersistStorage)
        return FALSE;

    m_pIAdviseSink=new CImpIAdviseSink(this, pIUnknown);

    if (NULL==m_pIAdviseSink)
        return FALSE;

    /*
     * Get an IUnknown on the default handler, passing pIUnknown
     * as the controlling unknown.
     */
    hr=OleCreateDefaultHandler(CLSID_Schmoo2Figure, pIUnknown, IID_IUnknown
        , (LPLPVOID)&m_pDefIUnknown);

    if (FAILED(hr))
        return FALSE;

    //Now try to get other interfaces to which we delegate
    hr=m_pDefIUnknown->QueryInterface(IID_IOleObject
        , (LPLPVOID)&m_pDefIOleObject);

    if (FAILED(hr))
        return FALSE;

    hr=m_pDefIUnknown->QueryInterface(IID_IViewObject
        , (LPLPVOID)&m_pDefIViewObject);

    if (FAILED(hr))
        return FALSE;

    hr=m_pDefIUnknown->QueryInterface(IID_IDataObject
        , (LPLPVOID)&m_pDefIDataObject);

    if (FAILED(hr))
        return FALSE;

    hr=m_pDefIUnknown->QueryInterface(IID_IPersistStorage
        , (LPLPVOID)&m_pDefIPersistStorage);

    if (FAILED(hr))
        return FALSE;

    //Set up an advise on native data so we can keep in sync
    SETDefFormatEtc(fe, m_cf, TYMED_HGLOBAL);
    m_pDefIDataObject->DAdvise(&fe, 0, m_pIAdviseSink, &dwConn);

    return TRUE;
    }





/*
 * CFigure::QueryInterface
 * CFigure::AddRef
 * CFigure::Release
 *
 * Purpose:
 *  IUnknown members for CFigure object.
 */

STDMETHODIMP CFigure::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    *ppv=NULL;

    /*
     * The only calls we get here for IUnknown are either in a non-aggregated
     * case or when we're created in an aggregation, so in either we always
     * return our IUnknown for IID_IUnknown.
     */
    if (IsEqualIID(riid, IID_IUnknown))
        *ppv=(LPVOID)this;

    if (IsEqualIID(riid, IID_IPersist) || IsEqualIID(riid, IID_IPersistStorage))
        *ppv=(LPVOID)m_pIPersistStorage;

    if (IsEqualIID(riid, IID_IOleObject))
        *ppv=(LPVOID)m_pIOleObject;

    if (IsEqualIID(riid, IID_IViewObject))
        *ppv=(LPVOID)m_pIViewObject;

    //AddRef any interface we'll return.
    if (NULL!=*ppv)
        {
        ((LPUNKNOWN)*ppv)->AddRef();
        return NOERROR;
        }

    //Otherwise see if the default handler knows it.
    return m_pDefIUnknown->QueryInterface(riid, ppv);
    }


STDMETHODIMP_(ULONG) CFigure::AddRef(void)
    {
    return ++m_cRef;
    }


STDMETHODIMP_(ULONG) CFigure::Release(void)
    {
    ULONG       cRefT;

    cRefT=--m_cRef;

    if (0==m_cRef)
        delete this;

    return cRefT;
    }





/*
 * CFigure::Draw
 *
 * Purpose:
 *  Paints the current window to an hDC which might be a printer.
 *
 * Parameters:
 *  hDC             HDC to draw on, could be a metafile or printer DC.
 *  pRect           LPRECT defining the bounds on hDC in which to draw.
 *  dwAspect        DWORD aspect to draw.
 *  ptd             DVTARGETDEVICE FAR * containing device information.
 *  hICDev          HDC containing the IC for the device.
 *  ppl             LPPOLYLINEDATA from which to draw.
 *
 * Return Value:
 *  None
 */

void CFigure::Draw(HDC hDC, LPRECT pRect, DWORD dwAspect
    , DVTARGETDEVICE FAR * ptd, HDC hICDev, LPPOLYLINEDATA ppl)
    {
    HBRUSH          hBrush;
    HPEN            hPen;
    HGDIOBJ         hObj1, hObj2;
    UINT            i, j;
    POINT           pt[2];
    int             nDC;

    nDC=SaveDC(hDC);

    hPen=CreatePen(ppl->iLineStyle, 1, ppl->rgbLine);
    hObj1=SelectObject(hDC, hPen);

    hBrush=CreateSolidBrush(ppl->rgbBackground);
    hObj2=SelectObject(hDC, hBrush);
    SetBkColor(hDC, ppl->rgbBackground);

    //If we only had one point, draw a dot to indicate it's position.
    if (1==ppl->cPoints)
        {
        pt[0]=ppl->rgpt[0];
        PointScale(pRect, &pt[0], TRUE);
        SetPixel(hDC, pt[0].x, pt[0].y, ppl->rgbLine);
        }
    else
        {
        //Erase the background for bitmaps and metafiles.
        SelectObject(hDC, GetStockObject(NULL_PEN));
        Rectangle(hDC, pRect->left, pRect->top, pRect->right+1, pRect->bottom+1);
        SelectObject(hDC, hPen);

        for (i=0; i < ppl->cPoints; i++)
            {
            for (j=i; j < ppl->cPoints; j++)
                {
                pt[0]=ppl->rgpt[i];
                pt[1]=ppl->rgpt[j];
                PointScale(pRect, &pt[0], TRUE);
                PointScale(pRect, &pt[1], TRUE);
                MoveTo(hDC, pt[0].x, pt[0].y);
                LineTo(hDC, pt[1].x, pt[1].y);
                }
            }
        }

    SelectObject(hDC, hObj1);
    SelectObject(hDC, hObj2);
    DeleteObject(hBrush);
    DeleteObject(hPen);

    RestoreDC(hDC, nDC);
    return;
    }







/*
 * CFIgure::PointScale
 *
 * Purpose:
 *  Scales a point from a 0-32767 coordinate to a rectangle relative
 *  coordinate.
 *
 * Parameters:
 *  pRect           LPRECT in which to scale.
 *  ppt             LPPOINT to convert
 *  fUnused         BOOL of no use.
 *
 * Return Value:
 *  None
 */

void CFigure::PointScale(LPRECT pRect, LPPOINT ppt, BOOL fUnused)
    {
    DWORD   cx, cy;

    cx=(DWORD)(pRect->right-pRect->left);
    cy=(DWORD)(pRect->bottom-pRect->top);

    //Must use DWORD to insure proper scaling.
    ppt->x=pRect->left+(UINT)(((DWORD)ppt->x*cx) >> 15);
    ppt->y=pRect->top+(UINT)(((DWORD)ppt->y*cy) >> 15);

    return;
    }
