/*
 * IADVSINK.CPP
 *
 * Implementation of the IAdviseSink interface for the Schmoo Handler
 * such that it will be notified when data is modified in the server.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "hschmoo.h"


/*
 * CImpIAdviseSink::CImpIAdviseSink
 * CImpIAdviseSink::~CImpIAdviseSink
 *
 * Parameters (Constructor):
 *  pObj            LPCFigure of the object we're in.
 *  punkOuter       LPUNKNOWN to which we delegate.
 */

CImpIAdviseSink::CImpIAdviseSink(LPCFigure pObj, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pObj=pObj;
    m_punkOuter=punkOuter;
    return;
    }

CImpIAdviseSink::~CImpIAdviseSink(void)
    {
    return;
    }




/*
 * CImpIAdviseSink::QueryInterface
 * CImpIAdviseSink::AddRef
 * CImpIAdviseSink::Release
 *
 * Purpose:
 *  IUnknown members for CImpIAdviseSink object.
 */

STDMETHODIMP CImpIAdviseSink::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    /*
     * For some reason or another, your IAdviseSink's QueryInterface
     * has to be separate from the rest of the handler's object.  We
     * can still pass on AddRefs and Releases without a big deal, since
     * we're controlled by the outer object's lifetime anyway.
     */
    if (IsEqualIID(riid, IID_IUnknown) || IsEqualIID(riid, IID_IAdviseSink))
        {
        *ppv=(LPVOID)this;
        AddRef();
        return NOERROR;
        }

    return ResultFromScode(E_NOINTERFACE);
    }


STDMETHODIMP_(ULONG) CImpIAdviseSink::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIAdviseSink::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }




/*
 * IAdviseSink::OnViewChange
 *
 * Purpose:
 *  We don't do anything here because we generate OnViewChange for the
 *  container inside OnDataChange.  The problem is that OnViewChange will
 *  come before OnDataChnage, so if we called the container's OnViewChange
 *  here it would turn around and call our IViewObject::Draw which would
 *  draw with outdated data.  Therefore we ignore this notification and
 *  wait for OnDataChange, since that implies a view change as well.  Then
 *  we can retrieve the new data first, then send OnViewChange to the
 *  container such that we'll repaint with the new data.
 */

STDMETHODIMP_(void) CImpIAdviseSink::OnViewChange(DWORD dwAspect, LONG lindex)
    {
    return;
    }



/*
 * IAdviseSink::OnDataChange
 *
 * Purpose:
 *  Tells us that things changed in the server.  We asked for data
 *  on the advise so we can copy it from here into our own structure
 *  such that on the next OnViewChange we can repaint with it.
 */

STDMETHODIMP_(void) CImpIAdviseSink::OnDataChange(LPFORMATETC pFE
    , LPSTGMEDIUM pSTM)
    {
    //Get the new data first, then notify the container to repaint.
    if ((pFE->cfFormat==m_pObj->m_cf) && (TYMED_HGLOBAL & pSTM->tymed))
        {
        LPPOLYLINEDATA      ppl;

        ppl=(LPPOLYLINEDATA)GlobalLock(pSTM->hGlobal);
        _fmemcpy(&m_pObj->m_pl, ppl, CBPOLYLINEDATA);
        GlobalUnlock(pSTM->hGlobal);

        /*
         * Now tell the container that the view changed, but only
         * if the view is not frozen.'
         */
        if (pFE->dwAspect & m_pObj->m_dwAdviseAspects
            && !(pFE->dwAspect & m_pObj->m_dwFrozenAspects))
            {
            //Pass this on to the container.
            if (NULL!=m_pObj->m_pIAdvSinkView)
                {
                m_pObj->m_pIAdvSinkView->OnViewChange(pFE->dwAspect
                    , pFE->lindex);
                }
            }
        }

    return;
    }



/*
 * All others are uninteresting because if the container wants these
 * it will have called IOleObject::Advise which we passed on through to
 * the default handler.  IViewObject::SetAdvise is the only one we
 * override.
 */

STDMETHODIMP_(void) CImpIAdviseSink::OnRename(LPMONIKER pmk)
    {
    return;
    }

STDMETHODIMP_(void) CImpIAdviseSink::OnSave(void)
    {
    return;
    }

STDMETHODIMP_(void) CImpIAdviseSink::OnClose(void)
    {
    return;
    }
