/*
 * IOLEOBJ.CPP
 * Polyline Component Object Chapter 11
 *
 * Implementation of the IOleObject interface for Polyline.  Some of
 * these just pass through to the default handler which does default
 * implementations.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "polyline.h"

//Need this to load strings.
extern HINSTANCE hgInst;



/*
 * CImpIOleObject::CImpIOleObject
 * CImpIOleObject::~CImpIOleObject
 *
 * Parameters (Constructor):
 *  pObj            LPCPolyline of the object we're in.
 *  punkOuter       LPUNKNOWN to which we delegate.
 */

CImpIOleObject::CImpIOleObject(LPCPolyline pObj, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pObj=pObj;
    m_punkOuter=punkOuter;
    return;
    }

CImpIOleObject::~CImpIOleObject(void)
    {
    return;
    }



/*
 * CImpIOleObject::QueryInterface
 * CImpIOleObject::AddRef
 * CImpIOleObject::Release
 *
 * Purpose:
 *  IUnknown members for CImpIOleObject object.
 */

STDMETHODIMP CImpIOleObject::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    return m_punkOuter->QueryInterface(riid, ppv);
    }


STDMETHODIMP_(ULONG) CImpIOleObject::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIOleObject::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }





/*
 * CImpIOleObject::SetClientSite
 * CImpIOleObject::GetClientSite
 *
 * Purpose:
 *  Manages the IOleClientSite pointer of our container.
 */

STDMETHODIMP CImpIOleObject::SetClientSite(LPOLECLIENTSITE pIOleClientSite)
    {
    if (NULL!=m_pObj->m_pIOleClientSite)
        m_pObj->m_pIOleClientSite->Release();

    m_pObj->m_pIOleClientSite=pIOleClientSite;
    m_pObj->m_pIOleClientSite->AddRef();

    return NOERROR;
    }

STDMETHODIMP CImpIOleObject::GetClientSite(LPOLECLIENTSITE FAR * ppSite)
    {
    //Be sure to AddRef the new pointer you are giving away.
    *ppSite=m_pObj->m_pIOleClientSite;
    m_pObj->m_pIOleClientSite->AddRef();

    return NOERROR;
    }





/*
 * CImpIOleObject::SetHostNames
 *
 * Purpose:
 *  Provides the object with names of the container application and the
 *  object in the container to use in object user interface.
 *
 * Parameters:
 *  pszApp          LPCSTR of the container application.
 *  pszObj          LPCSTR of some name that is useful in window titles.
 *
 * Return Value:
 *  HRESULT         NOERROR
 */

STDMETHODIMP CImpIOleObject::SetHostNames(LPCSTR pszApp, LPCSTR pszObj)
    {
    if (NULL!=m_pObj->m_hDlg)
        {
        char        szTemp[128];

        wsprintf(szTemp, SZPOLYFRAMETITLE, pszObj);
        SetWindowText(m_pObj->m_hDlg, szTemp);
        }

    return NOERROR;
    }





/*
 * CImpIOleObject::Close
 *
 * Purpose:
 *  Forces the object to close down its user interface and unload.
 *
 * Parameters:
 *  dwSaveOption    DWORD describing the circumstances under which the
 *                  object is being saved and closed.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, error code otherwise.
 */

STDMETHODIMP CImpIOleObject::Close(DWORD dwSaveOption)
    {
    HWND        hWnd;
    BOOL        fSave=FALSE;

    hWnd=m_pObj->m_hDlg;

    //If object is dirty and we're asked to save, save it and close.
    if (OLECLOSE_SAVEIFDIRTY==dwSaveOption && m_pObj->m_fDirty)
        fSave=TRUE;

    /*
     * If asked to prompt, only do so if dirty, then if we get a YES,
     * save as usual and close.  On NO, just close.  On CANCEL return
     * OLE_E_PROMPTSAVECANCELLED.
     */
    if (OLECLOSE_PROMPTSAVE==dwSaveOption && m_pObj->m_fDirty)
        {
        char        szTemp[80];
        char        szTitle[20];
        UINT        uRet;

        LoadString(hgInst, IDS_CLOSECAPTION, szTitle, sizeof(szTitle));
        LoadString(hgInst, IDS_CLOSEPROMPT, szTemp, sizeof(szTemp));
        MessageBox(hWnd, szTemp, szTitle, MB_YESNOCANCEL);

        if (IDCANCEL==uRet)
            return ResultFromScode(OLE_E_PROMPTSAVECANCELLED);

        if (IDYES==uRet)
            fSave=TRUE;
        }

    if (fSave)
        {
        m_pObj->SendAdvise(OBJECTCODE_SAVEOBJECT);
        m_pObj->SendAdvise(OBJECTCODE_SAVED);
        }

    //We get directly here on OLECLOSE_NOSAVE.
    if (NULL!=hWnd)
        PostMessage(hWnd, WM_CLOSE, 0, 0L);

    return NOERROR;
    }




/*
 * CImpIOleObject::DoVerb
 *
 * Purpose:
 *  Executes an object-defined action.
 *
 * Parameters:
 *  iVerb           LONG index of the verb to execute.
 *  pMSG            LPMSG describing the event causing the activation.
 *  pActiveSite     LPOLECLIENTSITE to the site involved.
 *  lIndex          LONG the piece on which execution is happening.
 *  hWndParent      HWND of the window in which the object can play in-place.
 *  pRectPos        LPRECT of the object in hWndParent where the object
 *                  can play in-place if desired.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, error code otherwise.
 */

STDMETHODIMP CImpIOleObject::DoVerb(LONG iVerb, LPMSG pMSG
    , LPOLECLIENTSITE pActiveSite, LONG lIndex, HWND hWndParent
    , LPCRECT pRectPos)
    {
    switch (iVerb)
        {
        case OLEIVERB_HIDE:
            if (NULL!=m_pObj->m_hDlg)
                ShowWindow(m_pObj->m_hDlg, SW_HIDE);
            break;

        case OLEIVERB_PRIMARY:
        case OLEIVERB_OPEN:
        case OLEIVERB_SHOW:
            //If we already have a window, just make sure it's showing
            if (NULL!=m_pObj->m_hDlg)
                {
                ShowWindow(m_pObj->m_hDlg, SW_NORMAL);
                SetFocus(m_pObj->m_hDlg);
                return NOERROR;
                }

            //This stores the dialog handle in m_pObj.
            DialogBoxParam(hgInst, MAKEINTRESOURCE(IDD_EDITDIALOG)
                , hWndParent, PolyDlgProc, (LPARAM)m_pObj);
            break;

        default:
            return ResultFromScode(OLEOBJ_S_INVALIDVERB);
        }

    return NOERROR;
    }






/*
 * CImpIOleObject::GetUserClassID
 *
 * Purpose:
 *  Used for linked objects, this returns the class ID of what end
 *  users think they are editing.
 *
 * Parameters:
 *  pClsID          LPCLSID in which to store the CLSID.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, error code otherwise.
 */

STDMETHODIMP CImpIOleObject::GetUserClassID(LPCLSID pClsID)
    {
    /*
     * If you are not registered to handle data other than yourself,
     * then you can just return your class ID here.  If you are registered
     * as usable from Treat-As dialogs, then you need to return the
     * CLSID of what you are really editing.
     */

    *pClsID=CLSID_Polyline11;
    return NOERROR;
    }





/*
 * CImpIOleObject::SetExtent
 *
 * Purpose:
 *  Sets the size of the object in HIMETRIC units.
 *
 * Parameters:
 *  dwAspect        DWORD of the aspect affected.
 *  pszl            LPSIZEL containing the new size.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, error code otherwise.
 */

STDMETHODIMP CImpIOleObject::SetExtent(DWORD dwAspect, LPSIZEL pszl)
    {
    RECT            rc;

    if (!(DVASPECT_CONTENT & dwAspect))
        return ResultFromScode(E_FAIL);

    SetRect(&rc, 0, 0, (int)pszl->cx, (int)pszl->cy);
    m_pObj->RectConvertMappings(&rc, TRUE);
    m_pObj->m_pIPolyline->SizeSet(&rc, TRUE);

    return NOERROR;
    }





/*
 * CImpIOleObject::GetExtent
 *
 * Purpose:
 *  Retrieves the size of the object in HIMETRIC units.
 *
 * Parameters:
 *  dwAspect        DWORD of the aspect requested
 *  pszl            LPSIZEL into which to store the size.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, error code otherwise.
 */

STDMETHODIMP CImpIOleObject::GetExtent(DWORD dwAspect, LPSIZEL pszl)
    {
    RECT            rc;

    if (!(DVASPECT_CONTENT & dwAspect))
        return m_pObj->m_pDefIOleObject->GetExtent(dwAspect, pszl);

    m_pObj->m_pIPolyline->RectGet(&rc);
    m_pObj->RectConvertMappings(&rc, FALSE);

    pszl->cx=rc.right-rc.left;
    pszl->cy=rc.bottom-rc.top;

    return NOERROR;
    }





/*
 * CImpIOleObject::Advise
 * CImpIOleObject::Unadvise
 * CImpIOleObject::EnumAdvise
 *
 * Purpose:
 *  Advisory connection functions.
 */

STDMETHODIMP CImpIOleObject::Advise(LPADVISESINK pIAdviseSink, LPDWORD pdwConn)
    {
    if (NULL==m_pObj->m_pIOleAdviseHolder)
        {
        HRESULT     hr;

        hr=CreateOleAdviseHolder(&m_pObj->m_pIOleAdviseHolder);

        if (FAILED(hr))
            return hr;
        }

    return m_pObj->m_pIOleAdviseHolder->Advise(pIAdviseSink, pdwConn);
    }


STDMETHODIMP CImpIOleObject::Unadvise(DWORD dwConn)
    {
    if (NULL!=m_pObj->m_pIOleAdviseHolder)
        return m_pObj->m_pIOleAdviseHolder->Unadvise(dwConn);

    return ResultFromScode(E_FAIL);
    }


STDMETHODIMP CImpIOleObject::EnumAdvise(LPENUMSTATDATA FAR * ppEnum)
    {
    if (NULL!=m_pObj->m_pIOleAdviseHolder)
        return m_pObj->m_pIOleAdviseHolder->EnumAdvise(ppEnum);

    return ResultFromScode(E_FAIL);
    }






//Methods not implemented or trivial

STDMETHODIMP CImpIOleObject::SetMoniker(DWORD dwWhich, LPMONIKER pmk)
    {
    return ResultFromScode(E_NOTIMPL);
    }

STDMETHODIMP CImpIOleObject::GetMoniker(DWORD dwAssign, DWORD dwWhich
    , LPMONIKER FAR * ppmk)
    {
    return ResultFromScode(E_NOTIMPL);
    }

STDMETHODIMP CImpIOleObject::InitFromData(LPDATAOBJECT pIDataObject
    , BOOL fCreation, DWORD dw)
    {
    return ResultFromScode(E_NOTIMPL);
    }

STDMETHODIMP CImpIOleObject::GetClipboardData(DWORD dwReserved
    , LPDATAOBJECT FAR * ppIDataObj)
    {
    return ResultFromScode(E_NOTIMPL);
    }

STDMETHODIMP CImpIOleObject::Update(void)
    {
    return NOERROR;
    }

STDMETHODIMP CImpIOleObject::IsUpToDate(void)
    {
    return NOERROR;
    }

STDMETHODIMP CImpIOleObject::SetColorScheme(LPLOGPALETTE pLP)
    {
    return ResultFromScode(E_NOTIMPL);
    }



//Methods implemented in the default handler.


STDMETHODIMP CImpIOleObject::EnumVerbs(LPENUMOLEVERB FAR * ppEnum)
    {
    return m_pObj->m_pDefIOleObject->EnumVerbs(ppEnum);
    }

STDMETHODIMP CImpIOleObject::GetUserType(DWORD dwForm, LPSTR FAR * ppszType)
    {
    return m_pObj->m_pDefIOleObject->GetUserType(dwForm, ppszType);
    }

STDMETHODIMP CImpIOleObject::GetMiscStatus(DWORD dwAspect, LPDWORD pdwStatus)
    {
    return m_pObj->m_pDefIOleObject->GetMiscStatus(dwAspect, pdwStatus);
    }
