/*
 * DOCUMENT.CPP
 * Component Schmoo Chapter 4
 *
 * Implementation of the CSchmooDoc derivation of CDocument as
 * well as an implementation of CPolylineAdviseSink.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "coschmoo.h"



/*
 * CSchmooDoc::CSchmooDoc
 * CSchmooDoc::~CSchmooDoc
 *
 * Constructor Parameters:
 *  hInst           HINSTANCE of the application.
 */

CSchmooDoc::CSchmooDoc(HINSTANCE hInst)
    : CDocument(hInst)
    {
    m_pPL=NULL;
    m_pPLAdv=NULL;
    m_uPrevSize=SIZE_RESTORED;
    return;
    }


CSchmooDoc::~CSchmooDoc(void)
    {
    //The client takes care of destroying document windows.

    //CHAPTER4MOD
    //Clean up the allocations we did in FInit
    if (NULL!=m_pPL)
        m_pPL->Release();

    if (NULL!=m_pPLAdv)
        m_pPLAdv->Release;

    //Be a good component user and let COMPOBJ.DLL free memory.
    CoFreeUnusedLibraries();
    //End CHAPTER4MOD

    return;
    }






/*
 * CSchmooDoc::FInit
 *
 * Purpose:
 *  Initializes an already created document window.  The client actually
 *  creates the window for us, then passes that here for further
 *  initialization.
 *
 * Parameters:
 *  pDI             LPDOCUMENTINIT containing initialization parameters.
 *
 * Return Value:
 *  BOOL            TRUE if the function succeeded, FALSE otherwise.
 */

BOOL CSchmooDoc::FInit(LPDOCUMENTINIT pDI)
    {
    RECT        rc;

    //Change the stringtable range to our customization.
    pDI->idsMin=IDS_DOCUMENTMIN;
    pDI->idsMax=IDS_DOCUMENTMAX;

    //Do default initialization
    if (!CDocument::FInit(pDI))
        return FALSE;

    //CHAPTER4MOD
    //Create the Polyline Component Object via COMPOBJ.DLL functions.
    CoCreateInstance(CLSID_Polyline4, NULL, CLSCTX_INPROC_SERVER
        , IID_IPolyline4, (LPVOID FAR *)&m_pPL);

    if (NULL==m_pPL)
        return FALSE;

    //Initialize the contained Polyline which creates a window.
    GetClientRect(m_hWnd, &rc);
    InflateRect(&rc, -8, -8);

    if (FAILED(m_pPL->Init(m_hWnd, &rc, WS_CHILD | WS_VISIBLE, ID_POLYLINE)))
        return FALSE;

    //Set up an advise on the Polyline.
    m_pPLAdv=new CPolylineAdviseSink((LPVOID)this);
    m_pPL->SetAdvise(m_pPLAdv);
    //End CHAPTER4MOD

    return TRUE;
    }







/*
 * CSchmooDoc::FMessageHook
 *
 * Purpose:
 *  Processes WM_SIZE for the document so we can resize the Polyline.
 *
 * Parameters:
 *  <WndProc Parameters>
 *  pLRes           LRESULT FAR * in which to store the return value
 *                  for the message.
 *
 * Return Value:
 *  BOOL            TRUE to prevent further processing, FALSE otherwise.
 */

BOOL CSchmooDoc::FMessageHook(HWND hWnd, UINT iMsg, WPARAM wParam
    , LPARAM lParam, LRESULT FAR *pLRes)
    {
    UINT        dx, dy;
    RECT        rc;

    if (WM_SIZE==iMsg)
        {
        //Don't effect the Polyline size to or from minimized state.
        if (SIZE_MINIMIZED!=wParam && SIZE_MINIMIZED !=m_uPrevSize)
            {
            //When we change size, resize any Polyline we hold.
            dx=LOWORD(lParam);
            dy=HIWORD(lParam);

            /*
             * If we are getting WM_SIZE in response to a Polyline
             * notification, then don't resize the Polyline window again.
             */
            if (!m_fNoSize && NULL!=m_pPL)
                {
                //Resize the polyline to fit the new client
                SetRect(&rc, 8, 8, dx-8, dy-8);
                m_pPL->RectSet(&rc, FALSE);

                /*
                 * We consider sizing something that makes the file dirty,
                 * but not until we've finished the create process, which
                 * is why we set fNoDirty to FALSE in WM_CREATE since we
                 * get a WM_SIZE on the first creation.
                 */
                if (!m_fNoDirty)
                    FDirtySet(TRUE);

                SetRect(&rc, 0, 0, dx, dy);

                if (NULL!=m_pAdv)
                    m_pAdv->OnSizeChange((LPCDocument)this, &rc);

                m_fNoDirty=FALSE;
                }
            }

        m_uPrevSize=wParam;
        }

    /*
     * We return FALSE even on WM_SIZE so we can let the default procedure
     * handle maximized MDI child windows appropriately.
     */
    return FALSE;
    }








/*
 * CSchmooDoc::Clear
 *
 * Purpose:
 *  Sets all contents in the document back to defaults with no filename.
 *
 * Paramters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Clear(void)
    {
    //Completely reset the polyline
    m_pPL->New();

    CDocument::Clear();
    return;
    }






/*
 * CSchmooDoc::ULoad
 *
 * Purpose:
 *  Loads a given document without any user interface overwriting the
 *  previous contents of the Polyline window.  We do this by opening
 *  the file and telling the Polyline to load itself from that file.
 *
 * Parameters:
 *  fChangeFile     BOOL indicating if we're to update the window title
 *                  and the filename from using this file.
 *  pszFile         LPSTR to the filename to load, NULL for untitled.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::ULoad(BOOL fChangeFile, LPSTR pszFile)
    {
    HRESULT         hr;

    if (NULL==pszFile)
        {
        //For a new untitled document, just rename ourselved.
        Rename(NULL);
        return DOCERR_NONE;
        }

    //CHAPTER4MOD
    hr=m_pPL->ReadFromFile(pszFile);

    if (POLYLINE_E_READFAILURE==GetScode(hr))
        return DOCERR_READFAILURE;
    //End CHAPTER4MOD

    if (fChangeFile)
        Rename(pszFile);

    //Importing a file makes things dirty
    FDirtySet(!fChangeFile);

    return DOCERR_NONE;
    }







/*
 * CSchmooDoc::USave
 *
 * Purpose:
 *  Writes the file to a known filename, requiring that the user has
 *  previously used FileOpen or FileSaveAs in order to have a filename.
 *
 * Parameters:
 *  uType           UINT indicating the type of file the user requested
 *                  to save in the File Save As dialog.
 *  pszFile         LPSTR under which to save.  If NULL, use the current name.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::USave(UINT uType, LPSTR pszFile)
    {
    BOOL        fRename=TRUE;
    //CHAPTER4MOD
    HRESULT     hr;
    //End CHAPTER4MOD

    if (NULL==pszFile)
        {
        fRename=FALSE;
        pszFile=m_szFile;
        }

    /*
     * In Component Schmoo, we only deal with one version of data,
     * so all the code in Chapter 2 Schmoo that dealt with 1.0 and
     * 2.0 files has been removed.
     */

    //CHAPTER4MOD
    hr=m_pPL->WriteToFile(pszFile);

    if (FAILED(hr))
        return DOCERR_WRITEFAILURE;
    //End CHAPTER4MOD

    //Saving makes us clean
    FDirtySet(FALSE);

    if (fRename)
        Rename(pszFile);

    return DOCERR_NONE;
    }





/*
 * CSchmooDoc::Undo
 *
 * Purpose:
 *  Reverses a previous action.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Undo(void)
    {
    m_pPL->Undo();
    return;
    }






/*
 * CSchmooDoc::FClip
 *
 * Purpose:
 *  Places a private format, a metafile, and a bitmap of the display
 *  on the clipboard, optionally implementing Cut by deleting the
 *  data in the current window after rendering.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window.
 *  fCut            BOOL indicating cut (TRUE) or copy (FALSE).
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FClip(HWND hWndFrame, BOOL fCut)
    {
    BOOL            fRet=TRUE;
    HGLOBAL         hMem;
    UINT            i;

    //This array is so we can loop over the formats we provide.
    static UINT     rgcf[3]={0, CF_METAFILEPICT, CF_BITMAP};
    const UINT      cFormats=3;

    if (!OpenClipboard(hWndFrame))
        return FALSE;

    //Clean out whatever junk is in the clipboard.
    EmptyClipboard();

    rgcf[0]=m_cf;

    for (i=0; i < cFormats; i++)
        {
        //Copy private data first.
        hMem=RenderFormat(rgcf[i]);

        if (NULL!=hMem)
            SetClipboardData(rgcf[i], hMem);
        else
            fRet &=FALSE;
        }

    //Free clipboard ownership.
    CloseClipboard();

    //Delete our current data if copying succeeded.
    if (fRet && fCut)
        {
        m_pPL->New();
        FDirtySet(TRUE);
        }

    return fRet;
    }





/*
 * CSchmooDoc::RenderFormat
 *
 * Purpose:
 *  Renders a specific clipboard format into global memory.  We have this
 *  function split out because we'll eventually move to delayed rendering
 *  and this will then be immediately callable from the frame.
 *
 * Parameters:
 *  cf              UINT format to render.
 *
 * Return Value:
 *  HGLOBAL         Global memory handle containing the data.
 */

HGLOBAL CSchmooDoc::RenderFormat(UINT cf)
    {
    HGLOBAL     hMem;

    if (cf==m_cf)
        {
        //CHAPTER4MOD
        m_pPL->DataGetMem(&hMem);
        //End CHAPTER4MOD
        return hMem;
        }

    switch (cf)
        {
        case CF_METAFILEPICT:
            //CHAPTER4MOD
            m_pPL->RenderMetafilePict(&hMem);
            return hMem;
            //End CHAPTER4MOD

        case CF_BITMAP:
            m_pPL->RenderBitmap((HBITMAP FAR *)&hMem);
            return hMem;
        }

    return NULL;
    }







/*
 * CSchmooDoc::FQueryPaste
 *
 * Purpose:
 *  Determines if we can paste data from the clipboard.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if data is available, FALSE otherwise.
 */

BOOL CSchmooDoc::FQueryPaste(void)
    {
    return IsClipboardFormatAvailable(m_cf);
    }





/*
 * CSchmooDoc::FPaste
 *
 * Purpose:
 *  Retrieves the private data format from the clipboard and sets it
 *  to the current figure in the editor window.
 *
 *  Note that if this function is called, then the clipboard format
 *  is available because the Paste menu item is only enabled if the
 *  format is present.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FPaste(HWND hWndFrame)
    {
    HGLOBAL         hMem;
    LPPOLYLINEDATA  ppl;
    BOOL            fRet=FALSE;

    if (!OpenClipboard(hWndFrame))
        return FALSE;

    hMem=GetClipboardData(m_cf);

    if (NULL!=hMem)
        {
        ppl=(LPPOLYLINEDATA)GlobalLock(hMem);

        //TRUE in wParam to cause PLN_SIZECHANGE notification
        m_pPL->DataSet(ppl, FALSE, TRUE);
        GlobalUnlock(hMem);

        FDirtySet(TRUE);
        fRet=TRUE;
        }

    CloseClipboard();
    return fRet;
    }







/*
 * CSchmooDoc::ColorSet
 *
 * Purpose:
 *  Changes a color used in our contained Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to change.
 *  cr              COLORREF new color.
 *
 * Return Value:
 *  COLORREF        Previous color for the given index.
 */

COLORREF CSchmooDoc::ColorSet(UINT iColor, COLORREF cr)
    {
    //CHAPTER4MOD
    COLORREF    crRet;

    m_pPL->ColorSet(iColor, cr, &crRet);
    return crRet;
    //End CHAPTER4MOD
    }





/*
 * CSchmooDoc::ColorGet
 *
 * Purpose:
 *  Retrieves a color currently in use in the Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to retrieve.
 *
 * Return Value:
 *  COLORREF        Current color for the given index.
 */

COLORREF CSchmooDoc::ColorGet(UINT iColor)
    {
    //CHAPTER4MOD
    COLORREF    crRet;

    m_pPL->ColorGet(iColor, &crRet);
    return crRet;
    //End CHAPTER4MOD
    }






/*
 * CSchmooDoc::LineStyleSet
 *
 * Purpose:
 *  Changes the line style currently used in the Polyline
 *
 * Parameters:
 *  iStyle          UINT index of the new line style to use.
 *
 * Return Value:
 *  UINT            Previous line style.
 */


UINT CSchmooDoc::LineStyleSet(UINT iStyle)
    {
    //CHAPTER4MOD
    UINT    i;

    m_pPL->LineStyleSet(iStyle, &i);
    return i;
    //End CHAPTER4MOD
    }







/*
 * CSchmooDoc::LineStyleGet
 *
 * Purpose:
 *  Retrieves the line style currently used in the Polyline
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  UINT            Current line style.
 */


UINT CSchmooDoc::LineStyleGet(void)
    {
    //CHAPTER4MOD
    UINT    i;

    m_pPL->LineStyleGet(&i);
    return i;
    //End CHAPTER4MOD
    }







/*
 * CPolylineAdviseSink::CPolylineAdviseSink
 *
 * Purpose:
 *  Constructs an advise object storing a pointer for the caller's use.
 *
 * Parameters:
 *  pv              LPVOID to store in this object
 */

CPolylineAdviseSink::CPolylineAdviseSink(LPVOID pv)
    {
    m_pv=pv;
    //CHAPTER4MOD
    m_cRef=0;
    AddRef();
    //End CHAPTER4MOD
    return;
    }



/*
 * CPolylineAdviseSink::~CPolylineAdviseSink
 *
 * Purpose:
 *  Destructor.
 *
 * Parameters:
 *  None
 */

CPolylineAdviseSink::~CPolylineAdviseSink(void)
    {
    return;
    }




//CHAPTER4MOD
/*
 * CPolylineAdviseSink::QueryInterface
 * CPolylineAdviseSink::AddRef
 * CPolylineAdviseSink::Release
 *
 * Purpose:
 *  IUnknown members for this IPolylineAdviseSink implementations.
 */

STDMETHODIMP CPolylineAdviseSink::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    *ppv=NULL;

    //Any interface on this object is the object pointer.
    if (IsEqualIID(riid, IID_IUnknown)
        || IsEqualIID(riid, IID_IPolylineAdviseSink4))
        *ppv=(LPVOID)this;

    /*
     * If we actually assign an interface to ppv we need to AddRef it
     * since we're returning a new pointer.
     */
    if (NULL!=*ppv)
        {
        ((LPUNKNOWN)*ppv)->AddRef();
        return NOERROR;
        }

    return ResultFromScode(S_FALSE);
    }


STDMETHODIMP_(ULONG) CPolylineAdviseSink::AddRef(void)
    {
    return ++m_cRef;
    }


STDMETHODIMP_(ULONG) CPolylineAdviseSink::Release(void)
    {
    ULONG   cRefT;

    cRefT=--m_cRef;

    if (0==m_cRef)
        delete this;

    return cRefT;
    }
//End CHAPTER4MOD




/*
 * CPolylineAdviseSink::OnPointChange
 *
 * Purpose:
 *  Informs the document that the polyline added or removed a point.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnPointChange(void)
    {
    LPCDocument pDoc=(LPCDocument)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }






/*
 * CPolylineAdviseSink::OnSizeChange
 *
 * Purpose:
 *  Informs the document that the polyline changed size.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnSizeChange(void)
    {
    LPCSchmooDoc pDoc=(LPCSchmooDoc)m_pv;
    RECT         rc;
    DWORD        dwStyle;

    //CHAPTER4MOD
    HWND         hWnd;

    /*
     * Polyline window is informing us that it changed size in
     * response to setting it's data.  Therefore we have to
     * size ourselves accordingly but without moving the screen
     * position of the polyline window.
     */

    pDoc->m_fNoSize=TRUE;

    //Set the document window size.
    //CHAPTER4MOD
    pDoc->m_pPL->Window(&hWnd);
    //End CHAPTER4MOD
    GetWindowRect(hWnd, &rc);
    InflateRect(&rc, 8, 8);

    //Adjust for a window sans menu
    dwStyle=GetWindowLong(pDoc->m_hWnd, GWL_STYLE);
    AdjustWindowRect(&rc, dwStyle, FALSE);

    SetWindowPos(pDoc->m_hWnd, NULL, 0, 0, rc.right-rc.left
        , rc.bottom-rc.top, SWP_NOMOVE | SWP_NOZORDER);

    if (NULL!=pDoc->m_pAdv)
        pDoc->m_pAdv->OnSizeChange(pDoc, &rc);

    pDoc->m_fNoSize=FALSE;
    pDoc->FDirtySet(TRUE);

    return;
    }





/*
 * CPolylineAdviseSink::OnDataChange
 *
 * Purpose:
 *  Informs the document that the polyline data changed.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnDataChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    if (NULL!=pDoc->m_pAdv)
        pDoc->m_pAdv->OnDataChange(pDoc);

    pDoc->FDirtySet(TRUE);
    return;
    }





/*
 * CPolylineAdviseSink::OnColorChange
 *
 * Purpose:
 *  Informs the document that the polyline data changed a color.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnColorChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }





/*
 * CPolylineAdviseSink::OnLineStyleChange
 *
 * Purpose:
 *  Informs the document that the polyline changed its line style.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CPolylineAdviseSink::OnLineStyleChange(void)
    {
    LPCSchmooDoc    pDoc=(LPCSchmooDoc)m_pv;

    pDoc->FDirtySet(TRUE);
    return;
    }
