/*
 * DATAOBJ.H
 *
 * Classes that implement the Data Object independent of whether
 * we live in a DLL or EXE.
 *
 * Copyright (c)1993 Microsoft Corporation, All Right Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#ifndef _DATAOBJ_H_
#define _DATAOBJ_H_


#include <windows.h>
#include <ole2.h>
#include <bookguid.h>

//Type for an object-destroyed callback
typedef void (FAR PASCAL *LPFNDESTROYED)(void);


//Data size types
#define DOSIZE_SMALL        0
#define DOSIZE_MEDIUM       1
#define DOSIZE_LARGE        2
#define DOSIZE_CSIZES       3

//Data sizes
#define CCHTEXTSMALL        64      //Text grows * 16 charactrs
#define CCHTEXTMEDIUM       1024
#define CCHTEXTLARGE        16384

#define CXYBITMAPSMALL      16      //Bitmaps grow * 4 pixels
#define CXYBITMAPMEDIUM     64
#define CXYBITMAPLARGE      256

#define CRECMETAFILESMALL   4       //Metafiles grows * 9 records since
#define CRECMETAFILEMEDIUM  12      //the number of actual records is the
#define CRECMETAFILELARGE   36      //square of these numbers.


//Resources for this data object.
#define IDB_MIN             1       //Keep this DOSIZE_SMALL+1
#define IDB_16BY16          1
#define IDB_64BY64          2
#define IDB_256BY256        3
#define IDB_MAX             3


#define IDR_MENU                        1
#define IDR_ADVISORICON                 1

#define IDM_ADVISEITERATIONSMIN         100
#define IDM_ADVISEITERATIONS16          100
#define IDM_ADVISEITERATIONS64          101
#define IDM_ADVISEITERATIONS144         102
#define IDM_ADVISEITERATIONS256         103
#define IDM_ADVISEITERATIONS400         104
#define IDM_ADVISEITERATIONS572         105




/*
 * The DataObject object is implemented in its own class with its own
 * IUnknown to support aggregation.  It contains one CImpIDataObject
 * object that we use to implement the externally exposed interfaces.
 */

#define CFORMATETCGET       3


//DATAOBJ.CPP
LRESULT FAR PASCAL __export AdvisorWndProc(HWND, UINT, WPARAM, LPARAM);


class __far CDataObject : public IUnknown
    {
    //Make any contained interfaces your friends so they can get at you
    friend class CImpIDataObject;
    friend LRESULT FAR PASCAL __export AdvisorWndProc(HWND, UINT, WPARAM, LPARAM);

    protected:
        ULONG               m_cRef;         //Object reference count.
        LPUNKNOWN           m_punkOuter;    //Cont. Unk. for aggregation
        LPFNDESTROYED       m_pfnDestroy;   //Function to call on closure.

        UINT                m_iSize;        //Data size for ::Render*
        HBITMAP             m_rghBmp[DOSIZE_CSIZES];    //Cached bitmaps

        HWND                m_hWndAdvise;   //Popup with Advise menu
        DWORD               m_dwAdvFlags;   //Notification flags.

        //Contained interface implemetation
        LPDATAOBJECT        m_pIDataObject;

        //Other interfaces used, implemented elsewhere
        LPDATAADVISEHOLDER  m_pIDataAdviseHolder;

        //Arrays for IDataObject::EnumFormatEtc
        ULONG               m_cfeGet;
        FORMATETC           m_rgfeGet[CFORMATETCGET];

    protected:
        //Functions for use from IDataObject::GetData
        HRESULT     RenderText(LPSTGMEDIUM);
        HRESULT     RenderBitmap(LPSTGMEDIUM);
        HRESULT     RenderMetafilePict(LPSTGMEDIUM);

    public:
        CDataObject(LPUNKNOWN, LPFNDESTROYED, UINT);
        ~CDataObject(void);

        BOOL FInit(void);

        //Non-delegating object IUnknown
        STDMETHODIMP         QueryInterface(REFIID, LPVOID FAR *);
        STDMETHODIMP_(ULONG) AddRef(void);
        STDMETHODIMP_(ULONG) Release(void);
    };

typedef CDataObject FAR * LPCDataObject;



/*
 * Interface implementations for the CDataObject object.
 */

class __far CImpIDataObject : public IDataObject
    {
    private:
        ULONG           m_cRef;         //Interface reference count.
        LPVOID          m_pObj;         //Back pointer to the object.
        LPUNKNOWN       m_punkOuter;    //Controlling unknown for delegation

    public:
        CImpIDataObject(LPVOID, LPUNKNOWN);
        ~CImpIDataObject(void);

        //IUnknown members that delegate to m_punkOuter.
        STDMETHODIMP         QueryInterface(REFIID, LPVOID FAR *);
        STDMETHODIMP_(ULONG) AddRef(void);
        STDMETHODIMP_(ULONG) Release(void);

        //IDataObject members
        STDMETHODIMP GetData(LPFORMATETC, LPSTGMEDIUM);
        STDMETHODIMP GetDataHere(LPFORMATETC, LPSTGMEDIUM);
        STDMETHODIMP QueryGetData(LPFORMATETC);
        STDMETHODIMP GetCanonicalFormatEtc(LPFORMATETC, LPFORMATETC);
        STDMETHODIMP SetData(LPFORMATETC, STGMEDIUM FAR *, BOOL);
        STDMETHODIMP EnumFormatEtc(DWORD, LPENUMFORMATETC FAR *);
        STDMETHODIMP DAdvise(FORMATETC FAR *, DWORD,  LPADVISESINK, DWORD FAR *);
        STDMETHODIMP DUnadvise(DWORD);
        STDMETHODIMP EnumDAdvise(LPENUMSTATDATA FAR *);
    };



/*
 * IEnumFORMATETC object that is created from IDataObject::EnumFormatEtc.
 * This object lives on its own, that is, QueryInterface only knowns
 * IUnknown and IEnumFormatEtc, nothing more.  We still use an outer
 * unknown for reference counting, because as long as this enumerator
 * lives, the data object should live, thereby keeping the application up.
 */

class __far CEnumFormatEtc : public IEnumFORMATETC
    {
    private:
        ULONG           m_cRef;         //Object reference count.
        LPUNKNOWN       m_punkRef;      //IUnknown for reference counting.
        ULONG           m_iCur;         //Current element.
        ULONG           m_cfe;          //Number of FORMATETCs in us
        LPFORMATETC     m_prgfe;        //Source of FORMATETCs

    public:
        CEnumFormatEtc(LPUNKNOWN, ULONG, LPFORMATETC);
        ~CEnumFormatEtc(void);

        //IUnknown members that delegate to m_punkOuter.
        STDMETHODIMP         QueryInterface(REFIID, LPVOID FAR *);
        STDMETHODIMP_(ULONG) AddRef(void);
        STDMETHODIMP_(ULONG) Release(void);

        //IEnumFORMATETC members
        STDMETHODIMP Next(ULONG, LPFORMATETC, ULONG FAR *);
        STDMETHODIMP Skip(ULONG);
        STDMETHODIMP Reset(void);
        STDMETHODIMP Clone(IEnumFORMATETC FAR * FAR *);
    };


typedef CEnumFormatEtc FAR * LPCEnumFormatEtc;



#endif //_DATAOBJ_H_
