/*
 * IPOLYLIN.CPP
 * Polyline Component Object Chapter 6
 *
 * Implementation of the IPolyline interface that we expose on the
 * CPolyline object.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "polyline.h"


/*
 * CImpIPolyline:CImpIPolyline
 * CImpIPolyline::~CImpIPolyline
 *
 * Constructor Parameters:
 *  pObj            LPVOID pointing to the object we live in.
 *  punkOuter       LPUNKNOWN of the controlling unknown.
 */

CImpIPolyline::CImpIPolyline(LPVOID pObj, LPUNKNOWN punkOuter)
    {
    m_cRef=0;
    m_pObj=pObj;
    m_punkOuter=punkOuter;
    return;
    }


CImpIPolyline::~CImpIPolyline(void)
    {
    return;
    }




/*
 * CImpIPolyline::QueryInterface
 * CImpIPolyline::AddRef
 * CImpIPolyline::Release
 *
 * Purpose:
 *  Standard set of IUnknown members for this interface
 */

STDMETHODIMP CImpIPolyline::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    return m_punkOuter->QueryInterface(riid, ppv);
    }

STDMETHODIMP_(ULONG) CImpIPolyline::AddRef(void)
    {
    ++m_cRef;
    return m_punkOuter->AddRef();
    }

STDMETHODIMP_(ULONG) CImpIPolyline::Release(void)
    {
    --m_cRef;
    return m_punkOuter->Release();
    }





/*
 * CImpIPolyline::Init
 *
 * Purpose:
 *  Instantiates a polyline window within a given parent.  The
 *  parent may be a main application window, could be an MDI child
 *  window. We really do not care.
 *
 * Parameters:
 *  hWndParent      HWND of the parent of this window
 *  pRect           LPRECT that this window should occupy
 *  dwStyle         DWORD containing the window's style flags
 *  uID             UINT ID to associate with this window
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, otherwise E_OUTOFMEMORY
 */

STDMETHODIMP CImpIPolyline::Init(HWND hWndParent, LPRECT pRect
    , DWORD dwStyle, UINT uID)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    SCODE           sc;

    pObj->m_hWnd=CreateWindowEx(WS_EX_NOPARENTNOTIFY, SZCLASSPOLYLINE
        , SZCLASSPOLYLINE, dwStyle, pRect->left, pRect->top
        , pRect->right-pRect->left, pRect->bottom-pRect->top
        , hWndParent, (HMENU)uID, pObj->m_hInst, (LPVOID)pObj);


    sc=(NULL!=pObj->m_hWnd) ? S_OK : E_OUTOFMEMORY;
    return ResultFromScode(sc);
    }




//CHAPTER6MOD
/*
 * ::Data* and ::Render* memebers moved into IDataObject.
 */
//CHAPTER6MOD







/*
 * CImpIPolyline::New
 *
 * Purpose:
 *  Cleans out and reinitializes the data to defaults.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  HRESULT         NOERROR always
 */

STDMETHODIMP CImpIPolyline::New(void)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    LPPOLYLINEDATA  ppl=&pObj->m_pl;
    UINT            i;

    ppl->wVerMaj=VERSIONMAJOR;
    ppl->wVerMin=VERSIONMINOR;

    //Our rectangle is the size of our window's client area.
    GetClientRect(pObj->m_hWnd, &ppl->rc);

    //Clean out the POLYLINEDATA structure and repaint the window.
    for (i=0; i< CPOLYLINEPOINTS; i++)
        {
        ppl->rgpt[i].x=0;
        ppl->rgpt[i].y=0;
        }

    ppl->cPoints      =0;
    ppl->rgbBackground=GetSysColor(COLOR_WINDOW);
    ppl->rgbLine      =GetSysColor(COLOR_WINDOWTEXT);
    ppl->iLineStyle   =PS_SOLID;

    InvalidateRect(pObj->m_hWnd, NULL, TRUE);
    UpdateWindow(pObj->m_hWnd);

    pObj->m_fDirty=TRUE;

    //CHAPTER6MOD
    //Inform the advise sink of this data change.
    if (NULL!=pObj->m_pIDataAdviseHolder)
        {
        pObj->m_pIDataAdviseHolder->SendOnDataChange(pObj->m_pIDataObject
            , DVASPECT_CONTENT, ADVF_NODATA);
        }
    //End CHAPTER6MOD

    return NOERROR;
    }






/*
 * CImpIPolyline::Undo
 *
 * Purpose:
 *  Reverses previous actions in a Polyline.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  HRESULT         S_OK if we can Undo more, S_FALSE otherwise.
 */

STDMETHODIMP CImpIPolyline::Undo(void)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    SCODE           sc;

    //Decrement the number of active points and repaint.
    if (pObj->m_pl.cPoints > 0)
        {
        pObj->m_pl.cPoints--;
        InvalidateRect(pObj->m_hWnd, NULL, TRUE);
        UpdateWindow(pObj->m_hWnd);
        }

    if (NULL!=pObj->m_pAdv)
        {
        pObj->m_fDirty=TRUE;
        pObj->m_pAdv->OnPointChange();
        }

    //Return if we can undo any more.
    sc=(0!=pObj->m_pl.cPoints) ? S_OK : S_FALSE;
    return ResultFromScode(sc);
    }






/*
 * CImpIPolyline::Window
 *
 * Purpose:
 *  Returns the window handle associated with this polyline.
 *
 * Parameters:
 *  phWnd           HWND FAR * in which to return the window handle.
 *
 * Return Value:
 *  HRESULT         NOERROR always.
 */

STDMETHODIMP CImpIPolyline::Window(HWND FAR *phWnd)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;

    *phWnd=pObj->m_hWnd;
    return NOERROR;
    }






/*
 * CImpIPolyline::SetAdvise
 *
 * Purpose:
 *  Provides this object with an IPolylineAdviseSink that's interested
 *  in our notifications.  We AddRef and store this pointer, Releasing
 *  the old one.
 *
 * Parameters:
 *  pAdv            LPPOLYLINEADVISESINK to notify.
 *
 * Return Value:
 *  HRESULT         NOERROR always.
 */

STDMETHODIMP CImpIPolyline::SetAdvise(LPPOLYLINEADVISESINK pAdv)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;

    if (NULL!=pObj->m_pAdv)
        pObj->m_pAdv->Release();

    pObj->m_pAdv=pAdv;

    if (NULL!=pObj->m_pAdv)
        pObj->m_pAdv->AddRef();

    return NOERROR;
    }




/*
 * CImpIPolyline::GetAdvise
 *
 * Purpose:
 *  Returns the IPolylineAdviseSink that was last passed to SetAdvise.
 *
 * Parameters:
 *  ppAdv           LPPOLYLINEADVISESINK FAR * in which to return the
 *                  pointer.
 *
 * Return Value:
 *  HRESULT         NOERROR always.
 */

STDMETHODIMP CImpIPolyline::GetAdvise(LPPOLYLINEADVISESINK FAR *ppAdv)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;

    *ppAdv=pObj->m_pAdv;
    return NOERROR;
    }






/*
 * CImpIPolyline::RectGet
 *
 * Purpose:
 *  Returns the rectangle of the Polyline in parent coordinates.
 *
 * Parameters:
 *  pRect           LPRECT in which to return the rectangle.
 *
 * Return Value:
 *  HRESULT         NOERROR always
 */

STDMETHODIMP CImpIPolyline::RectGet(LPRECT pRect)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    RECT            rc;
    POINT           pt;

    //Retrieve the size of our rectangle in parent coordinates.
    GetWindowRect(pObj->m_hWnd, &rc);
    pt.x=rc.left;
    pt.y=rc.top;
    ScreenToClient(GetParent(pObj->m_hWnd), &pt);

    SetRect(pRect, pt.x, pt.y, pt.x+(rc.right-rc.left)
        , pt.y+(rc.bottom-rc.top));

    return NOERROR;
    }





/*
 * CImpIPolyline::SizeGet
 *
 * Purpose:
 *  Retrieves the size of the Polyline in parent coordinates.
 *
 * Parameters:
 *  pRect           LPRECT in which to return the size.  The right and
 *                  bottom fields will contain the dimensions.
 *
 * Return Value:
 *  HRESULT         NOERROR always
 */

STDMETHODIMP CImpIPolyline::SizeGet(LPRECT pRect)
    {
    RectGet(pRect);
    return NOERROR;
    }






/*
 * CImpIPolyline::RectSet
 *
 * Purpose:
 *  Sets a new rectangle for the Polyline which sizes to fit.
 *
 * Parameters:
 *  pRect           LPRECT containing the new rectangle.
 *  fNotify         BOOL indicating if we're to notify anyone of the change.
 *
 * Return Value:
 *  HRESULT         NOERROR always
 */

STDMETHODIMP CImpIPolyline::RectSet(LPRECT pRect, BOOL fNotify)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    UINT            cx, cy;

    //Scale the points from our current size to the new size
    cx=pRect->right-pRect->left;
    cy=pRect->bottom-pRect->top;

    SetWindowPos(pObj->m_hWnd, NULL, pRect->left, pRect->top
        , (UINT)cx, (UINT)cy, SWP_NOZORDER);

    if (fNotify && NULL!=pObj->m_pAdv)
        {
        pObj->m_fDirty=TRUE;
        pObj->m_pAdv->OnSizeChange();
        }

    InvalidateRect(pObj->m_hWnd, NULL, TRUE);
    UpdateWindow(pObj->m_hWnd);

    return NOERROR;
    }







/*
 * CImpIPolyline::SizeSet
 *
 * Purpose:
 *  Sets a new size for the Polyline which sizes to fit.
 *
 * Parameters:
 *  pRect           LPRECT containing the new rectangle.
 *  fNotify         BOOL indicating if we're to notify anyone of the change.
 *
 * Return Value:
 *  HRESULT         NOERROR always
 */

STDMETHODIMP CImpIPolyline::SizeSet(LPRECT pRect, BOOL fNotify)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    UINT            cx, cy;

    //Scale the points from our current size to the new size
    cx=pRect->right-pRect->left;
    cy=pRect->bottom-pRect->top;

    SetWindowPos(pObj->m_hWnd, NULL, 0, 0, (UINT)cx, (UINT)cy
        , SWP_NOMOVE | SWP_NOZORDER);

    if (fNotify && NULL!=pObj->m_pAdv)
        {
        pObj->m_fDirty=TRUE;
        pObj->m_pAdv->OnSizeChange();
        }

    InvalidateRect(pObj->m_hWnd, NULL, TRUE);
    UpdateWindow(pObj->m_hWnd);

    return NOERROR;
    }






/*
 * CImpIPolyline::ColorSet
 *
 * Purpose:
 *  Changes for background or line color in the Polyline
 *
 * Parameters:
 *  iColor          UINT index of the color to change.
 *  cr              COLORREF new color to use.
 *  pcrPrev         COLORREF FAR * in whch to store the previous color.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, otherwise a POLYLINE_E_ value.
 */

STDMETHODIMP CImpIPolyline::ColorSet(UINT iColor, COLORREF cr
    , COLORREF FAR *pcrPrev)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    COLORREF        crRet;

    if (NULL==pcrPrev)
        return ResultFromScode(POLYLINE_E_INVALIDPOINTER);

    switch (iColor)
        {
        case POLYLINECOLOR_BACKGROUND:
            crRet=pObj->m_pl.rgbBackground;
            pObj->m_pl.rgbBackground=cr;
            break;

        case POLYLINECOLOR_LINE:
            crRet=pObj->m_pl.rgbLine;
            pObj->m_pl.rgbLine=cr;
            break;
        }

    //If the color changed, repaint
    if (crRet!=cr)
        {
        if (NULL!=pObj->m_pAdv)
            {
            pObj->m_fDirty=TRUE;
            pObj->m_pAdv->OnColorChange();
            }

        InvalidateRect(pObj->m_hWnd, NULL, TRUE);
        UpdateWindow(pObj->m_hWnd);
        }

    *pcrPrev=crRet;
    return NOERROR;
    }







/*
 * CImpIPolyline::ColorGet
 *
 * Purpose:
 *  Retrieves one of the colors currently in use by the Polyline.
 *
 * Parameters:
 *  iColor          UINT identifying the color of interest.
 *  pcr             COLORREF FAR * in which to return the color.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, otherwise a POLYLINE_E_ value.
 */

STDMETHODIMP CImpIPolyline::ColorGet(UINT iColor, COLORREF FAR *pcr)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    COLORREF        crRet;

    if (NULL==pcr)
        return ResultFromScode(POLYLINE_E_INVALIDPOINTER);

    crRet=(POLYLINECOLOR_BACKGROUND==iColor)
        ? pObj->m_pl.rgbBackground : pObj->m_pl.rgbLine;

    *pcr=crRet;
    return NOERROR;
    }








/*
 * CImpIPolyline::LineStyleSet
 *
 * Purpose:
 *  Changes the line style in use by the Polyline
 *
 * Parameters:
 *  iStyle          UINT style of the line to use.
 *  piPrev          UINT FAR * in which to store the previous style.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, otherwise a POLYLINE_E_ value.
 */

STDMETHODIMP CImpIPolyline::LineStyleSet(UINT iStyle, UINT FAR *piPrev)
    {
    LPCPolyline     pObj=(LPCPolyline)m_pObj;
    UINT            uRet;

    uRet=(UINT)pObj->m_pl.iLineStyle;

    if (NULL==piPrev)
        return ResultFromScode(POLYLINE_E_INVALIDPOINTER);

    //Validate the line style
    if (PS_SOLID==iStyle || PS_DASH==iStyle || PS_DOT==iStyle
        || PS_DASHDOT==iStyle || PS_DASHDOTDOT==iStyle)
        {
        pObj->m_pl.iLineStyle=iStyle;

        if (uRet!=(UINT)pObj->m_pl.iLineStyle)
            {
            if (NULL!=pObj->m_pAdv)
                {
                pObj->m_fDirty=TRUE;
                pObj->m_pAdv->OnLineStyleChange();
                }

            InvalidateRect(pObj->m_hWnd, NULL, TRUE);
            UpdateWindow(pObj->m_hWnd);
            }
        }

    *piPrev=uRet;
    return NOERROR;
    }







/*
 * CImpIPolyline::LineStyleGet
 *
 * Purpose:
 *  Retrieves the current line style in use in the Polyline
 *
 * Parameters:
 *  piStyle         UINT FAR * in which to store the style.
 *
 * Return Value:
 *  HRESULT         NOERROR if successful, otherwise a POLYLINE_E_ value.
 */

STDMETHODIMP CImpIPolyline::LineStyleGet(UINT FAR *piStyle)
    {
    if (NULL==piStyle)
        return ResultFromScode(POLYLINE_E_INVALIDPOINTER);

    *piStyle=((LPCPolyline)m_pObj)->m_pl.iLineStyle;
    return NOERROR;
    }
