/*
 * CSTRTABL.CPP
 * Sample Code Class Libraries
 *
 * Implementation of a string table handler.  The CStringTable
 * class hides details of storage from the user.  The strings might
 * be cached, or they might be loaded as necessary.  In either case,
 * we must know the number of strings so we know whether or not to
 * reload strings.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include <windows.h>
#include <malloc.h>
#include "classlib.h"




/*
 * CStringTable::CStringTable
 * CStringTable::~CStringTable
 *
 * Constructor Parameters:
 *  hInst           HANDLE to the application instance from which we
 *                  load strings.
 */

CStringTable::CStringTable(HINSTANCE hInst)
    {
    m_hInst=hInst;
    m_pszStrings=NULL;
    m_ppszTable=NULL;
    return;
    }


CStringTable::~CStringTable(void)
    {
    if (NULL!=m_pszStrings)
        _ffree(m_pszStrings);

    if (NULL!=m_ppszTable)
        _ffree(m_ppszTable);

    return;
    }







/*
 * CStringTable::FInit
 *
 * Purpose:
 *  Initialization function for a StringTable that is prone to failure.
 *  If this fails then the caller is responsible for guaranteeing that
 *  the destructor is called quickly.
 *
 * Parameters:
 *  idsMin          UINT first identifier in the stringtable
 *  idsMax          UINT last identifier in the stringtable.
 *
 * Return Value:
 *  BOOL            TRUE if the function is successful, FALSE otherwise.
 */


BOOL CStringTable::FInit(UINT idsMin, UINT idsMax)
    {
    UINT        i;
    UINT        cch;
    UINT        cchUsed;
    LPSTR       psz;

    m_idsMin=idsMin;
    m_idsMax=idsMax;
    m_cStrings=(idsMax-idsMin+1);


    //Allocate space for the pointer table.
    m_ppszTable=(LPSTR FAR *)_fmalloc(sizeof(LPSTR)*m_cStrings);

    if (NULL==m_ppszTable)
        return FALSE;


    /*
     * Allocate enough memory for cStrings*256 characters.  256 characters
     * is the maximum string length we allow.
     */
    m_pszStrings=(LPSTR)_fmalloc(m_cStrings << 8);

    if (NULL==m_pszStrings)
        {
        _ffree(m_ppszTable);
        m_ppszTable=NULL;
        return FALSE;
        }


    /*
     * Load the strings:  we load each string in turn into psz,
     * store the string pointer into the table and increment psz
     * to the next positions.
     */

    psz=m_pszStrings;

    for (i=idsMin; i <= idsMax; i++)
        {
        m_ppszTable[i-idsMin]=psz;
        cch=LoadString(m_hInst, i, psz, 255);

        //Account for a null terminator with +1
        psz    +=cch+1;
        cchUsed+=cch;
        }

    //Now reallocate the string memory to however much we used, plus 1
    psz=(LPSTR)_frealloc(m_pszStrings, cchUsed+1);

    if (NULL!=psz)
        m_pszStrings=psz;

    return TRUE;
    }






/*
 * CStringTable::operator[]
 *
 * Purpose:
 *  Returns a pointer to the requested string in the stringtable or
 *  NULL if the specified string does not exist.
 */

const LPSTR CStringTable::operator[](const UINT uID) const
    {
    if (uID < m_idsMin || uID > m_idsMax)
        return NULL;

    return (const LPSTR)m_ppszTable[uID-m_idsMin];
    }
