/*
 * DATAOBJ.CPP
 *
 * Implementation of the CDataObject and CImpIDataObject that work
 * in a DLL.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 */


#include "dataobj.h"

extern HINSTANCE   g_hInst;

DWORD              g_dwID=0;

/*
 * CDataObject::CDataObject
 * CDataObject::~CDataObject
 *
 * Parameters (Constructor):
 *  punkOuter       LPUNKNOWN of a controlling unknown, if it exists.
 *  pfnDestroy      LPFNDESTROYED to call when an object is destroyed.
 *  iSize           UINT specifying the size of the data set to use.
 */

CDataObject::CDataObject(LPUNKNOWN punkOuter, LPFNDESTROYED pfnDestroy
    , UINT iSize)
    {
    m_cRef=0;
    m_punkOuter=punkOuter;
    m_pfnDestroy=pfnDestroy;
    m_iSize=iSize;

    m_hWndAdvise=NULL;
    m_dwAdvFlags=ADVF_NODATA;

    //NULL any contained interfaces initially.
    m_pIDataObject=NULL;
    m_pIDataAdviseHolder=NULL;

    //IDataAdviseHolder is disabled as defualt
    m_UseDataAdviseHolder = 0;

    //Initialize STGMEDIUM and alloc 64 bytes;
    m_stg.tymed = TYMED_HGLOBAL;
    m_stg.hGlobal = GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE, sizeof(MARKETDATA));
    return;
    }


CDataObject::~CDataObject(void)
    {
    if (NULL!=m_pIDataAdviseHolder)
        m_pIDataAdviseHolder->Release();

    if (NULL!=m_pIDataObject)
        delete m_pIDataObject;

    if (NULL!=m_hWndAdvise)
        DestroyWindow(m_hWndAdvise);

    GlobalFree(m_stg.hGlobal);
    return;
    }



/*
 * CDataObject::FInit
 *
 * Purpose:
 *  Performs any intialization of a CDataObject that's prone to failure
 *  that we also use internally before exposing the object outside.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if the function is successful, FALSE otherwise.
 */

BOOL CDataObject::FInit(void){
    LPUNKNOWN       pIUnknown=(LPUNKNOWN)this;
    UINT            cy;

    if (NULL!=m_punkOuter)
        pIUnknown=m_punkOuter;

    //Allocate contained interfaces.
    m_pIDataObject=new CImpIDataObject(this, pIUnknown);

    if (NULL==m_pIDataObject){
        return FALSE;
    }


    /*
     * Register the Advise window class first time through (g_dwID==0)
     */

    if (0==g_dwID)
        {
        WNDCLASS    wc;

        wc.style          = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc    = AdvisorWndProc;
        wc.cbClsExtra     = 0;
        wc.cbWndExtra     = sizeof(LPCDataObject);
        wc.hInstance      = g_hInst;
        wc.hIcon          = LoadIcon(g_hInst, MAKEINTRESOURCE(IDR_ADVISORICON));
        wc.hCursor        = NULL;
        wc.hbrBackground  = (HBRUSH)(COLOR_WINDOW + 1);
        wc.lpszMenuName   = MAKEINTRESOURCE(IDR_MENU);
        wc.lpszClassName  = "Advisor";

        if (!RegisterClass(&wc))
            return FALSE;
        }

    g_dwID++;

    cy=(GetSystemMetrics(SM_CYBORDER)*2)+GetSystemMetrics(SM_CYMENU)
        + GetSystemMetrics(SM_CYCAPTION);

    m_hWndAdvise=CreateWindow("Advisor", "IAdviseSink Update Interface"
        , WS_OVERLAPPED | WS_CAPTION | WS_MINIMIZEBOX | WS_BORDER | WS_VISIBLE
        , CW_USEDEFAULT, CW_USEDEFAULT, 200, cy, HWND_DESKTOP
        , NULL, g_hInst, this);

    if (NULL==m_hWndAdvise)
        return FALSE;

    return TRUE;
}





/*
 * CDataObject::QueryInterface
 * CDataObject::AddRef
 * CDataObject::Release
 *
 * Purpose:
 *  IUnknown members for CDataObject object.
 */

STDMETHODIMP CDataObject::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    *ppv=NULL;

    if (IsEqualGUID(riid, IID_IUnknown))
        *ppv=(LPVOID)this;

    if (IsEqualGUID(riid, IID_IDataObject))
        *ppv=(LPVOID)m_pIDataObject;

    //AddRef any interface we'll return.
    if (NULL!=*ppv)
        {
        ((LPUNKNOWN)*ppv)->AddRef();
        return NOERROR;
        }

    return ResultFromScode(E_NOINTERFACE);
    }


STDMETHODIMP_(ULONG) CDataObject::AddRef(void)
    {
    return ++m_cRef;
    }


STDMETHODIMP_(ULONG) CDataObject::Release(void)
    {
    ULONG       cRefT;

    cRefT=--m_cRef;

    if (0==m_cRef)
        {
        /*
         * Tell the housing that an object is going away so it can
         * shut down if appropriate.
         */
        if (NULL!=m_pfnDestroy)
            (*m_pfnDestroy)();

        delete this;
        }

    return cRefT;
    }


/*
 * AdvisorWndProc
 *
 * Purpose:
 *  Window class procedure.  Standard callback.
 *
 * Parameters:
 *  Standard
 *
 * Return Value:
 *  Standard
 *
 */


LRESULT FAR PASCAL __export AdvisorWndProc(HWND hWnd, UINT iMsg
    , WPARAM wParam, LPARAM lParam)
    {
    LPCDataObject   pDO;
    UINT            iAdviseSecs;
    DWORD           dwAdviseCount;
    DWORD           dwStartTime;
    DWORD   	    dwAvgAdviseCount;
    char            szTime[128];
    char            szTitle[80];
    HCURSOR         hCur, hCurT;

    //This will be valid for all messages except WM_NCCREATE
    pDO=(LPCDataObject)GetWindowLong(hWnd, 0);

    switch (iMsg)
        {
        case WM_NCCREATE:
            pDO=(LPCDataObject)((LONG)((LPCREATESTRUCT)lParam)->lpCreateParams);

            SetWindowLong(hWnd, 0, (LONG)pDO);
            return (DefWindowProc(hWnd, iMsg, wParam, lParam));

        case WM_CLOSE:
            //Eat to forbid task manager from closing us.
            return 0L;

        case WM_COMMAND:
	{
	    FORMATETC   fe;
	    HMENU	hMenu;

	    if(wParam == IDM_USEDATAADVISEHOLDER){
		hMenu = GetMenu(hWnd);

	    	pDO->m_UseDataAdviseHolder = !pDO->m_UseDataAdviseHolder;
		CheckMenuItem(hMenu, wParam, pDO->m_UseDataAdviseHolder	? MF_CHECKED : MF_UNCHECKED);
		break;	
	    }	

            //Send IAdviseSink::OnDataChange many times.
            iAdviseSecs=(LOWORD(wParam)-IDM_ADVISEMIN);
	    dwAdviseCount = 0;

	    //Display hourglass
            hCur=LoadCursor(NULL, MAKEINTRESOURCE(IDC_WAIT));
            hCurT=SetCursor(hCur);
            ShowCursor(TRUE);

            dwStartTime=GetTickCount();

            while (dwStartTime + (iAdviseSecs * 1000) > GetTickCount() )
                {						  
		    if(pDO->m_UseDataAdviseHolder){
		        if (NULL==pDO->m_pIDataAdviseHolder)
                		break;

        	        pDO->m_pIDataAdviseHolder->SendOnDataChange(
                	    pDO->m_pIDataObject, NULL, ADVF_NODATA);
		    }else{ 

/* NOTE: fe.cfFormat is | with NODATAADVISE to inform the client not
to attempt to call GetData when it receives this advise. I am already sending
OnDataChange data directly. You will not implement NODATAADVISEHOLDER in 
your code because you will either use IDataAdviseHolder or not use it.  
This sample code demostrates performance with and without it. */

		        SETDefFormatEtc(fe, CF_TEXT | NODATAADVISEHOLDER, TYMED_HGLOBAL);
		       	pDO->RenderSymbol((LPSTGMEDIUM) &pDO->m_stg);
		    	pDO->m_pIAdviseSink->OnDataChange(&fe, &pDO->m_stg);
		    }
		    dwAdviseCount ++;	
                }

	    dwAvgAdviseCount = (DWORD) (dwAdviseCount / iAdviseSecs);	

	    // Display default cursor
            SetCursor(hCurT);
            ShowCursor(FALSE);

            wsprintf(szTime, "Total\t=%lu updates\n\rAverage\t=%lu updates/sec"
                , dwAdviseCount, dwAvgAdviseCount);

            GetWindowText(hWnd, szTitle, sizeof(szTitle));
            MessageBox(hWnd, szTime, szTitle, MB_OK);
         }
	 break;

        default:
            return (DefWindowProc(hWnd, iMsg, wParam, lParam));
        }

    return 0L;
    }


