********************************************************************
* This module implements the XBRA protocol for use with C programs *
*
* Added in the update from : 29/5/90.
*     KKP      + added switch for parameters on the stack.
*     KKP      + 'Free registers' is a REG directive.
*
* - Klaus K. Pedersen (micro@imada.dk)


************************** EXPORTS *********************************
export XB_isinstal, XB_install, XB_remove, XB_kill


********************* NOTES (HOW TO PORT) **************************
TurboC  	EQU 1           ;Set if compiler passes params in registers
FreeRegs 	REG D0-D2/A0-A1 ;The registers which the compiler consider
                            ;free.
;The routines uses the registers D0-D2/A0-A1 (the free regs in TC)
;If your compiler don't consider these free, then you are in for a
;major rewrite.


************************** ENTRY POINTS ****************************
*    Entry points :
*        XB_isinstal(long vector, char *magic);
*           If 'magic' is in XBRA chain 'vector' : return 1
*           NOTE : This can be called with an 8 byte 'magic' which
*                  contains the "XBRA" and '_id_' only!
*			NOTE : Make sure that the 'magic' struct is wordaligned!
*
*        XB_install(long vector, char *magic, void *routine);
*           Installs an userdefined routine.
*			NOTE : Make sure that the 'magic' struct is wordaligned!
*
*        XB_remove(long vector, char *magic);
*           removes the routine from the vector, returns 1 if the
*           routine could be found.
*           NOTE : This can be called with an 8 byte 'magic' which
*                  contains the "XBRA" and '_id_' only!
*			NOTE : Make sure that the 'magic' struct is wordaligned!
*
*        XB_kill(long vector, char *magic);  (the -9 option!)
*           Like 'XB_remove' but will allways remove the routine
*           from the vector (also if an non XBRA routine has taken the
*           vector [- this will also be killed]).
*           This routine depends on the fact that 'Magic' contains 
*           the 'Oldvec' field. It uses this to unchain all the 
*           routines after the last XBRA routine til your routine.
*           NOTE : This _must_ be called with the same 'magic' struct
*                  as when the routine was installed...
*


************************* PARAMETERS *******************************
*    Parameters :
*        vector:
*           Passed first. This is the address of the vector to 
*           intercept.
*
*        magic:
*           Passed second. Is a pointer to a ~32 bytes struct,
*           with the string "XBRA" and followed by your 4 byte
*           id-string : ex. "_id_".
*			NOTE : Make sure that the 'magic' struct is wordaligned!
*
*       routine:
*           Passed last. This is a pointer to your C routine.

************************ XBRA STRUCTURE ****************************
* The XBRA structure looks as follows:
*    xb_magic : dc.b "XBRA"
*    xb_id    : dc.b "xxxx"         ;Your ID.
     xb_oldvec: dc.l 0              ;Storage for Old vector.
     xb_entry : 
         MOVEM.L #FreeRegs, -(SP)
     xb_myvector:
         JSR $80000                 ;Call 'routine' (self modifying)
         MOVEM.L (SP)+, #FreeRegs  
         MOVE.L xb_oldvec(PC), -(SP);Push 'xb_oldvec'.
         RTS                        ;Call next routine in chain.
     xb_end:
********************************************************************



********************************************************************
* Routine 'XB_isinstal'
* entry : 
*	vector, Pointer to 'Magic',
* exit  : 
*	D0 : Retval (1=was in chain).
*	D1 : ->'Prev_routine',   \ if routine was in chain.
*	A1 : ->->'later_routine' /
*  or:
*	D1 : ->'last_routine',   \ if routine was not in chain.
*	A0+4:->->'later_routine' /
* uses:
*   D0-D2/A0-A1
********************************************************************
XB_isinstal:
	IFF TurboC
		MOVEM.L 4(A7), D0/A0
	ENDIF
xb_search:                  ;Entry-point for other 'XB_*'
	* entry:( A0:Pointer to 'Magic', D0:vector ). *
	MOVE.L D0, A1           ;A1 now points to 'vector'
	MOVE.L A0, D2           ;D2 contains copy of pointer to 'magic' 
xb_next:
	MOVE.L A1, D1           ;At end D1 points to old 'next' pointer
	MOVE.L (A1), A1         ;Find next XBRA struct.
	LEA    -12(A1), A1
	MOVE.L D2, A0           ;Copy pointer to magic into A0 
	CMP.L (A0)+, (A1)+      ;Is 'XBRA' ok ?
	BNE.B not_xbra
	CMP.L (A0)+, (A1)+      ;Is ID ok ?
	BNE.B xb_next           ;Try next in chain...
xb_inchain:
	MOVEQ #1, D0
	RTS


********************************************************************
* Routine 'XB_install'
* entry: (vector, Pointer to 'Magic', Pointer to 'routine').
*
* exit: 
*	D0 : Retval (1=Error-was in chain, 0=OK-installed).
* uses:
*   D0-D2/A0-A1 ; D2 is only used by xb_search
********************************************************************
XB_install:
	IFF TurboC
		MOVEM.L 4(A7), D0/A0-A1
	ENDIF
	MOVE.L A1, xb_myvector+2 ;Write 'routine' into prototype code...
	IF TurboC
		MOVEM.L D0/A0, -(A7) ;Push D0 and A0.
		BSR.B xb_search
		MOVEM.L (A7)+, A1/A0 ;Pop D0 into A0, A0 into A1.
	ELSE
		BSR.B xb_search
		MOVEM.L 4(A7), A0/A1
	ENDIF
	BNE.B xb_inchain         ;Now A0->vector, A1->magic
	MOVE.L (A0), 8(A1)       ;Write old vector into 'xb_oldvec'

	LEA 12(A1), A1			 ;A1->text of magic.
	
	MOVEQ #0, D1             ;Set index register at zero.
	MOVE #(xb_end-xb_entry)/4,D0 ;Nr of longs to move...
xb_stuff:                    ;Copy long at a time.
	MOVE.L xb_entry(PC,D1.w),(A1,D1.w)
	ADDQ #4, D1
	DBRA D0, xb_stuff
	
	MOVE.L A1, (A0)          ;Write 'xb_entry' into the vector...
not_xbra:
	MOVEQ #0, D0
	RTS


********************************************************************
* Routine 'XB_remove'
* entry: (vector, Pointer to 'Magic').
*
* exit: 
*	D0 : Retval (1=OK-was in chain, 0=Error-id not found).
* uses:
*   D0-D2/A0-A1 ; D2 is only used by xb_search
********************************************************************
XB_remove:
	IFF TurboC
		MOVEM.L 4(A7), D0/A0
	ENDIF
	BSR.B xb_search
	BZE.B xb_cantremove      ;If routine is hidden - return (0)
xb_rem:                      ;(A1)->XBRA.xb_oldvec of later routine.
	MOVE.L D1, A0            ;D1->XBRA.xb_oldvec of routine before our.
	MOVE.L (A1), (A0)        ;Make previous routine point on later one.
xb_cantremove:				 ;(ie Unlink routine...)
	RTS


********************************************************************
* Routine 'XB_kill'  (the -9 option!)
* Note !
*   This routine depends on the fact that 'Magic' contains the 
*   'Oldvec' field. It uses this to unchain all the routines after
*   the last XBRA routine til your routine.
*
* entry: (vector, Pointer to 'Magic').
*
* exit  : 
*	D0 : Retval (1=was in chain - OK [-but removed, not killed]).
* uses:
*   D0-D2/A0-A1 ; D2 is only used by xb_search
********************************************************************
XB_kill:
	IFF TurboC
		MOVEM.L 4(A7), D0/A0
	ENDIF
	BSR.B xb_search          ;Get last XBRA routine in chain.            
	BNZ.B xb_rem             ;If the routine was found - dispose normally
                             ;A0+4 is our pointer to next in chain. 
	MOVE.L D1, A1            ;D1 is the Last XBRA pointer to next routine
	MOVE.L 4(A0), (A1)       ;Make previous XBRA point on later routine.
	RTS
	