/****************************************************************************
*
*						      PM/Lite Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: pmlite.c $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	IBM PC (MSDOS) Real mode and 16/32 bit Protected Mode
*
* Description:	Module implementing DOS extender independant protected
*				mode programming. This module will need to be included in
*				all programs that use SciTech Software's products that
*				are to be compiled in protected mode, and will need to be
*				compiled with the correct defines for the DOS extender that
*				you will be using (or with no defines for real mode
*				emulation of these routines).
*
*				The Lite version of this library is freely redistributable
*				in source code form.
*
* $Id: pmlite.c 1.1 1995/02/06 12:43:43 kjb release $
*
****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include "pmode.h"

#pragma pack(1)

/*--------------------------- Global variables ----------------------------*/

/* Set the type of target being compiled for so that we can do switching at
 * runtime as well as compile time.
 */

#if	defined(REALMODE)
int	_PM_modeType = PM_realMode;
#elif defined(PM286)
int _PM_modeType = PM_286;
#elif defined(PM386)
int _PM_modeType = PM_386;
#endif

/*----------------------------- Implementation ----------------------------*/

/*-------------------------------------------------------------------------*/
/* Generic routines common to all extenders								   */
/*-------------------------------------------------------------------------*/

unsigned _cdecl PM_getBIOSSelector(void)
{
	static unsigned BIOS_sel = 0;
	if (!BIOS_sel)
		BIOS_sel = PM_createSelector(0x400,0xFFFF);
	return BIOS_sel;
}

unsigned _cdecl PM_getVGASelector(void)
{
	switch (PM_getByte(PM_getBIOSSelector(),0x49)) {
		case 0x3:
			return PM_getVGAColorTextSelector();
		case 0x7:
			return PM_getVGAMonoTextSelector();
		default:
			return PM_getVGAGraphSelector();
		}
}

unsigned _cdecl PM_getVGAColorTextSelector(void)
{
	static long	VGA_sel = 0;
	if (!VGA_sel)
		VGA_sel = PM_createSelector(0xB8000L,0xFFFF);
	return (unsigned)VGA_sel;
}

unsigned _cdecl PM_getVGAMonoTextSelector(void)
{
	static long	VGA_sel = 0;
	if (!VGA_sel)
		VGA_sel = PM_createSelector(0xB0000L,0xFFFF);
	return (unsigned)VGA_sel;
}

unsigned _cdecl PM_getVGAGraphSelector(void)
{
	static long	VGA_sel = 0;
	if (!VGA_sel)
		VGA_sel = PM_createSelector(0xA0000L,0xFFFF);
	return (unsigned)VGA_sel;
}

int _cdecl PM_int386(int intno, PMREGS *in, PMREGS *out)
{
	PMSREGS	sregs;
	PM_segread(&sregs);
	return PM_int386x(intno,in,out,&sregs);
}

/* Routines to set and get the real mode interrupt vectors, by making
 * direct real mode calls to DOS and bypassing the DOS extenders API.
 * This is the safest way to handle this, as some servers try to be
 * smart about changing real mode vectors.
 */

void _cdecl _PM_getRMvect(int intno, long *realisr)
{
	RMREGS	regs;
	RMSREGS	sregs;

	PM_saveDS();
	regs.h.ah = 0x35;
	regs.h.al = intno;
	PM_int86x(0x21, &regs, &regs, &sregs);
	*realisr = ((long)sregs.es << 16) | regs.x.bx;
}

void _cdecl _PM_setRMvect(int intno, long realisr)
{
	RMREGS	regs;
	RMSREGS	sregs;

	PM_saveDS();
	regs.h.ah = 0x25;
	regs.h.al = intno;
	sregs.ds = (int)(realisr >> 16);
	regs.x.dx = (int)(realisr & 0xFFFF);
	PM_int86x(0x21, &regs, &regs, &sregs);
}

#ifdef PM386

/* Some DOS extender implementations do not directly support calling a
 * real mode procedure from protected mode. However we can simulate what
 * we need temporarily hooking the INT 2Fh vector with a small real mode
 * stub that will call our real mode code for us.
 */

static unsigned char int2FHandler[] = {
	0x00,0x00,0x00,0x00,		/* 	__PMODE_callReal variable			*/
	0xFB,						/*	sti									*/
	0x2E,0xFF,0x1E,0x00,0x00,	/*  call	[cs:__PMODE_callReal]		*/
	0xCF,						/*	iretf								*/
	};
static unsigned crSel=0,crOff;	/* Selector:offset of int 2F handler	*/
static unsigned crRSeg,crROff;	/* Real mode seg:offset of handler		*/

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	unsigned		psel,poff;
	unsigned		oldSeg,oldOff;

	if (!crSel) {
		/* Allocate and copy the memory block only once */
		PM_allocRealSeg(sizeof(int2FHandler), &crSel, &crOff,
			&crRSeg, &crROff);
		PM_memcpyfn(crSel,crOff,int2FHandler,sizeof(int2FHandler));
		}
	PM_setWord(crSel,crOff,off);		/* Plug in address to call	*/
	PM_setWord(crSel,crOff+2,seg);
	PM_mapRealPointer(&psel,&poff,0,0x2F * 4);
	oldOff = PM_getWord(psel,poff);		/* Save old handler address	*/
	oldSeg = PM_getWord(psel,poff+2);
	PM_setWord(psel,poff,crROff+4);		/* Hook 2F handler			*/
	PM_setWord(psel,poff+2,crRSeg);
	PM_int86x(0x2F, in, in, sregs);		/* Call real mode code		*/
	PM_setWord(psel,poff,oldOff);		/* Restore old handler		*/
	PM_setWord(psel,poff+2,oldSeg);
}

#endif

/*-------------------------------------------------------------------------*/
/* DOS Real Mode support.												   */
/*-------------------------------------------------------------------------*/

#ifdef REALMODE

#ifndef	MK_FP
#define MK_FP(s,o)  ( (void far *)( ((unsigned long)(s) << 16) + \
					(unsigned long)(o) ))
#endif

unsigned char _cdecl PM_getByte(unsigned s, unsigned o)
{ return *((unsigned char*)MK_FP(s,o)); }

unsigned short _cdecl PM_getWord(unsigned s, unsigned o)
{ return *((unsigned*)MK_FP(s,o)); }

unsigned long _cdecl PM_getLong(unsigned s, unsigned o)
{ return *((unsigned long*)MK_FP(s,o)); }

void _cdecl PM_setByte(unsigned s, unsigned o, unsigned char v)
{ *((unsigned char*)MK_FP(s,o)) = v; }

void _cdecl PM_setWord(unsigned s, unsigned o, unsigned short v)
{ *((unsigned short *)MK_FP(s,o)) = v; }

void _cdecl PM_setLong(unsigned s, unsigned o, unsigned long v)
{ *((unsigned long*)MK_FP(s,o)) = v; }

void _cdecl PM_memcpynf(void *dst,unsigned src_s,unsigned src_o,unsigned n)
{ memcpy(dst,MK_FP(src_s,src_o),n); }

void _cdecl PM_memcpyfn(unsigned dst_s,unsigned dst_o,void *src,unsigned n)
{ memcpy(MK_FP(dst_s,dst_o),src,n); }

void _cdecl PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{ *sel = r_seg; *off = r_off; }

unsigned _cdecl PM_createSelector(unsigned long base,unsigned long limit)
{
	limit = limit;
	return (unsigned)(base >> 4);
}

void _cdecl PM_freeSelector(unsigned sel)
{ sel = sel; }

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	/* Call malloc() to allocate the memory for us */
	void *p = malloc(size);
	*sel = *r_seg = FP_SEG(p);
	*off = *r_off = FP_OFF(p);
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{
	free(MK_FP(sel,off));
}

int _cdecl PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	return int86(intno,in,out);
}

int _cdecl PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	return int86x(intno,in,out,sregs);
}

int _cdecl PM_int386x(int intno, PMREGS *in, PMREGS *out, PMSREGS *sregs)
{
	return int86x(intno,in,out,sregs);
}

void _cdecl PM_segread(PMSREGS *sregs)
{
	segread(sregs);
}

#endif

/*-------------------------------------------------------------------------*/
/* Windows 3.1 16 bit DPMI and Borland DPMI16 DOS Power Pack support.	   */
/*-------------------------------------------------------------------------*/

#if defined(WINDPMI16) || defined(DPMI16)
#define	GENERIC_DPMI

unsigned char _cdecl PM_getByte(unsigned s, unsigned o)
{ return *((unsigned char*)MK_FP(s,o)); }

unsigned short _cdecl PM_getWord(unsigned s, unsigned o)
{ return *((unsigned*)MK_FP(s,o)); }

unsigned long _cdecl PM_getLong(unsigned s, unsigned o)
{ return *((unsigned long*)MK_FP(s,o)); }

void _cdecl PM_setByte(unsigned s, unsigned o, unsigned char v)
{ *((unsigned char*)MK_FP(s,o)) = v; }

void _cdecl PM_setWord(unsigned s, unsigned o, unsigned short v)
{ *((unsigned short *)MK_FP(s,o)) = v; }

void _cdecl PM_setLong(unsigned s, unsigned o, unsigned long v)
{ *((unsigned long*)MK_FP(s,o)) = v; }

void _cdecl PM_memcpynf(void *dst,unsigned src_s,unsigned src_o,unsigned n)
{ memcpy(dst,MK_FP(src_s,src_o),n); }

void _cdecl PM_memcpyfn(unsigned dst_s,unsigned dst_o,void *src,unsigned n)
{ memcpy(MK_FP(dst_s,dst_o),src,n); }

void _cdecl PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	static	unsigned	staticSel = 0;

	if (staticSel)
		PM_freeSelector(staticSel);
	staticSel = PM_createSelector(MK_PHYS(r_seg,r_off), 0xFFFF);
	*sel = staticSel;
	*off = 0;
}

unsigned _cdecl PM_createCode32Alias(unsigned sel)
{
	unsigned	alias;
	union REGS	r;

	r.x.ax = 0xA;
	r.x.bx = sel;
	int86(0x31,&r,&r);			/* Create Alias descriptor				*/
	if (r.x.cflag) return 0;
	alias = r.x.ax;

	/* Set descriptor access rights (for a 32 bit code segment)			*/
	r.x.ax = 9;
	r.x.bx = alias;
	r.x.cx = 0x40FB;
	int86(0x31, &r, &r);
	if (r.x.cflag) return 0;
    return alias;
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = (int)rmregs.e##reg
#define OUT1(reg)   in->x.reg = (int)rmregs.e##reg

void _cdecl PM_callRealMode(unsigned seg,unsigned off, RMREGS *in,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;
	void			*p;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	rmregs.cs = seg;
	rmregs.ip = off;

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x301;					/* DPMI call real mode			*/
	r.h.bh = 0;
	r.x.cx = 0;
	p = &rmregs;
	sr.es = FP_SEG(p);
	r.x.di = FP_OFF(p);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT1(ax); OUT1(bx); OUT1(cx); OUT1(dx); OUT1(si); OUT1(di);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
}

int _cdecl PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;
	void			*p;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	p = &rmregs;
	sr.es = FP_SEG(p);
	r.x.di = FP_OFF(p);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int _cdecl PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;
	void			*p;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	memset(&sr, 0, sizeof(sr));
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	p = &rmregs;
	sr.es = FP_SEG(p);
	r.x.di = FP_OFF(p);
	int86x(0x31, &r, &r, &sr);		/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int _cdecl PM_int386x(int intno, PMREGS *in, PMREGS *out, PMSREGS *sregs)
{
	return int86x(intno,in,out,sregs);
}

void _cdecl PM_segread(PMSREGS *sregs)
{
	segread(sregs);
}

#endif

/*-------------------------------------------------------------------------*/
/* Windows 3.1 16 bit DPMI specific support.	   						   */
/*-------------------------------------------------------------------------*/

#if defined(WINDPMI16)

#include <windows.h>

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	DWORD	val;

	if ((val = GlobalDosAlloc(size)) == 0)
		return 0;

	*sel = (WORD)val;
	*off = 0;
	*r_seg = (WORD)(val >> 16);
	*r_off = 0;
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{ GlobalDosFree(sel); off = off; }

#endif

/*-------------------------------------------------------------------------*/
/* Borland DPMI16 DOS Power Pack support.	   							   */
/*-------------------------------------------------------------------------*/

#if defined(DPMI16)

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	union REGS		r;

	r.x.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.x.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.x.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.x.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{
	union REGS	r;

	r.x.ax = 0x101;					/* DPMI free DOS memory			*/
	r.x.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	int86(0x31, &r, &r);
}

#endif

/*-------------------------------------------------------------------------*/
/* Phar Lap TNT DOS Extender support.									   */
/*-------------------------------------------------------------------------*/

#ifdef TNT

#include <pldos32.h>
#include <pharlap.h>
#include <hw386.h>

void _cdecl PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	CONFIG_INF	config;

	_dx_config_inf(&config, (UCHAR*)&config);
	*sel = config.c_dos_sel;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned _cdecl _PL_allocsel(void);

unsigned _cdecl PM_createSelector(unsigned long base,unsigned long limit)
{
	USHORT		sel;
	ULONG		off;
	CD_DES		desc;

	if ((sel = _PL_allocsel()) == 0)
		return 0;
	if (base > 0xFFFFFL) {
		/* Map in the physical memory above the 1Mb memory boundary */
		_dx_map_phys(sel,base,(limit + 4095) / 4096,&off);
		}
	else {
		if (_dx_ldt_rd(sel,(UCHAR*)&desc) != 0)
			return 0;
		if (limit >= 0x100000) {
			limit >>= 12;			/* Page granular for > 1Mb limit	*/
			desc.limit0_15 = limit & 0xFFFF;
			desc.limit16_19 = ((limit >> 16) & 0xF) | DOS_32 | SG_PAGE;
			}
		else {						/* Byte granular for < 1Mb limit	*/
			desc.limit0_15 = limit & 0xFFFF;
			desc.limit16_19 = ((limit >> 16) & 0xF) | DOS_32 | SG_BYTE;
			}
		desc.base0_15 = base & 0xFFFF;
		desc.base16_23 = (base >> 16) & 0xFF;
		desc.base24_31 = (base >> 24) & 0xFF;
		desc.arights = AR_DATA | AR_DPL3;
		if (_dx_ldt_wr(sel,(UCHAR*)&desc) != 0)
			return 0;
		}
	return sel;
}

void _cdecl PM_freeSelector(unsigned sel)
{
	CD_DES		desc;

	/* Set the selector's values to all zero's, so that Phar Lap will
	 * re-use the selector entry.
	 */
	if (_dx_ldt_rd(sel,(UCHAR*)&desc) != 0)
		return;
	desc.limit0_15 = 0;
	desc.limit16_19 = 0;
	desc.base0_15 = 0;
	desc.base16_23 = 0;
	desc.base24_31 = 0;
	desc.arights = 0;
	_dx_ldt_wr(sel,(UCHAR*)&desc);
}

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	CONFIG_INF	config;
	USHORT      addr,t;

	if (_dx_real_alloc((size + 0xF) >> 4,&addr,&t) != 0)
		return 0;
	_dx_config_inf(&config, (UCHAR*)&config);
	*sel = config.c_dos_sel;		/* Map with DOS 1Mb selector	*/
	*off = (ULONG)addr << 4;
	*r_seg = addr;					/* Real mode segment address	*/
	*r_off = 0;						/* Real mode segment offset		*/
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{
	sel = sel;
	_dx_real_free(off >> 4);
}

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg

int _cdecl PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	_dx_real_int(intno,&rmregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int _cdecl PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	SWI_REGS	rmregs;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

    _dx_real_int(intno,&rmregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Symantec C++ DOSX and FlashTek X-32/X-32VM support					   */
/*-------------------------------------------------------------------------*/

#if	defined(DOSX) || defined(X32VM)

#ifdef	X32VM
#include <x32.h>

#define	_x386_mk_protected_ptr(p)	_x32_mk_protected_ptr((void*)p)
#define	_x386_free_protected_ptr(p)	_x32_free_protected_ptr(p)
#endif

void _cdecl PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	*sel = _x386_zero_base_selector;
	*off = MK_PHYS(r_seg,r_off);
}

unsigned _cdecl PM_createSelector(unsigned long base,unsigned long limit)
{
	void _far *p;

	p = (void _far *)_x386_mk_protected_ptr(base);
	return FP_SEG(p);
}

void _cdecl PM_freeSelector(unsigned sel)
{
	_x386_free_protected_ptr(MK_FP(sel,0));
}

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	PMREGS	r;

	r.h.ah = 0x48;					/* DOS function 48h - allocate mem	*/
	r.x.bx = (size + 0xF) >> 4;		/* Number of paragraphs to allocate	*/
	PM_int386(0x21, &r, &r);		/* Call DOS extender				*/
	if (r.x.cflag)
		return 0;					/* Could not allocate the memory	*/
	*sel = _x386_zero_base_selector;
	*off = r.e.eax << 4;
	*r_seg = r.e.eax;
	*r_off = 0;
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{
	/* Cannot de-allocate this memory */
	sel = sel; off = off;
}

typedef struct {
	unsigned short	intno;
	unsigned short	ds;
	unsigned short	es;
	unsigned short	fs;
	unsigned short	gs;
	unsigned long	eax;
	unsigned long	edx;
	} _RMREGS;

#define IN(reg)     regs.e.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = regs.x.reg

int _cdecl PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS rmregs;
	PMREGS  regs;
	PMSREGS	pmsregs;

	rmregs.intno = intno;
	rmregs.eax = in->x.ax;
	rmregs.edx = in->x.dx;
	IN(bx); IN(cx); IN(si); IN(di);
	regs.x.ax = 0x2511;
	regs.e.edx = (unsigned)(&rmregs);
	PM_segread(&pmsregs);
	PM_int386x(0x21,&regs,&regs,&pmsregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(si); OUT(di);
	out->x.dx = rmregs.edx;
	out->x.cflag = regs.x.cflag;
	return out->x.ax;
}

int _cdecl PM_int86x(int intno, RMREGS *in, RMREGS *out, RMSREGS *sregs)
{
	_RMREGS rmregs;
	PMREGS  regs;
	PMSREGS	pmsregs;

	rmregs.intno = intno;
	rmregs.eax = in->x.ax;
	rmregs.edx = in->x.dx;
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;
	IN(bx); IN(cx); IN(si); IN(di);
	regs.x.ax = 0x2511;
	regs.e.edx = (unsigned)(&rmregs);
	PM_segread(&pmsregs);
	PM_int386x(0x21,&regs,&regs,&pmsregs);

	OUT(ax); OUT(bx); OUT(cx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->ds = rmregs.ds;
	out->x.dx = rmregs.edx;
	out->x.cflag = regs.x.cflag;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Borland's DPMI32 DOS Power Pack Extender support.					   */
/*-------------------------------------------------------------------------*/

#ifdef  DPMI32
#define	GENERIC_DPMI32			/* Use generic 32 bit DPMI routines	*/
#endif

/*-------------------------------------------------------------------------*/
/* Watcom C/C++ with Rational DOS/4GW support.							   */
/*-------------------------------------------------------------------------*/

#ifdef	DOS4GW
#define	GENERIC_DPMI32			/* Use generic 32 bit DPMI routines	*/
#endif

/*-------------------------------------------------------------------------*/
/* DJGPP port of GNU C++ support.										   */
/*-------------------------------------------------------------------------*/

#ifdef DJGPP
#define	GENERIC_DPMI32			/* Use generic 32 bit DPMI routines	*/
#endif

/*-------------------------------------------------------------------------*/
/* Generic 32 bit DPMI routines											   */
/*-------------------------------------------------------------------------*/

#if	defined(GENERIC_DPMI32) || defined(WINDPMI32)
#define	GENERIC_DPMI

void _cdecl PM_mapRealPointer(unsigned *sel,unsigned *off,unsigned r_seg,
	unsigned r_off)
{
	static unsigned lowMem = 0;
	if (!lowMem)
		lowMem = PM_createSelector(0, 0xFFFFF);
	*sel = lowMem;
	*off = MK_PHYS(r_seg,r_off);
}

int _cdecl PM_allocRealSeg(unsigned size, unsigned *sel,unsigned *off,
	unsigned *r_seg, unsigned *r_off)
{
	PMREGS		r;

	r.x.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.x.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	PM_int386(0x31, &r, &r);
	if (r.x.cflag) return NULL;		/* DPMI call failed				*/

	*sel = r.x.dx;					/* Protected mode selector		*/
	*off = 0;
	*r_seg = r.x.ax;				/* Real mode segment			*/
	*r_off = 0;
	return 1;
}

void _cdecl PM_freeRealSeg(unsigned sel,unsigned off)
{
	PMREGS	r;

	r.x.ax = 0x101;					/* DPMI free DOS memory			*/
	r.x.dx = sel;					/* DX := selector from 0x100	*/
	off = off;
	PM_int386(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg

int _cdecl PM_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS	rmregs;
	PMREGS	r;
	PMSREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	PM_segread(&sr);
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = sr.ds;
	r.e.edi = (unsigned)&rmregs;
	PM_int386x(0x31, &r, &r, &sr);	/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int _cdecl PM_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS	rmregs;
	PMREGS	r;
	PMSREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	PM_segread(&sr);
	r.x.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.x.cx = 0;
	sr.es = sr.ds;
	r.e.edi = (unsigned)&rmregs;
	PM_int386x(0x31, &r, &r, &sr);	/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Generic DPMI routines common to 16/32 bit code						   */
/*-------------------------------------------------------------------------*/

#ifdef	GENERIC_DPMI

unsigned _cdecl PM_createSelector(unsigned long base,unsigned long limit)
{
	unsigned	sel;
	PMREGS		r;

	/* Allocate 1 descriptor */
	r.x.ax = 0;
	r.x.cx = 1;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag) return 0;
	sel = r.x.ax;

	if (base > 0x100000L) {
        /* Set the descriptor access rights (for a 32 bit page granular
         * segment).
         */
        r.x.ax = 9;
        r.x.bx = sel;
        r.x.cx = 0x8092;
        PM_int386(0x31, &r, &r);

        /* Map physical memory above 1Mb */
        r.x.ax = 0x800;
		r.x.bx = base >> 16;
		r.x.cx = base & 0xFFFF;
		r.x.si = limit >> 16;
		r.x.di = limit & 0xFFFF;
		PM_int386(0x31, &r, &r);
		if (r.x.cflag) return 0;

		r.x.dx = r.x.cx;
		r.x.cx = r.x.bx;
        }
	else {
		r.x.cx = base >> 16;
		r.x.dx = base & 0xFFFF;
		}

	/* Set base address */
	r.x.ax = 7;
	r.x.bx = sel;
	PM_int386(0x31, &r, &r);
    if (r.x.cflag) return 0;

	/* Set limit */
	r.x.ax = 8;
	r.x.bx = sel;
	r.x.cx = limit >> 16;
	r.x.dx = limit & 0xFFFF;
	PM_int386(0x31, &r, &r);
	if (r.x.cflag) return 0;

	return sel;
}

void _cdecl PM_freeSelector(unsigned sel)
{
	PMREGS	r;

	r.x.ax = 1;
	r.x.bx = sel;
	PM_int386(0x31, &r, &r);
}

#endif

