{ =============================================================================
  Name:   				AISINI.PAS
	Version:				1.0
  Description:    This component gives the users access to Paradox like
  								PRIV and WORK alias.  The component works by creating two
                  Databases when the component is dropped on a form.  There
                  are functions included that allow the user to get the path
                  of the WORK or PRIV directory.
									The location of the WORK and PRIV directories is found by
                  searching WIN.INI for the Paradox section where this
                  information is located.  If the Paradox information is not
                  found the programs searches for the Database Desktop
                  information.

  Author:         Dana Scott Kaufman
  								Apogee Information Systems
                  10 Technology Dr. Suite 1040
                  Hudson, MA  01749
                  (508)481-1400
                  CompuServe: 75240, 1376
									Internet: aisys@ix.netcom.com

  Date Created:   9/95
  ==========================================================================}

unit AisIni;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs,IniFiles, DB, DsgnIntf, DirName;

type
  TAisIni = class(TComponent)
  private
    { Private declarations }
		function PrivExist: Boolean;
		function WorkExist: Boolean;
  protected
    { Protected declarations }
  public
    { Public declarations }
    PrivDatabase, WorkDatabase: TDatabase;
    function GetPrivDir: String;
    function GetWorkDir: String;
		procedure ChangePrivDir(Directory: String);
		procedure ChangeWorkDir(Directory: String);
		procedure WriteWorkDir(Directory: String);
		procedure WritePrivDir(Directory: String);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
  end;

   TDirectoryEditor = class(TComponentEditor)
  private
    FComponent: TComponent;
	  DirectoryDlg: TDirForm;
    { Private declarations }
  protected
    { Protected declarations }
  public
    constructor Create(AComponent: TComponent; ADesigner: TFormDesigner); override;
		destructor Destroy; override;
    procedure Edit; override;
    { Public declarations }
  end;

procedure Register;

implementation

uses DbiTypes, DbiProcs, DBiErrs;

{===========================================================================
  Function/Proc Name:   Register
  Description:          Register the AISINI component and place it in the
  											palette on the Apogee page. Also register the
                        Component Editor to allow the changing of the
                        WORK and PRIV directories in the WIN.INI
  Notes:
  Calling Convention:
  Returns :             None
 ===========================================================================}
procedure Register;
begin
  RegisterComponents('Apogee', [TAisIni]);
	RegisterComponentEditor(TAisIni, TDirectoryEditor);
end;

{===========================================================================
  Function/Proc Name:   Create(AComponent: TComponent; ADesigner: TFormDesigner);
  Description:          Override the Create method.  This allows us to grab
  											a reference to the instance of the Component being
                        edited.  It also allows us to allocate the Directory
                        Edit Dialog.
  Notes:                We make sure to call the inherited Create method.
  Calling Convention:   AComponent is the component being edited.  ADesigner
  											is the Designer.
  Returns :             None
 ===========================================================================}
constructor TDirectoryEditor.Create(AComponent: TComponent; ADesigner: TFormDesigner);
begin
  inherited Create(AComponent, ADesigner);
  FComponent := AComponent;
  DirectoryDlg:=TDirForm.Create(nil); {Create Directory Edit Dialog}
end;

{===========================================================================
  Function/Proc Name:   Destroy
  Description:          Override the Destroy method.  Here is where we free
  											the Directory Edit Dialog.
  Notes:
  Calling Convention:
  Returns :             None
 ===========================================================================}
destructor TDirectoryEditor.Destroy;
begin
  DirectoryDlg.Free; {Free resources from Directory Edit Dialog}
	inherited Destroy;
end;

{===========================================================================
  Function/Proc Name:   Edit
  Description:          This method is called when the user double-clicks on
  											the component.  Here is where we display the Dialog.
                        After we display the Dialog with the ShowModal, we
                        check to see if the user has changed any of the text
                        if so, we change the proper database and write the
                        newly entered directory to the INI.
  Notes:                We are able to manipulate the Component currently
  											being edited because we got a handle to it during
                        Create.  To use the Object we need to type-cast it
                        TAisIni().
  Calling Convention:
  Returns :             None
 ===========================================================================}
{
}
procedure TDirectoryEditor.Edit;
begin
	with DirectoryDlg do
  begin
		WorkDirEdit.Text:=TAisIni(FComponent).GetWorkDir; {Display WorkDir on the form}
    PrivDirEdit.Text:=TAisIni(FComponent).GetPrivDir; {Display PrivDir on the form}

    ShowModal; {Show the Dialog}

		{If WorkDir has been edited, write the changes to the WIN.INI and change the database}
    if WorkDirEdit.Modified then
    begin
			TAisIni(FComponent).WriteWorkDir(WorkDirEdit.Text);
      TAisIni(FComponent).ChangeWorkDir(WorkDirEdit.Text);
    end;
		{If PrivDir has been edited, write the changes to the WIN.INI and change the database}
    if PrivDirEdit.Modified then
    begin
			TAisIni(FComponent).WritePrivDir(PrivDirEdit.Text);
      TAisIni(FComponent).ChangePrivDir(PrivDirEdit.Text);
    end;
  end;
end;

{===========================================================================
  Function/Proc Name:   Create(AOwner: TComponent)
  Description:          This is where the two databases are created.  The
  											method checks to see if the PRIV or WORK database is
                        already defined.  If they exist, the new databases
                        will not be assign.
  Notes:                The database components are created no matter what.
  											if the databases exist, they won't be given names or
                        assigned.
  Calling Convention:
  Returns :             None
 ===========================================================================}
constructor TAisIni.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
	{Create Private Database, if PRIV database already exists
  don't configure database}
  PrivDatabase:=TDatabase.Create(nil);
  if not PrivExist then
  begin
    ChangePrivDir(GetPrivDir);
	end;
	{Create Work Database, if WORK alias already exists
  don't configure database}
  WorkDatabase:=TDatabase.Create(nil);
  if not WorkExist then
  begin
		ChangeWorkDir(GetWorkDir); 
  end;
end;

{===========================================================================
  Function/Proc Name:   Destroy
  Description:          Free the two database resources
  Notes:
  Calling Convention:
  Returns :             None
 ===========================================================================}
destructor TAisIni.Destroy;
begin
  PrivDatabase.Free;
	WorkDatabase.Free;
  inherited Destroy;
end;

{===========================================================================
  Function/Proc Name:   ChangeWorkDir(Directory: String)
  Description:          Closes the Work Database, and puts the proper
  											parameters in the database including the new
                        directory.  It save the database connection state so
                        that it can return it to the proper state when it is
                        done.
  Notes:
  Calling Convention:   Directory is the path of the database to be
  											referenced.
  Returns :             None
 ===========================================================================}
procedure TAisIni.ChangeWorkDir(Directory: String);
var
  State: Boolean;
begin
	State:=WorkDatabase.Connected; { Save connection State}
  WorkDatabase.Connected:=False; {Close connections so we can modify it}
  WorkDatabase.DatabaseName:='WORK';
  WorkDatabase.DriverName:='STANDARD'; {Set driver type, STANDARD is Paradox}
  WorkDatabase.Params.Clear; {Clear out old Params}
  WorkDatabase.Params.Add('PATH='+Directory);  {Add the new directory as a Params}
	WorkDatabase.Connected:=State; {Return Database back to its old State}
end;

{===========================================================================
  Function/Proc Name:   ChangePrivDir(Directory: String)
  Description:          Closes the Priv Database, and puts the proper
  											parameters in the database including the new
                        directory.  It save the database connection state so
                        that it can return it to the proper state when it is
                        done.
  Notes:
  Calling Convention:   Directory is the path of the database to be
  											referenced.
  Returns :             None
 ===========================================================================}
procedure TAisIni.ChangePrivDir(Directory: String);
var
  State: Boolean;
begin
	State:=PrivDatabase.Connected; { Save connection State}
  PrivDatabase.Connected:=False; {Close connections so we can modify it}
  PrivDatabase.DatabaseName:='PRIV';
  PrivDatabase.DriverName:='STANDARD'; {Set driver type, STANDARD is Paradox}
  PrivDatabase.Params.Clear; {Clear out old Params}
  PrivDatabase.Params.Add('PATH='+Directory);  {Add the new directory as a Params}
	PrivDatabase.Connected:=State; {Return Database back to its old State}
end;

{===========================================================================
  Function/Proc Name:   WritePrivDir(Directory: String)
  Description:          Writes the Private Directory path to the appropriate
  											place in the WIN.INI.  It first checks to see if
                        Paradox is installed and writes that private
                        directory.  If Paradox is not installed, it modifies
                        the DBD private directory.
  Notes:
  Calling Convention:   Directory is the path of the database to be
  											written to the WIN.INI.
  Returns :             None
 ===========================================================================}
procedure TAisIni.WritePrivDir(Directory: String);
var
  WinIni: TIniFile;
begin
  try
    WinIni:=TIniFile.Create('win.ini'); {Use WIN.INI file}
    with WinIni do
    begin
    	{Look for PDOXWIN section PrivDir= if ERROR is returned Paradox isn't installed}
      if ReadString('PDOXWIN','PrivDir','ERROR')<> 'ERROR' then
        WriteString('PDOXWIN','PrivDir',Directory) {Write new path}
      else
    	  {If not in PDOXWIN look in DBD section}
        if ReadString('DBD','PrivDir','ERROR')<> 'ERROR' then
          WriteString('DBD','PrivDir',Directory); {Write new path}
    end;
  finally
    WinIni.Free;
  end;
end;

{===========================================================================
  Function/Proc Name:   WriteWorkDir(Directory: String)
  Description:          Writes the Working Directory path to the appropriate
  											place in the WIN.INI.  It first checks to see if
                        Paradox is installed and writes that working
                        directory.  If Paradox is not installed, it modifies
                        the DBD working directory.
  Notes:
  Calling Convention:   Directory is the path of the database to be
  											written to the WIN.INI.
  Returns :             None
 ===========================================================================}
procedure TAisIni.WriteWorkDir(Directory: String);
var
  WinIni: TIniFile;
begin
  try
    WinIni:=TIniFile.Create('win.ini'); {Use WIN.INI file}
    with WinIni do
    begin
    	{Look for PDOXWIN section WorkDir= if ERROR is returned Paradox is not installed}
      if ReadString('PDOXWIN','WorkDir','ERROR')<> 'ERROR' then
        WriteString('PDOXWIN','WorkDir',Directory) {Write Path to WIN.INI}
      else
    	  {If not in PDOXWIN look in DBD section}
        if ReadString('DBD','WorkDir','ERROR')<> 'ERROR' then
          WriteString('DBD','WorkDir',Directory); {Write Path to WIN.INI}
    end;
  finally
    WinIni.Free;
  end;
end;

{===========================================================================
  Function/Proc Name:   PrivExist
  Description:          This function checks to see if a PRIV database
  											already exists, if it does return True
  Notes:
  Calling Convention:
  Returns :             True if PRIV database already exists otherwise False
 ===========================================================================}
function TAisIni.PrivExist: Boolean;
var
  DBString: TStringList;
  Index: Integer;
begin
	Result:=False;
  DBString:=TStringList.Create;
	try
    Session.GetDatabaseNames(DBString);
    if DBString.Find('PRIV',Index) then
      Result:=True;
  finally
    DBString.Free;
  end;
end;

{===========================================================================
  Function/Proc Name:   WorkExist
  Description:          This function checks to see if a WORK database
  											already exists, if it does return True
  Notes:
  Calling Convention:
  Returns :             True if WORK database already exists otherwise False
 ===========================================================================}
function TAisIni.WorkExist: Boolean;
var
  DBString: TStringList;
  Index: Integer;
begin
	Result:=False;
  DBString:=TStringList.Create;
	try
    Session.GetDatabaseNames(DBString);
    if DBString.Find('WORK',Index) then
      Result:=True;
  finally
    DBString.Free;
  end;
end;

{===========================================================================
  Function/Proc Name:   GetPrivDir
  Description:          This function looks in the WIN.INI file
  											to find the location of the private directory
  Notes:								Looks for the Paradox section first in the WIN.INI
  											file.  If that is not found then check the Database
                        Desktop.  Future version might have a property that
                        allows you to pick which one to check first.
  Calling Convention:
  Returns :             The Private Directory path or ERROR
 ===========================================================================}
function TAisIni.GetPrivDir: String;
var
  WinIni: TIniFile;
begin
  try
    WinIni:=TIniFile.Create('win.ini'); {Use WIN.INI file}
    with WinIni do
    begin
    	{Look for PDOXWIN section PrivDir= and return that value otherwise
      the call returns ERROR}
      Result:=ReadString('PDOXWIN','PrivDir','ERROR');
    	{If not in PDOXWIN look in DBD section}
      if Result='ERROR' then
        Result:=ReadString('DBD','PrivDir','ERROR');
    end;
  finally
    WinIni.Free;
  end;
end;

{===========================================================================
  Function/Proc Name:   GetWorkDir
  Description:          This function looks in the WIN.INI file
  											to find the location of the work directory
  Notes:								Looks for the Paradox section first in the WIN.INI
  											file.  If that is not found then check the Database
                        Desktop.  Future version might have a property that
                        allows you to pick which one to check first.
  Calling Convention:
  Returns :             The Work Directory path or ERROR
 ===========================================================================}
function TAisIni.GetWorkDir: String;
var
  WinIni: TIniFile;
begin
  try
    WinIni:=TIniFile.Create('win.ini');
    with WinIni do
    begin
    	{Look for PDOXWIN section WorkDir= and return that value otherwise
      the call returns ERROR}
      Result:=ReadString('PDOXWIN','WorkDir','ERROR');
    	{If not in PDOXWIN look in DBD section}
      if Result='ERROR' then
        Result:=ReadString('DBD','WorkDir','ERROR');
    end;
  finally
    WinIni.Free;
  end;
end;

end.
