
unit BitField; {Source code for Delphi 1.0}

{This unit is (C)opyrighted by:
 Juergen Schlottke
 46 Schoenaich-C.-Str.
 D-25336 Elmshorn (Germany, so forgive me my English)

 Please send me Your comments, bug fixes, corrections or
 improvements via Compuserve, my Compuserve-ID is 100106,3034

 You are allowed to use this code in Your compiled applications for free.
 But selling this source code is not allowed.
 If You use this code, do it on Your own risk. I give You no warranties.

 And please: If You have any useful code snippets for DELPHI, load them
 to the DELPHI forum at Compuserve - Thank You very much!

 Whats the use of this unit?
 If You use a normal "array of boolean" with Delphi, this will
 cost You 1 Byte of RAM per Boolean value. This unit defines an object
 TBitArray, that only will use one Bit to hold a boolean value.
 One machine word of 16 bits will hold 16 boolean values!
 Therefore You dont have to mess with the 64 KB barrier, because
 this type of Bitarray can hold up to 524264 boolean values within
 a 64 KB segment!

 And more: These BitArrays can be logical connected with the
 logical operations AND, OR and XOR!
 For example, in a dataset You might want to find all persons who are
 "male" AND "married". So initialize two bitarrays, each in the
 size of the number of data records. One with the boolean result
 of "male", the other with the boolean result of "married".
 Then logical connect those bitfields with AND. The connected
 bitfield will have the result of "male" AND "married". Sounds easy?
 Sure You can find many other opportunities to use the TBitArray object!
 }

{$ifdef debug}
  {$R+,S+,D+,A+,B-,G+}
{$else}
  {$R-,S-,D-,A+,B-,G+}
{$endif}

interface

const
  MinFreeMemory=512*1024; {Dont allocate this memory}
  MaxBitsInArray=524264; {(maxint*2-1)*8; with maxint=32767}
type
  LogicOperType = (LogicAND, LogicOR, LogicXOR);
  PWordArray=^TWordArray;
  TWordArray=array[0..maxint-1] of word;

  TBitArray =class
               BitSize   : longint; {Bits within the Bitarray}
               ByteSize  : word;    {Memory bytes to hold all bits}
               WordSize  : word;    {Machine words to hold all bits}
               BArray    : PWordArray; {Pointer to bitarray, dynamically initialized}
               constructor Create(SizeOfBitArray: longint);
               destructor  Destroy;override;
               procedure   Clear;
               procedure   SetBit (ix: longint);
               procedure   ResetBit (ix: longint);
               procedure   ToggleBit (ix: longint);
               function    TestBit (ix: longint):boolean;
               procedure   LogicConnect(SecondBArray: TBitArray;
                                        Connect: LogicOperType);
               procedure   Invert;
             end;

implementation
uses sysutils,wintypes,winprocs;

procedure CheckLowMemory;
{Raise an exception if low memory conditions are found}
begin
  if (memavail<MinFreeMemory) and (maxavail<MinFreeMemory) then
    raise EOutofMemory.CreateFmt('Unit BITFIELD: Low memory! Memavail: %d KB, Maxavail: %d KB',
                              [Memavail div 1024,MaxAvail div 1024]);
end;

procedure SetBitInWord(var target:word;bitnr:word);
{Set a single bit within a word}
begin
  target:=target or (1 shl bitnr);
end;

procedure ClrBitInWord(var target:word;bitnr:word);
{Clear a single bit within a word}
begin
  target:=target and (not(1 shl bitnr));
end;

function TstBitInWord(target,bitnr:word):boolean;
{Test if the specified bit within a word is set or not}
begin
  TstBitInWord:=target and (1 shl bitnr) <>0;
end;

constructor TBitArray.Create(SizeOfBitArray: longint);
{Contructor to create a bitfield,
 will raise an exception under two conditons:
 1. SizeofBitArray is more than MaxBitsInArray
 2. Low memory condition}
begin
  inherited create;
  if ((SizeOfBitArray > MaxBitsInArray) or (SizeOfBitArray < 1)) then
    raise EOverflow.CreateFmt('Unit BITFIELD cannot create %d bits! Only %d bits allowed',
                              [SizeofBitArray,MaxBitsInArray]);
  BitSize := SizeOfBitArray;
  ByteSize:=sizeof(word)+(BitSize div 8); {round to next word}
  WordSize:=ByteSize div (Sizeof(word)); {if odd ByteSize, still enough words because line above}
  CheckLowMemory;
  GetMem (BArray,ByteSize);
  clear;
end;

destructor TBitArray.Destroy;
{will be called, if an exception happens during
 create or when free ist called while object is assigned}
begin
  if BArray<>nil then FreeMem (BArray, ByteSize);
  inherited destroy;
end;


procedure TBitArray.Clear;
{Clear the whole bitarray}
begin
  FillChar (BArray^,ByteSize,0);
end;

procedure TBitArray.Invert;
{Invert the whole bitarray}
var
  i : word;
begin
  for i:=0 to WordSize-1 do
    BArray^[i]:=not BArray^[i];
end;


procedure  TBitArray.LogicConnect (SecondBArray: TBitArray;
                                  Connect: LogicOperType);
{Logical connect two bitarrays. They must be the same size}
var
  i : word;
begin
  if (BitSize = SecondBArray.BitSize) then
  begin
    case Connect of
      LogicAND    : for i:=0 to WordSize-1 do
                      BArray^[i]:=BArray^[i] and SecondBArray.BArray^[i];
      LogicOR     : for i:=0 to WordSize-1 do
                      BArray^[i]:=BArray^[i] or SecondBArray.BArray^[i];
      LogicXOR    : for i:=0 to WordSize-1 do
                      BArray^[i]:=BArray^[i] xor SecondBArray.BArray^[i];
    end;
  end;
end;

procedure TBitArray.SetBit (ix: longint);
{Set specified bit in bitarray}
var
  wordpos,bitpos:word;
begin
  if (ix >= 0) and (Ix <= BitSize) then
  begin
    wordpos:=ix shr 4;
    bitpos:=ix and $0F;
    SetBitInWord(BArray^[wordpos],bitpos);
  end;
end;

procedure TBitArray.ResetBit (ix: longint);
{Clear specified bit in bitarray}
var
  wordpos,bitpos:word;
begin
  if (ix >= 0) and (Ix <= BitSize) then
  begin
    wordpos:=ix shr 4;
    bitpos:=ix and $0F;
    ClrBitInWord(BArray^[wordpos],bitpos);
  end;
end;


procedure TBitArray.ToggleBit (ix: longint);
{Toggle specified bit in bitarray}
var
  wordpos,bitpos:word;
begin
  if (ix >= 0) and (Ix <= BitSize) then
  begin
    wordpos:=ix shr 4;
    bitpos:=ix and $0F;
    if TstBitInWord(BArray^[wordpos],bitpos) then
      clrBitInWord(BArray^[wordpos],bitpos)
    else
      setBitInWord(BArray^[wordpos],bitpos);
  end;
end;


function TBitArray.TestBit (ix: longint): boolean;
{Test if specified bit in bitarray is set or not}
var
  wordpos,bitpos:word;
begin
  if (ix >= 0) and (Ix <= BitSize)then
  begin
    wordpos:=ix shr 4;
    bitpos:=ix and $0F;
    TestBit:=TstBitInWord(BArray^[wordpos],bitpos);
  end
  else TestBit := False;
end;

end.
