unit Clinnav;

(*
   Description

   Demostrates the use of the TButtonArray class

   This class duplicates DBNavigator except for :
   Uses TButtonArray class for the buttons so that more buttons
   may be added by stretching the control

   Instead of using BOF and EOF to grey out buttons it uses a call to
   the database engine so that the BOF is when the cursor is on
   the first record and EOF when the cursor is on the last record.

   Usually BOF is one before the first record
   Usually EOF is one after the last record

   hint help text and button captions are stored in a RES file CLINNAV.RES

   Most of this file is base on DBNavigator

   Author Mike Lindre CompuServe USERID 100567,2225

   Last Edit 12 July 1995
 *)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ButArray, DB;

type
  ClinButEnum = (enFirst,enPrev,enNext,enLast);
  TClinDataLink = class;
  TClinNav = class(TButtonArray)
  private
    { Private declarations }
    FDataLink:TClinDataLink;
    function  GetDataSource:TDataSource;
    procedure SetDataSource(Value:TDataSource);
    procedure CMEnabledChanged(var Message:TMessage);message CM_ENABLEDCHANGED;
  protected
    { Protected declarations }
    procedure DataChanged;
    procedure ActiveChanged;
    procedure Loaded;override;
    procedure Notification(AComponent:TComponent; Op:TOperation); override;
    procedure ButClick(ButtonIndex:integer);  override;
  public
    { Public declarations }
     constructor Create(AOwner:TComponent);override;
     destructor  Destroy;override;
  published
    { Published declarations }
    property DataSource:TDataSource read GetDataSource write SetDataSource;
  end;

  TClinDataLink = Class(TDataLink)
  private
     FClinNav :TClinNav;
  protected
     procedure DataSetChanged;override;
     procedure ActiveChanged; override;
  public
     constructor Create(AClinNav:TClinNav);
     destructor  Destroy;override;
  end;

implementation

uses DbiTypes, DbiProcs, DbiErrs;
{$R CLINNAV.RES}

constructor TClinNav.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  {Set the Defaults}
  NoOfButtons  := 4;
  {Set the Resource numbers of the help text and button captions}
  {The text is stored in CLINNAV.RES as two string tables }
  NameResource := 200;
  HintResource := 400;
  {make a datalink}
  FDataLink := TClinDataLink.Create(Self);
  ShowHint := True;
end;

destructor TClinNav.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TClinNav.ButClick(ButtonIndex:integer);
{Add the database navigation to the buttons}
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then begin
    with DataSource.DataSet do begin
       case ClinButEnum(ButtonIndex) of
         enFirst:First;
         enNext:Next;
         enPrev:Prior;
         enLast:Last;
       end;
    end;
  end;
  inherited ButClick(ButtonIndex);
end;

procedure TClinNav.Notification(AComponent:TComponent; Op:TOperation);
{If the database goes set field to nil}
begin
  inherited Notification(AComponent,Op);
  if (Op = opRemove) and (FDataLink <> nil ) and
     (AComponent = DataSource) then DataSource := nil;
end;

procedure TClinNav.SetDataSource(Value:TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csloading in ComponentState) then ActiveChanged;
end;

function TClinNav.GetDataSource: TDataSource;
begin
  Result:=FDataLink.DataSource;
end;

procedure TClinNav.DataChanged;
var UpEnabled:Boolean;
    DnEnabled:Boolean;
    Count:longint;
begin
   {Check position in database, by first updating the cursor position}
   FDataLink.DataSet.UpdateCursorPos;
   {Reading the current record position}
   DbiGetSeqNo (FDataLink.DataSet.Handle, Count);
   {Enable or Disable according to the record position}
   UpEnabled := Enabled and FDataLink.Active
      and not ((Count <= 1) or FDataLink.DataSet.BOF);
   DnEnabled := Enabled and FDataLink.Active
      and not ((Count >= FDataLink.DataSet.RecordCount) or FDataLink.DataSet.EOF);
   {button order first prev next last}
   EnableButtons([UpEnabled,UpEnabled,DnEnabled,DnEnabled]);
end;

procedure TClinNav.ActiveChanged;
begin
  if not (Enabled and FDataLink.Active) then
    EnableButtons([false,false,false,false])
  else DataChanged;
end;

procedure TClinNav.CMEnabledChanged(var Message:TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TClinNav.Loaded;
begin
  inherited Loaded;
  ActiveChanged;
end;

{The DataLink}

constructor TClinDataLink.Create(AClinNav:TClinNav);
begin
  inherited Create;
  FClinNav := AClinNav;
end;

destructor TclinDataLink.Destroy;
begin
  FClinNav := nil;
  inherited Destroy;
end;

procedure TClinDataLink.DataSetChanged;
begin
  if FClinNav <> nil then FClinNav.DataChanged;
end;

procedure TClinDataLink.ActiveChanged;
begin
  if FClinNav <> nil then FClinNav.ActiveChanged;
end;

end.
