unit ColCombo;

{***************************************************************************************
 * This unit is supplied free for everybody and is supplied as-is.                     *
 * I explicitly disclaim all responsibility for its use or misuse by anybody.          *
 * Also, no warranties whatsoever are offered for this unit.                           *
 * Permission to use this program is given to individuals and companies                *
 * only if they are willing to assume ALL risk for its use.                            *
 *                                                                                     *
 * You can incorporate the unit in any program.                                        *
 * You can modify the source but than you HAVE TO change the Author name (see below).  *
 *                      ************************************                           *
 *                                                                                     *
 *                       "Thanks for downloading this unit."                           *
 *                                                                                     *
 *                      ************************************                           *
 ***************************************************************************************}



{----------------------------------------------------------------------------
If you will find this unit usefull, please send e-mail to kh@gryzmak.pdi.lodz.pl,
Subject or Text: TColorComboBox or anything else. I just wonder how many
people use it.
                                                  Thanks,
                                                  Chris
-----------------------------------------------------------------------------}


{******************************************************************************
   File        : ColCombo.PAS
   Date        : 03.10.95
   Version     : 1.01
   Contents    : TColorComboBox component
   Author      : Krzysztof Hryniewiecki, e-mail: kh@gryzmak.pdi.lodz.pl
  -----------------------------------------------------------------------------
   Description :
        TColorComboBox component is a combo allowing to select a color.

   Installing :
       On default TColorComboBox installs the newly created page 'Colors'.
      If you want to change it jump to the end of the file and modify the
      Register procedure.
        In Delphi select Options | Install components | Add then type in the
      full path to this file name. Press OK button.

   Changes vs. version 1.0
      Click method has beed added.
      The following bugs have been removed:
          - Changing selection in combo did not change SelectedColor property
          - Name of the selection color was drawn out of boundaries
          - Yellow color was missed

   Properties:
      SelectedColor : TColor (published)
            Currently selected color. It can be any value (not only value from
          the list of colors. What is diplayed in the combo list depends of
          RGB value of the color. The combo is looking for the color in the list
          with the same RGB value as SelectedColor. If found the found color
          is displayed in the list - if not the combo list has no selection
          (et. ItemIndex = -1).
            For example, at design time you can set value clWindow. What will
          be displayed as initial value at runtime depends on settings in the
          Control Panel of the customer's computer.
            You can use user-defined colors overriding GetColorNum and GetColor
          methods or supporting OnGetColorNum and OnGetColor events.

      SelectedColorName : string (published)
            The name of the color displayed in the combo (not neccessary
          SelectedColor). To find the name TColorListBox calls GetColor
          virtual method.

      PictureWidth : Integer (published)
            The width of the colored rectangle in the combo. It has to be
          greater or equal 0. If 0 then no colored rectangle is displayed.

   Event properties:
      OnGetColorNum and OnGetColor  (published)
            You can use these events to make combo box diplay your own colors
          in your own order.

         OnGetColorNum : procedure (Sender : TObject; var Num : Integer) of object;
             Sender
                is the TColorComboBox requesting for the amount of colors to display.
             Num
                is the number of colors to display in the combo.

         OnGetColor : procedure ( Sender : TObject;
                                 Index : Integer;
                                 var Color : TColor; var ColorName : string) of object;
             Sender
                is the TColorComboBox requesting for the color value and name.

             Index
                is the index of the color in the combo.

             Color
                is the value of color to display in the combo

             ColorName
                is the name of the color to display in the combo

   Methods:
      Click (protected)
        Changes current selection and calls inherited Click;

      Reload (public)
          This method refills the combo. Remember to use the methods each time
        you change OnGetColorNum and OnGetColor events at runtime:
            OnGetColorNum := MyOnColorNum;
            OnGetColor    := MyOnGetColor;
            Reload;

   Restrictions:
      1. TColorComboBox.Style always is csOwnerDrawFixed
      2. The folliwing properties of TComboBox as no longer accessible:
            property Items;
            property MaxLength;
            property Sorted;
            property Text;
            property Style;

         Items
              Is used to hold colors name at runtime

         MaxLength
              Has default settings but you cannot modify it

         Sorted
              Always is False. If you want to change the order of colors use
            OnGetColorNum and OnGetColor events and call Reload instead.

         Text
              It is always an empty string and is not used

         Style
              Always csOwnerDrawFixed

 ******************************************************************************}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Menus;

type
  TOnGetColorNumEvent = procedure ( Sender : TObject; var Num : Integer) of object;
  TOnGetColorEvent    = procedure ( Sender : TObject;
                                    Index : Integer;
                                    var Color : TColor;
                                    var ColorName : string) of object;


  TCustomColorComboBox = class(TCustomComboBox)
  private
    { properties' containers }
    FPictureWidth  : Integer;
    FSelectedColor : TColor;

    { event properties' containers }
    FOnGetColorNum : TOnGetColorNumEvent;
    FOnGetColor    : TOnGetColorEvent;

    { properties' read/write methods }
    procedure SetSelectedColor(Color : TColor);
    function  GetSelectedColorName : string;
    function  GetPictureWidth : Integer;
    procedure SetPictureWidth(PictWidth : Integer);

  protected
    { inherited methods }
    procedure Click; override;

    { combo box contents methods }
    function  GetColorNum : Integer; virtual;
    procedure GetColor(Index : Integer; var Color : TColor; var ColorName : string); virtual;

    { item draw method (inherited) }
    procedure DrawItem(Index : Integer; Rect : TRect; State : TOwnerDrawState); override;

    { inherited properties hidden in this class }
    property Style;
    property Items;
    property MaxLength;
    property Sorted;
    property Text;
    property OnMeasureItem;

  public
    { methods }
    constructor Create(AOwner : TComponent); override;
    procedure   Loaded; override;
    procedure   Reload; virtual;

    { properties }
    property SelectedColorName : string  read GetSelectedColorName;

  published
    { properties }
    property SelectedColor     : TColor  read FSelectedColor  write SetSelectedColor;
    property PictureWidth      : Integer read GetPictureWidth write SetPictureWidth;

     { events }
    property OnGetColorNum : TOnGetColorNumEvent read FOnGetColorNum write FOnGetColorNum;
    property OnGetColor    : TOnGetColorEvent    read FOnGetColor    write FOnGetColor;
  end;



TColorComboBox = class (TCustomColorComboBox)
published
    property SelectedColor;
    property PictureWidth default 32;
    property OnGetColorNum;
    property OnGetColor;
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property DropDownCount;
    property Enabled;
    property Font;
    property ItemHeight;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
end;



procedure Register;

implementation



const
   { *************************************************
     *  Array of colors used as a defualt color list *
     ************************************************* }
  _COLORSARRAY : array [0..15] of TColor =
    (
     clBlack,
     clMaroon,
     clGreen,
     clOlive,
     clNavy,
     clPurple,
     clTeal,
     clGray,
     clSilver,
     clRed,
     clLime,
     clYellow,
     clBlue,
     clFuchsia,
     clAqua,
     clWhite
    ); { _COLORSARRAY }



{************************* inherited methods ***************************** }


        procedure TCustomColorComboBox.Click;
var
   ColorName : string;
begin
   if ItemIndex >= 0 then
     GetColor(ItemIndex, FSelectedColor, ColorName);
   inherited Click;
end;



{************************* constructor ********************************** }

    constructor TCustomColorComboBox.Create(AOwner : TComponent);
begin
   inherited Create(AOwner);

   if Owner is TWinControl then
     Parent := (Owner as TWinControl);

   Sorted         := False;
   Style          := csOwnerDrawFixed;
   FPictureWidth  := 32;

   Reload; { fill with default colors }
end;



procedure TCustomColorComboBox.Loaded;
begin
   inherited Loaded;

   if not (csDesigning in ComponentState) then
      Reload; { if not design time then fill with user-defined colors if any }
end;





{ ************************ filling the combo box ************************ }

function  TCustomColorComboBox.GetColorNum : Integer;
var
  ret : Integer;
begin
  ret := High(_COLORSARRAY) - Low(_COLORSARRAY) + 1;
  if Assigned(FOnGetColorNum) then
     FOnGetColorNum(Self, ret);
  GetColorNum := ret;
end;



procedure TCustomColorComboBox.GetColor( Index : Integer;
                                   var Color : TColor;
                                   var ColorName : string);
begin
  if (Low(_COLORSARRAY) <= Index) and (Index <= High(_COLORSARRAY)) then
     begin
       Color := _COLORSARRAY[Index];

       { ColorName is ColorToString without leading 'cl' }
       ColorName := ColorToString(_COLORSARRAY[Index]);
       if Copy(ColorName, 1, 2) = 'cl' then
         ColorName := Copy(ColorName, 3, Length(ColorName) - 2);
       { else it is be HEX represantation of the _COLORSARRAY[Index] }
     end;

  if Assigned(FOnGetColor) then
     FOnGetColor(Self, Index, Color, ColorName);
end;




    procedure TCustomColorComboBox.Reload;
var
  i               : Integer;
  colNum          : Integer;
  ColorName       : string;
  oldColor, Color : TColor;
  selectedIndex   : Integer;
begin
  { store currently selected color }
  oldColor := SelectedColor;
  selectedIndex  := -1;

  { clear the combo list and fill it again }
  Clear;
  colNum := GetColorNum;
  for i := 0 to colNum - 1 do
    begin
      GetColor(i, Color, ColorName);
      Items.Add(ColorName);

      { if this is previously selected color }
      if ColorToRGB(Color) = ColorToRGB(oldColor) then
         selectedIndex  := i;
    end;

  { restore previously selected color }
  ItemIndex := selectedIndex;
end;










{************************* Property methods ********************************}



    procedure TCustomColorComboBox.SetSelectedColor(Color : TColor);
var
  i         : Integer;
  colNum    : Integer;
  ColorName : string;
  fColor    : TColor;
begin
  { store value in FSelectedColor }
  FSelectedColor := Color;

  { look for color in the combo list }
  i := Items.Count;
  repeat
    Dec(i);
    GetColor(i, fColor, ColorName);
  until (i <= 0) or (ColorToRGB(fColor) = ColorToRGB(Color));

  if ColorToRGB(fColor) = ColorToRGB(Color) then
     ItemIndex := i
  else
     ItemIndex := -1;
end;


    function  TCustomColorComboBox.GetSelectedColorName : string;
begin
   GetSelectedColorName := '';
   if ItemIndex >= 0 then
     GetSelectedColorName := Items.Strings[ItemIndex];
end;





    function  TCustomColorComboBox.GetPictureWidth : Integer;
begin
  GetPictureWidth := FPictureWidth;
  if FPictureWidth < 0 then { Picture width cannot be negative value }
     GetPictureWidth := 32;
end;


    procedure TCustomColorComboBox.SetPictureWidth(PictWidth : Integer);
begin
  if FPictureWidth <> PictWidth then
     begin
       FPictureWidth := PictWidth;
       Refresh;
     end;
end;









{********************************* Drawing ************************************}

procedure TCustomColorComboBox.DrawItem(Index : Integer; Rect : TRect; State : TOwnerDrawState);
var
  orgRect     : TRect;
  pictRect    : TRect;
  oldColor    : TColor;
  Color       : TColor;
  ColorName   : string;
  FOnDrawItem : TDrawItemEvent;
begin
  { Set margin between items }
  orgRect := Rect;
  InflateRect(Rect, -3, 0);

  { get color and color name to draw }
  GetColor(Index, Color, ColorName);

  with Canvas do
    begin
        { if colored rectangle (picture) is to be drawn }
        if PictureWidth > 0 then
          begin
            { set coordinates of the colored rectangle }
            pictRect := Rect;
            pictRect.Right := pictRect.Left + PictureWidth;
            InflateRect(pictRect, 0, -2);

            { set coordinates of color name area }
            Rect.Left := pictRect.Right + 3;

            { set color of the rectangle to draw }
            oldColor := Brush.Color;
            Brush.Color := Color;
            Brush.Style := bsSolid;

            { draw the colored rectangle }
            Rectangle(pictRect.Left, pictRect.Top, pictRect.Right, pictRect.Bottom);

            { restore color }
            Brush.Color := oldColor;
          end;

        { draw the color name text }
        TextRect( Rect,
                  Rect.Left,
                  Rect.Top + (Rect.Bottom - Rect.Top - TextHeight(ColorName)) div 2,
                  ColorName);
    end;


  { call OnDrawItem event if supported }
   FOnDrawItem := OnDrawItem;
   if Assigned(FOnDrawItem) then  { check if OnDrawItem assigned }
      FOnDrawItem(Self, Index, orgRect, State);
end;







procedure Register;
begin
  RegisterComponents('Colors', [TColorComboBox]);
end;

end.
