{========
Newsgroups: comp.lang.pascal.delphi.components
Subject: Quick and dirty TDBLookUpComboInc component.
From: sean@crm.co.nz (Sean Cross)
Date: Wed, 04 Oct 95 12:40:55 GMT


As a number of people appear to be having problems finding or implementing an 
Incremental Search Combo box, I am posting mine.  This is fairly quick and 
dirty but it appears to work reasonaly well.

Use it just like a normal combo box, but for best performance set the 
IndexNames property of the Lookup table (if using a table) to the Display 
field.  If LimitToList is true then on leaving the combobox, the cursor of the 
Lookup table is set to the row displayed in the combo box so that you can then 
use the Fields of the LookUp table to get further informaion.

The FindRow function moves the cursor to the appropriate row in the Lookup 
dataset (use if if the combo box has not been entered recently).

Easiest way to test it is to drop a Inc C Box onto a form and add a DBGrid 
connected to the lookup table so you can see what happens.

This unit is freeware etc with no warranty given or implied.  However if any 
major improvements are made, I would appreciate a copy.


Sean


****************  Cut here  **********
}
unit Icombo;
{Version 0.9
  modified TDBLookUpCombo that performs an incremental search during editing,
  if the first character is a * or % then will look for a substring rathr than 
matching
  at the start of the Filed value.
  Note only works well with strings, or with data of constant length
    eg 1012 4518 8945 OK but 1 45 895 a bit strange
  adds two extra properties,
    LimitToList : boolean. If true then a value must be selected from the 
list,
                           if no value is selected then the closest one is 
used
    Ordered : boolean. Not currently used, I'm intending to write a faster 
search for ordered but unindexed fields. Ignore!
}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, DBLookup, DB, DBTables;

type
  TDBLookUpComboInc = class(TDBLookupCombo)
  private
    { Private declarations }
    CanChange: Boolean;   {Used to temparerily switch off Change events}
    FieldLength: Integer;  {Records the length of the entered string in the 
edit box}
    LimitedToList: boolean;  {if true will only allow a value to be entered 
that matchs a value in the list
                            if needed will select the current value from the 
table}
    HadWildCard: Boolean;  {Set once a wild card has been entered and used to 
search}
    Ordered: Boolean;      {Set if lookupdisplay Column is in ascending order}
    Entered: Boolean;      {Set once the control has been entered}
    SearchField: TField;   {The field we actually search on}
    SearchType:(SuperSlow,Slow,Fast);

  protected
    { Protected declarations }
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure Change; override;  {These methods are overwriten to provide the 
incremental search capacity }

  public
    constructor Create(AOwner: TComponent); override;
    procedure FindRow;   {Moves the cursor on the related table to the 
appropriate row}

    { Public declarations }
  published
    { Published declarations }
    property LimitTolist: Boolean read LimitedToList write LimitedToList 
default False;
             {If true then values will be accepted only if they match an entry 
in the list,
              values not matching will be changed to the closest entry}
    property IsOrdered: boolean read Ordered write Ordered default False;
             {Should be set by designer if the LookupDisplay column is in 
ascending order, only important for queries}
  end;

procedure Register;
procedure FindCarefully(Table: TDataSet;SearchField: TField; const SubString: 
String);
procedure FindContaining(Table: TDataSet ;SearchField: TField; const 
SubString: String; Down: Boolean);
{procedure FindQuick(Table: TDataSet;Field: String; const SubString: String);
procedure SearchQuick(const DataSet: TDataSet;const SearchField: TField; const 
SubString: String 1024: integer);}

implementation

procedure FindCarefully(Table: TDataSet;SearchField: TField; const SubString: 
String);
{Searches through table (could be a query actually) until if finds the closest 
value
 in Field to Substring,
 quite slow because it must search every record,
  use only on non ordered tables}

begin
try
    with Table do
    begin
      DisableControls; {faster, smoother}
      if CompareText(SubString,(SearchField.AsString))>0 then  {if not far 
enough down the list then go down, else go up}
        while (CompareText(SubString,(SearchField.AsString))>0) and (not eof) 
do Next
      else
      begin
        while (CompareText(SubString,(SearchField.AsString))<0) and (not bof) 
do Prior;
        if (CompareText(SubString,(SearchField.AsString))>0) then Next; {stop 
overshooting}
        end;
    end;
  finally
    Table.EnableControls;
  end;
end;

procedure FindContaining(Table: TDataSet;SearchField: TField; const SubString: 
String; Down: Boolean);
{Searches through table (could be a query actually) until if finds
 a Field containing Substring,
 quite slow because it must search every record}

var
  CapSubString: String;

begin
  try
    CapSubString:=UpperCase(SubString);
    with Table do
    begin
      DisableControls; {faster, smoother}
      if Down then
        while ((Pos(CapSubString,UpperCase(SearchField.AsString))=0) and (not 
eof)) do Next
      else while ((Pos(CapSubString,UpperCase(SearchField.AsString))=0) and 
(not bof)) do Prior;
      if EOF then First;
    end;
  finally
    Table.EnableControls;
  end;
end;

procedure Register;
begin
  RegisterComponents('Data Controls', [TDBLookUpComboInc]);
end;

{*
***********************************************
*        TDBLookUpComboInc methods           *
***********************************************}

procedure TDBLookUpComboInc.Change;
var
  source: TDataSource;
  CurrentValue: String;
  FirstChar: string[1];
begin
  inherited Change;
  if CanChange then
  begin
    try
      CanChange:=False;
      CurrentValue:=Value;
      FirstChar:= CurrentValue;   {Grabs the first character to test for 
wildcard}
      if (FirstChar='*') or (FirstChar='%') then {has wildcard so do slow 
record by record search for matching item}
      begin
        if HadWildCard=False or (FieldLength>Length(CurrentValue)) then 
LookUpSource.DataSet.First;
          {if have deleted, or just entered wildcard then start from top}
        
FindContaining(LookUpSource.DataSet,SearchField,Copy(CurrentValue,2,255),True)
;
        HadWildCard:=True;
      end
      else
      begin
        case SearchType of
          Superslow: 
FindCarefully(LookUpSource.DataSet,SearchField,CurrentValue);
          Slow :  
FindCarefully(LookUpSource.DataSet,SearchField,CurrentValue);
            {Is ordered but not index, one day I'll write a faster find 
routine}

          Fast :(LookUpSource.DataSet as TTable).FindNearest([CurrentValue]); 
{no wildcards, do indexed search}
        end;
        HadWildCard:=False;
      end;
    finally

      CanChange:=True;
      FieldLength:=Length(CurrentValue);
    end;
  end;
end;

constructor TDBlookUpComboInc.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
  CanChange:=False; {Don't want our OnChange updates unless we actually change 
it}
  LimitedToList:=False;
  HadWildCard:=False;
  FieldLength:=0;
  Ordered:=False;
  Entered:=False;
end;

procedure TDBLookupComboInc.DoEnter;
var
   Field, IndexField: String;
begin
{Set index to Lookupdisplay,  if it fails or table type is not string then use 
slow search}
  try
    HadWildCard:=False;
    CanChange:=True;
    Entered:=True;
    if Pos(';',Lookupdisplay)>0 then 
Field:=Copy(Lookupdisplay,1,Pos(';',Lookupdisplay)-1)
      else Field:= Lookupdisplay;
    SearchField:=LookUpSource.DataSet.FindField(Field);  {Get Field once}
    IndexField:=Copy((LookUpSource.DataSet as 
TTable).IndexFieldNames,1,length(Field));
    CanChange:=Assigned(SearchField);
    if (LookUpSource.DataSet is TTable) then
      if CompareText(Field,IndexField)=0
        then SearchType:=Fast
        else If Ordered then SearchType:=Slow else SearchType:=SuperSlow
        {If table and index field names set to first field in Lookupdisplay
         then can use FindNearest, otherwise use a slower search}
    else If Ordered then SearchType:=Slow else SearchType:=SuperSlow;
      {is query so use slow (skip) search if ordered else use Superslow (check 
every value) search}
    Inherited DoEnter;
    Change;  {Now go imediately to the right row}
  except
    SearchType:=SuperSlow;  {Had error, probably in seting index,
                            dataset is not ordered therefore do superslow 
search}
  end;
end;

procedure TDBLookUpComboInc.DoExit;

begin
  try
    if LimitedToList then
    begin
      {CanChange:=False;}
      with DataSource.DataSet do
        if (State=dsEdit) or (State=dsInsert) then
        begin
          FieldByName(DataField).AsString:=SearchField.AsString; {Set value to 
current field}
          {FindRow; }{make sure cursor is in correct place in underlying 
table}
        end;
    end;
  finally
    inherited DoExit;
  end;
end;

procedure TDBLookUpComboInc.FindRow;
{Moves the cursor on the related table to the appropriate row
only needed before the control has been entered, or if it has been a while
as the cursor is automatically set when the control is entered}
begin
  if Entered then DoEnter else Change;
end;

end.
