unit Dllload;

(*********************************************
TDLLLoader -> TComponent

Manages dynamic load libraries.  Will return a proc
address based on the DLL and Proc name supplied.

PROPERTIES:

Path - Specifies a search path.  Leave blank for default DLL search paths.
CacheSize - The number of DLLs to cache.  If this number is exceeded, DLLs
  are unloaded on an LRU (least recently used) basis.

METHODS:

ProcAddress - Returns the proc address of the specified DLL and Proc name.
  Will cache the DLL handle for future reference.
*********************************************)
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type

  TDLLLoaderException = class( Exception );

  TDLLLoader = class( TComponent )
  private
     FPath: string;
     FCache: byte;
     lstDLL: TStrings;
     sDLLName: string;
     lpDLL: PChar;
  protected
     procedure SetName( const sDLL: string );
  public
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     function ProcAddress( const sDLL, sProc: string ): TFarProc;
  published
     property Path: string read FPath write FPath;
     property CacheSize: byte read FCache write FCache default 5;
  end;

procedure Register;

implementation

constructor TDLLLoader.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FCache := 5;
  lstDLL := TStringList.Create;
  lpDLL := nil;
end;

destructor TDLLLoader.Destroy;
begin
  while lstDLL.Count > 0 do
     begin
        FreeLibrary( THandle( lstDLL.Objects[0] ) );
        lstDLL.Delete( 0 );
     end;
  lstDLL.Free;
  if lpDLL <> nil then
     StrDispose( lpDLL );
  inherited Destroy;
end;

(*********************************************
1) Check to see if the DLL is loaded
2) If not, load it, removing the last one loaded
   if cached limit was exceeded.
3) Return the proc address.
*********************************************)
function TDLLLoader.ProcAddress( const sDLL, sProc: string ): TFarProc;
var
  i, nIndex: integer;
  h: THandle;
  lpProcName: PChar;
  s: string;
begin
  nIndex := -1;
  for i := 0 to lstDLL.Count - 1 do
     if lstDLL[i] = sDLL then
        begin
           nIndex := i;
           Break;
        end;

  if nIndex = -1 then
     begin
        if lstDLL.Count = FCache then
           begin

{ Need to release the last allocated DLL }
              h := THandle( lstDLL.Objects[lstDLL.Count - 1] );
              FreeLibrary( h );
              lstDLL.Delete( lstDLL.Count - 1 );
           end;

{ Load the DLL and cache it }
        SetName( sDLL );
        h := LoadLibrary( lpDLL );
        nIndex := 0;
        lstDLL.Insert( 0, sDLL );
        lstDLL.Objects[0] := pointer( h );

     end
  else
     begin

{ Bump this DLL to first position }
        h := THandle( lstDLL.Objects[nIndex] );
        if nIndex <> 0 then
           begin
              lstDLL.Delete( nIndex );
              lstDLL.Insert( 0, sDLL );
              lstDLL.Objects[0] := pointer( h );
           end;
     end;

{ We should now have the proper handle in any event, so get the proc address }
  lpProcName := StrAlloc( Length( sProc ) + 1 );
  StrPCopy( lpProcName, sProc );
  Result := GetProcAddress( h, lpProcName );
  StrDispose( lpProcName );

end;

(*********************************************
Set the DLL name and PChar variables.
*********************************************)
procedure TDLLLoader.SetName( const sDLL: string );
begin
  sDLLName := FPath + sDLL;
  if Pos( '.', sDLLName ) = 0 then
     sDLLName := sDLLName + '.DLL';
  if lpDLL <> nil then
     StrDispose( lpDLL );
  lpDLL := StrAlloc( Length( sDLLName ) + 1 );
  StrPCopy( lpDLL, sDLLName );
end;

procedure Register;
begin
  RegisterComponents( 'System', [TDLLLoader] );
end;

end.
