{ TDragGauge v1.0 by Jason Jasmin (therogue@gate.net)                         }
{                                                                             }
{ This unit implements a flexible gauge component with features such as 3d    }
{ appearance, variable formats for text displayed inside the gauge, and the   }
{ ability for the user to set the gauge value via the mouse. Refer to the     }
{ documentation for usage information.                                        }

unit Drggauge;

interface

uses
	SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
	Forms, Dialogs, Buttons;

type
	TBorderStyle = (bsSimple, bsComplex);
   TTextStyle = (tsPercent, tsNumbers, tsNone);
   TGaugeChangeEvent = procedure(Sender : TObject; Value : Longint) of object;

	TDragGauge = class(TCustomControl)
	private
	{ Private declarations }
	FMaxValue 			: Longint;
   FCurValue 			: Longint;
   FColor 				: TColor;
   FBorderStyle 		: TBorderStyle;
	FShowText 			: TTextStyle;
   FDrag 				: Boolean;
   FGaugeChangeEvent 	: TGaugeChangeEvent;
   FollowMouse 		: Boolean;
   procedure SetMaxValue(Value : Longint);
   procedure SetCurValue(Value : Longint);
   procedure SetColor(Color : TColor);
   procedure SetBorderStyle(Style : TBorderStyle);
   procedure SetShowText(Value : TTextStyle);
   procedure SetDrag(Value : Boolean);
   function  GetPercentDone : Longint;
   function  SolveForX(Y, Z : Longint) : Integer;
   function  SolveForY(X, Z : Longint) : Integer;
	function  ConstrainValue(Minimum, Maximum, Value : Longint) : Longint;

   protected
	{ Protected declarations }
	procedure Paint; override;
	procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
	procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
   procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
   procedure GaugeChange; dynamic;

	public
	{ Public declarations }
	constructor Create(AOwner : TComponent); override;

   published
	{ Published declarations }
	property Align;
   property AllowUserDrag : Boolean read FDrag write SetDrag default True;
	property BarColor : TColor read FColor write SetColor default clRed;
	property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsComplex;
   property CurValue : Longint read FCurValue write SetCurValue;
	property Enabled;
	property Font;
	property MaxValue : Longint read FMaxValue write SetMaxValue;
	property OnChange : TGaugeChangeEvent read FGaugeChangeEvent write FGaugeChangeEvent;
   property ParentColor;
	property ParentFont;
	property ParentShowHint;
	property ShowHint;
   property TextStyle : TTextStyle read FShowText write SetShowText default tsPercent;
	property Visible;

	end;

procedure Register;

implementation

constructor TDragGauge.Create(AOwner : TComponent);
begin
	inherited Create(AOwner);
	ControlStyle := ControlStyle + [csOpaque];
   Width := 200;
   Height := 20;
   FMaxValue := 100;
   FCurValue := 0;
   SetBorderStyle(bsComplex);
   SetColor(clRed);
   SetShowText(tsPercent);
   SetDrag(True);
   SetShowText(tsPercent);
end;

function TDragGauge.GetPercentDone : Longint;
begin
	result := SolveForY(FCurValue, FMaxValue);
end;

{ This function solves for x in the equation "x is y% of z". }
function TDragGauge.SolveForX(Y, Z : Longint) : Integer;
begin
	result := Trunc(Z * (Y * 0.01));
end;

{ This function solves for y in the equation "x is y% of z". }
function TDragGauge.SolveForY(X, Z : Longint) : Integer;
begin
	if Z = 0 then result := 0
   else result := Trunc( (X * 100) / Z);
end;

{ This function constrains a value between two other values }
function TDragGauge.ConstrainValue(Minimum, Maximum, Value : Longint) : Longint;
begin
	Result := Value;
   if Value < Minimum then Result := Minimum;
   if Value > Maximum then Result := Maximum;
end;

procedure TDragGauge.SetMaxValue(Value : Longint);
begin
	if Value <> FMaxValue then
   begin
   	FMaxValue := Value;
       Refresh;
   end;
end;

procedure TDragGauge.SetCurValue(Value : Longint);
begin
	if Value <> FCurValue then
   begin
   	FCurValue := ConstrainValue(0, FMaxValue, Value);
       Refresh;
		GaugeChange;
   end;
end;

procedure TDragGauge.SetColor(color : TColor);
begin
	FColor := color;
   Refresh;
end;

procedure TDragGauge.SetBorderStyle(style : TBorderStyle);
begin
	if style <> FBorderStyle then
   begin
   	FBorderStyle := style;
       Refresh;
   end;
end;

procedure TDragGauge.SetShowText(Value : TTextStyle);
begin
	if Value <> FShowText then
   begin
   	FShowText := Value;
       Refresh;
   end;
end;

procedure TDragGauge.SetDrag(Value : Boolean);
begin
	if Value <> FDrag then
   begin
   	FDrag := Value;
   	Refresh;
   end;
end;

procedure TDragGauge.GaugeChange;
begin
	if Assigned(FGaugeChangeEvent) then
   	FGaugeChangeEvent(Self, FCurValue);
end;

procedure TDragGauge.Paint;
var	rect, progressRect : TRect;
		bm : TBitMap;
		FillSize : integer;
		S: string;
		X, Y: Integer;
begin
	{ Initialize offscreen BitMap }
   bm := TBitMap.Create;
	bm.Width := Width;
   bm.Height := Height;

   case FBorderStyle of
		bsSimple:
   	begin
			{ Draw top and left sides of container }
		  	bm.Canvas.Brush.Color := clBtnShadow;
			rect.Left := 0;
			rect.Top := 0;
			rect.Right := Width - 1;
			rect.Bottom := Height - 1;
			bm.Canvas.FillRect(rect);

		  	{ Draw bottom and right sides of container }
			bm.Canvas.Brush.Color := clBtnHighlight;
			rect.Left := 2;
			rect.Top := 2;
			rect.Right := Width;
			rect.Bottom := Height;
			bm.Canvas.FillRect(rect);

			{ Erase center of container }
			bm.Canvas.Brush.Color := clBtnFace;
			rect.Left := 1;
			rect.Top := 1;
			rect.Right := Width - 1;
			rect.Bottom := Height - 1;
			bm.Canvas.FillRect(rect);

	    	{ Draw progress bar }
			progressRect := ClientRect;
			bm.Canvas.Brush.Color := FColor;
			FillSize := SolveForX(GetPercentDone, Width - 1);
			if FillSize > Width - 1 then FillSize := Width - 1;
			if FillSize > 0 then
			begin
		       	Inc(progressRect.Left);
	   			progressRect.Right := FillSize;
           	InflateRect(progressRect, 0, -1);
       		bm.Canvas.FillRect(progressRect);
   		end;
		end;

   bsComplex:
   	begin
       	{ Draw border }
			rect := ClientRect;
			rect := DrawButtonFace(bm.Canvas, rect, 1, bsNew, False, True, False);

			{ Draw progress bar }
			progressRect := rect;
			bm.Canvas.Brush.Color := FColor;
			FillSize := SolveForX(GetPercentDone, Width - 1);
			if FillSize > Width - 1 then FillSize := Width - 1;
			if FillSize > 0 then
			begin
   			progressRect.Right := FillSize;
				InflateRect(progressRect, 0, -1);
       		OffsetRect(progressRect, -1, -1);
           	Inc(progressRect.Left);
       		bm.Canvas.FillRect(progressRect);
   		end;
		end;
   end;

   { Perform text formatting }
   case FShowText of
   	tsPercent: S := Format('%d%%', [GetPercentDone]);
       tsNumbers: S := Format('%d of %d', [FCurValue, FMaxValue]);
       tsNone:    S := '';
   end;

   { ... and draw the text }
  	rect := ClientRect;
	with bm.Canvas do
	begin
		Brush.Style := bsClear;
		Font := Self.Font;
		with rect do
		begin
			X := (Right - Left + 1 - TextWidth(S)) div 2;
			Y := (Bottom - Top + 1 - TextHeight(S)) div 2;
		end;
		TextRect(rect, X, Y, S);
	end;

   { Copy offscreen bitmap to onscreen component }
	Canvas.CopyMode := cmSrcCopy;
   Canvas.Draw(0, 0, bm);

   { Free the BitMap }
   bm.Destroy;
end;

procedure TDragGauge.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	inherited MouseDown(Button, Shift, X, Y);
	if FDrag = True then
   begin
		FollowMouse := True;
		SetCurValue(SolveForY(x + 2, Width));
		Refresh;
   end;
end;

procedure TDragGauge.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	inherited MouseUp(Button, Shift, X, Y);
	FollowMouse := False;
end;

procedure TDragGauge.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
	inherited MouseMove(Shift, X, Y);
	if FollowMouse = True then
   begin
   	SetCurValue(SolveForY(x + 2, Width));
   	Refresh;
   end;
end;

procedure Register;
begin
	RegisterComponents('Samples', [TDragGauge]);
end;

end.
